// -*-c++-*-

/*!
  \file draw_config.h
  \brief draw object holder class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCERWINDOW2_QT_DRAW_CONFIG_H
#define SOCCERWINDOW2_QT_DRAW_CONFIG_H

#include <QBrush>
#include <QColor>
#include <QFont>
#include <QPen>
#include <QString>

#include <rcsc/types.h>
#include <iostream>
#include <cmath>

//! draw object configuration manager class
class DrawConfig {
public:
    //--------------------------------------------------
    // default color settings
    static const QColor SCORE_BOARD_COLOR;
    static const QColor SCORE_BOARD_FONT_COLOR;
    static const QColor FIELD_COLOR;
    static const QColor FIELD_DARK_COLOR;
    static const QColor LINE_COLOR;
    static const QColor BALL_PEN_COLOR;
    static const QColor BALL_COLOR;
    static const QColor BALL_FONT_COLOR;
    static const QColor PLAYER_PEN_COLOR;
    static const QColor REAL_BODY_PEN_COLOR;
    static const QColor LEFT_TEAM_COLOR;
    static const QColor LEFT_GOALIE_COLOR;
    static const QColor RIGHT_TEAM_COLOR;
    static const QColor RIGHT_GOALIE_COLOR;
    static const QColor PLAYER_SHADOW_COLOR;
    static const QColor PLAYER_NUMBER_FONT_COLOR;
    static const QColor PLAYER_STAMINA_FONT_COLOR;
    static const QColor PLAYER_DETAIL_FONT_COLOR;
    static const QColor SELECTED_PLAYER_FONT_COLOR;
    static const QColor EFFORT_DECAYED_PEN_COLOR;
    static const QColor RECOVERY_DECAYED_PEN_COLOR;
    static const QColor VIEW_CONE_PEN_COLOR;
    static const QColor VIEW_AREA_PEN_COLOR;
    static const QColor KICK_PEN_COLOR;
    static const QColor KICK_FAULT_COLOR;
    static const QColor CATCH_COLOR;
    static const QColor CATCH_FAULT_COLOR;
    static const QColor COLLIDE_BALL_COLOR;
    static const QColor COLLIDE_PLAYER_COLOR;
    static const QColor TACKLE_COLOR;
    static const QColor TACKLE_FAULT_COLOR;
    static const QColor MEASURE_COLOR;
    static const QColor MEASURE_FONT_COLOR;
    static const QColor DEBUG_BALL_COLOR;
    static const QColor DEBUG_SELF_COLOR;
    static const QColor DEBUG_TEAMMATE_COLOR;
    static const QColor DEBUG_OPPONENT_COLOR;
    static const QColor DEBUG_UNKNOWN_TEAMMATE_COLOR;
    static const QColor DEBUG_UNKNOWN_OPPONENT_COLOR;
    static const QColor DEBUG_UNKNOWN_PLAYER_COLOR;

private:
    //--------------------------------------------------
    // general
    QPen M_transparent_pen;
    QBrush M_transparent_brush;

    //--------------------------------------------------
    // score board

    QBrush M_score_board_brush;
    QFont M_score_board_font;
    QPen M_score_board_font_pen;
    QColor M_score_board_font_color;

    //--------------------------------------------------
    // field

    //! field grass brush
    QBrush M_field_brush;
    //! field dark grass brush
    QBrush M_field_dark_brush;
    //! line pen
    QPen M_line_pen;

    //--------------------------------------------------
    // ball

    QPen M_ball_pen; //!< ball edge
    QBrush M_ball_brush; //!< ball body paint info
    QFont M_ball_font; //!< ball info font info
    QPen M_ball_font_pen; //!< ball font foreground color
    QColor M_ball_font_color; //!< ball font foreground color

    //--------------------------------------------------
    // players

    //! player edge pen
    QPen M_player_pen;
    //! real body edge pen
    QPen M_real_body_pen;

    //! shadow brush
    QBrush M_shadow_brush;
    QBrush M_shadow_brushes[256];

    //! left team color pen
    QPen M_left_team_pen;
    //! left team brush (= left player body brush)
    QBrush M_left_team_brush;
    //! left goalie color pen
    QPen M_left_goalie_pen;
    //! left goalie body brush
    QBrush M_left_goalie_brush;

    //! right team pen
    QPen M_right_team_pen;
    //! right team brush(= right player body brush)
    QBrush M_right_team_brush;
    //! right goalie color pen
    QPen M_right_goalie_pen;
    //! left goalie body brush
    QBrush M_right_goalie_brush;

    //! player info font
    QFont M_player_font;
    //! foreground color of the player number font
    QPen M_player_number_font_pen;
    QColor M_player_number_font_color;
    //! foreground color of the player stamina font
    QPen M_player_stamina_font_pen;
    QColor M_player_stamina_font_color;
    //! foreground color of the player detail
    QPen M_player_detail_font_pen;
    QColor M_player_detail_font_color;
    //! foreground color of the selected player font
    QPen M_selected_player_font_pen;
    QColor M_selected_player_font_color;

    //! the pen of the effort decayed player indication circle
    QPen M_effort_decayed_pen;
    //! the pen of the recovery decayed player indication circle
    QPen M_recovery_decayed_pen;

    //! the pen of players' view cone
    QPen M_view_cone_pen;
    //! the pen of players' view area
    QPen M_view_area_pen;


    //--------------------------------------------------
    // player status

    QPen M_kick_pen; //!< edge color when kicking
    QPen M_kick_fault_pen; //!< edge color when kick falut
    QBrush M_kick_fault_brush; //!< body paint brush when kick falut
    QBrush M_catch_brush; //!< body paint brush when goalie catch
    QBrush M_catch_fault_brush; //!< body paint brush when goalie catch fault
    QBrush M_collide_ball_brush; //!< body paint brush when collision with ball
    QBrush M_collide_player_brush; //!< body paint brush when collision with player
    QPen M_tackle_pen; //!< edge pen when tackling
    QBrush M_tackle_brush; //!< body paint brush when tackling
    QBrush M_tackle_fault_brush; //!< body paint brush when tackle fault

    //! kick accel info pen
    QPen M_kick_accel_pen;
    //! tackle area rectangle line pen
    QPen M_tackle_area_pen;

    //--------------------------------------------------
    // misc

    // mouse measure
    QColor M_measure_color; //!< color of mouse measure pen
    QPen M_measure_pen;
    QFont M_measure_font;  //!< mouse measure font
    QPen M_measure_font_pen; //!< color of mouse measure font
    QColor M_measure_font_color; //!< color of mouse measure font

    //--------------------------------------------------
    // debug

    // ball
    QPen M_debug_ball_pen;
    QBrush M_debug_ball_brush;

    // ball trace
    QPen M_debug_ball_trace_pen;
    QBrush M_debug_ball_trace_brush;

    // player
    QPen M_debug_player_pen; //!< debug info player's body edge pen
    QPen M_debug_body_dir_pen; //!< debug info player's body direction
    QBrush M_debug_self_brush;
    QBrush M_debug_teammate_brush;
    QBrush M_debug_opponent_brush;
    QBrush M_debug_unknown_teammate_brush;
    QBrush M_debug_unknown_opponent_brush;
    QBrush M_debug_unknown_player_brush;

    // target
    QPen M_debug_target_pen;

    // line, triangle, rectanble, circle
    QPen M_debug_figure_pen;

    // message
    QFont M_debug_comment_font;
    QPen M_debug_comment_font_pen;
    QColor M_debug_comment_font_color;
    QFont M_debug_message_font;
    QPen M_debug_message_font_pen;
    QColor M_debug_message_font_color; //!< debug message foreground color


    //! constructor
    DrawConfig();
    DrawConfig( const DrawConfig & );
    const DrawConfig & operator=( const DrawConfig & );
public:
    //! destructor
    ~DrawConfig();

    //! singleton interface
    static
    DrawConfig & instance()
      {
          static DrawConfig s_instance;
          return s_instance;
      }

    //--------------------------------------------------
    //! load configuration from file.
    void loadSettings();
    //! save configuration to file.
    void saveSettings();

    void resizeScoreBoardFont( const int size );

private:

    //! convert config string to color object
    QColor toColor( const QString & conf_str );
    //! convert clor object to config string
    QString toString( const QColor & color );

public:

    //--------------------------------------------------
    // accessor

    const
    QPen & transparentPen() const
      {
          return M_transparent_pen;
      }

    const
    QBrush & transparentBrush() const
      {
          return M_transparent_brush;
      }

    //! get const reference
    const
    QBrush & scoreBoardBrush() const
      {
          return M_score_board_brush;
      }
    //! get const reference
    const
    QFont & scoreBoardFont() const
      {
          return M_score_board_font;
      }

    const
    QPen & scoreBoardFontPen() const
      {
          return M_score_board_font_pen;
      }
    /*
    const
    QColor & scoreBoardFontColor() const
      {
          return M_score_board_font_color;
      }
    */

    //! get const reference
    const
    QBrush & fieldBrush() const
      {
          return M_field_brush;
      }
    //! get const reference
    const
    QBrush & fieldDarkBrush() const
      {
          return M_field_dark_brush;
      }
    //! get const reference
    const
    QPen & linePen() const
      {
          return M_line_pen;
      }

    //! get const reference
    const
    QPen & ballPen() const
      {
          return M_ball_pen;
      }
    //! get const reference
    const
    QBrush & ballBrush() const
      {
          return M_ball_brush;
      }
    //! get const reference
    const
    QFont & ballFont() const
      {
          return M_ball_font;
      }

    const
    QPen & ballFontPen() const
      {
          return M_ball_font_pen;
      }
    /*
    const
    QColor & ballFontColor() const
      {
          return M_ball_font_color;
      }
    */
    //! get const reference
    const
    QPen & playerPen() const
      {
          return M_player_pen;
      }
    //! get const reference
    const
    QPen & realBodyPen() const
      {
          return M_real_body_pen;
      }
    //! get const reference
    const
    QBrush & shadowBrush() const
      {
          return M_shadow_brush;
      }
    const
    QBrush & shadowBrush( const int level ) const
      {
          if ( level < 0 || 255 < level ) return M_shadow_brush;
          return M_shadow_brushes[level];
      }

    //! get const reference
    const
    QPen & leftTeamPen() const
      {
          return M_left_team_pen;
      }
    //! get const reference
    const
    QBrush & leftTeamBrush() const
      {
          return M_left_team_brush;
      }
    //! get const reference
    const
    QPen & leftGoaliePen() const
      {
          return M_left_goalie_pen;
      }
    //! get const reference
    const
    QBrush & leftGoalieBrush() const
      {
          return M_left_goalie_brush;
      }

    //! get const reference
    const
    QPen & rightTeamPen() const
      {
          return M_right_team_pen;
      }
    //! get const reference
    const
    QBrush & rightTeamBrush() const
      {
          return M_right_team_brush;
      }
    //! get const reference
    const
    QPen & rightGoaliePen() const
      {
          return M_right_goalie_pen;
      }
    //! get const reference
    const
    QBrush & rightGoalieBrush() const
      {
          return M_right_goalie_brush;
      }

    //! get const reference
    const
    QFont & playerFont() const
      {
          return M_player_font;
      }

    const
    QPen & playerNumberFontPen() const
      {
          return M_player_number_font_pen;
      }
    /*
    const
    QColor & playerNumberFontColor() const
      {
          return M_player_number_font_color;
      }
    */
    const
    QPen & playerStaminaFontPen() const
      {
          return M_player_stamina_font_pen;
      }
    /*
    const
    QColor & playerStaminaFontColor() const
      {
          return M_player_stamina_font_color;
      }
    */
    const
    QPen & playerDetailFontPen() const
      {
          return M_player_detail_font_pen;
      }
    /*
    const
    QColor & playerDetailFontColor() const
      {
          return M_player_detail_font_color;
      }
    */
    const
    QPen & selectedPlayerFontPen() const
      {
          return M_selected_player_font_pen;
      }
    /*
    const
    QColor & selectedPlayerFontColor() const
      {
          return M_selected_player_font_color;
      }
    */
    //! get const reference
    const
    QPen & effortDecayedPen() const
      {
          return M_effort_decayed_pen;
      }
    //! get const reference
    const
    QPen & recoveryDecayedPen() const
      {
          return M_recovery_decayed_pen;
      }

    //! get const reference
    const
    QPen & viewConePen() const
      {
          return M_view_cone_pen;
      }
    //! get const reference
    const
    QPen & viewAreaPen() const
      {
          return M_view_area_pen;
      }

    //! get const reference
    const
    QPen & kickPen() const
      {
          return M_kick_pen;
      }
    //! get const reference
    const
    QPen & kickFaultPen() const
      {
          return M_kick_fault_pen;
      }
    //! get const reference
    const
    QBrush & kickFaultBrush() const
      {
          return M_kick_fault_brush;
      }
    //! get const reference
    const
    QBrush & catchBrush() const
      {
          return M_catch_brush;
      }
    //! get const reference
    const
    QBrush & catchFaultBrush() const
      {
          return M_catch_fault_brush;
      }
    //! get const reference
    const
    QBrush & collideBallBrush() const
      {
          return M_collide_ball_brush;
      }
    //! get const reference
    const
    QBrush & collidePlayerBrush() const
      {
          return M_collide_player_brush;
      }
    //! get const reference
    const
    QPen & tacklePen() const
      {
          return M_tackle_pen;
      }
    //! get const reference
    const
    QBrush & tackleBrush() const
      {
          return M_tackle_brush;
      }
    //! get const reference
    const
    QBrush & tackleFaultBrush() const
      {
          return M_tackle_fault_brush;
      }

    //! get const reference
    const
    QPen & kickAccelPen() const
      {
          return M_kick_accel_pen;
      }
    //! get const reference
    const
    QPen & tackleAreaPen() const
      {
          return M_tackle_area_pen;
      }

    //! get const reference
    const
    QColor & measureColor() const
      {
          return M_measure_color;
      }
    //! get const reference
    const
    QPen & measurePen() const
      {
          return M_measure_pen;
      }

    //! get const reference
    const
    QFont & measureFont() const
      {
          return M_measure_font;
      }

    const
    QPen & measureFontPen() const
      {
          return M_measure_font_pen;
      }
    /*
    const
    QColor & measureFontColor() const
      {
          return M_measure_font_color;
      }
    */
    const
    QPen & debugBallPen() const
      {
          return M_debug_ball_pen;
      }
    const
    QBrush & debugBallBrush() const
      {
          return M_debug_ball_brush;
      }

    const
    QPen & debugBallTracePen() const
      {
          return M_debug_ball_trace_pen;
      }
    const
    QBrush & debugBallTraceBrush() const
      {
          return M_debug_ball_trace_brush;
      }

    const
    QPen & debugPlayerPen() const
      {
          return M_debug_player_pen;
      }
    const
    QPen & debugBodyDirPen() const
      {
          return M_debug_body_dir_pen;
      }
    const
    QBrush & debugSelfBrush() const
      {
          return M_debug_self_brush;
      }
    const
    QBrush & debugTeammateBrush() const
      {
          return M_debug_teammate_brush;
      }
    const
    QBrush & debugOpponentBrush() const
      {
          return M_debug_opponent_brush;
      }
    const
    QBrush & debugUnknownTeammateBrush() const
      {
          return M_debug_unknown_teammate_brush;
      }
    const
    QBrush & debugUnknownOpponentBrush() const
      {
          return M_debug_unknown_opponent_brush;
      }
    const
    QBrush & debugUnknownPlayerBrush() const
      {
          return M_debug_unknown_player_brush;
      }
    const
    QPen & debugTargetPen() const
      {
          return M_debug_target_pen;
      }
    const
    QPen & debugFigurePen() const
      {
          return M_debug_figure_pen;
      }

    const
    QFont & debugCommentFont() const
      {
          return M_debug_comment_font;
      }
    const
    QPen & debugCommentFontPen() const
      {
          return M_debug_comment_font_pen;
      }
    /*
    const
    QColor & debugCommentFontColor() const
      {
          return M_debug_comment_font_color;
      }
    */
    const
    QFont & debugMessageFont() const
      {
          return M_debug_message_font;
      }
    const
    QPen & debugMessageFontPen() const
      {
          return M_debug_message_font_pen;
      }
    /*
    const
    QColor & debugMessageFontColor() const
      {
          return M_debug_message_font_color;
      }
    */


    //! get const reference
    const
    QBrush & getPlayerBodyBrush( const rcsc::SideID side,
                                 const bool goalie ) const;
};

#endif
