// -*-c++-*-

/*!
  \file ball_painter.cpp
  \brief ball painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "ball_painter.h"

#include "draw_config.h"

#include "main_data.h"
#include "monitor_view_data.h"
#include "app_config.h"

#include <rcsc/common/server_param.h>
#include <rcsc/geom/vector_2d.h>

#include <vector>

/*-------------------------------------------------------------------*/
/*!

*/
void
BallPainter::draw( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    if ( ! vconf.isShownBall() )
    {
        return;
    }

    MonitorViewConstPtr view
        = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! view )
    {
        return;
    }

    const DrawConfig & dconf = DrawConfig::instance();

    // decide radius
    const int ball_radius
        = std::max( 1,
                    ( vconf.isEnlarged()
                      ? vconf.scale( AppConfig::instance().enlargedBallSize() )
                      : vconf.scale( rcsc::ServerParam::i().ballSize() ) )
                    );
    const int ball_diameter = ball_radius * 2;
    const int ix = vconf.getScreenX( view->ball().x() );
    const int iy = vconf.getScreenY( view->ball().y() );


    // set GDI objects
    painter.setPen( dconf.transparentPen() );
    painter.setBrush( dconf.ballBrush() );

    // draw ball body
    painter.drawEllipse( ix - ball_radius,
                         iy - ball_radius,
                         ball_diameter,
                         ball_diameter );

    // draw setplay ball owner team color circle
    if ( view->playmode().isLeftSetPlay() )
    {
        painter.setBrush( dconf.leftTeamBrush() );
        int color_radius = std::max( 1, ball_radius - 2 );
        painter.drawEllipse( ix - color_radius,
                             iy - color_radius,
                             color_radius * 2,
                             color_radius * 2 );
    }
    else if ( view->playmode().isRightSetPlay() )
    {
        painter.setBrush( dconf.rightTeamBrush() );
        int color_radius = std::max( 1, ball_radius - 2 );
        painter.drawEllipse( ix - color_radius,
                             iy - color_radius,
                             color_radius * 2,
                             color_radius * 2 );
    }

    // draw additional circle
    if ( ! vconf.isEnlarged()
         || ball_radius <= 1  )
    {
        painter.setPen( dconf.ballPen() );
        painter.setBrush( dconf.transparentBrush() );

        int kickable_radius
            = vconf.scale( rcsc::ServerParam::i().defaultKickableMargin() );
        painter.drawEllipse( ix - kickable_radius,
                             iy - kickable_radius,
                             kickable_radius * 2,
                             kickable_radius * 2 );
    }

    // draw future status
    if ( vconf.ballFutureCycle() > 0
         && view->ball().hasDelta() )
    {
        drawFutureState( painter );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
BallPainter::drawFutureState( QPainter & painter ) const
{
    MonitorViewConstPtr view
        = M_main_data.getViewData( M_main_data.viewIndex() );

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();
    const rcsc::ServerParam & sparam = rcsc::ServerParam::i();

    rcsc::Vector2D bpos( view->ball().x(),
                         view->ball().y() );
    rcsc::Vector2D bvel( view->ball().deltaX(),
                         view->ball().deltaY() );

    QPoint first_point( vconf.getScreenX( bpos.x ),
                        vconf.getScreenY( bpos.y ) );
    QPoint last_point = first_point;
    const int last = vconf.ballFutureCycle();

    painter.setPen( dconf.ballPen() );
    painter.setBrush( dconf.transparentBrush() );

    for ( int i = 0; i < last; ++i )
    {
        bpos += bvel;
        bvel *= sparam.ballDecay();

        QPoint pt( vconf.getScreenX( bpos.x ),
                   vconf.getScreenY( bpos.y ) );
        if ( std::abs( last_point.x() - pt.x() ) < 1
             && std::abs( last_point.y() - pt.y() ) < 1 )
        {
            break;
        }
        last_point = pt;
        painter.drawEllipse( pt.x() - 1,
                             pt.y() - 1,
                             3,
                             3 );
    }

    painter.drawLine( first_point, last_point );
}
