// -*-c++-*-

/*!
  \file ball_trace_painter.cpp
  \brief ball trace painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "ball_trace_painter.h"

#include "draw_config.h"

#include "main_data.h"
#include "monitor_view_data.h"


/*-------------------------------------------------------------------*/
/*!

*/
BallTracePainter::BallTracePainter( const MainData & main_data )
    : M_main_data( main_data )
    , M_line_trace( false )
    , M_center_x( 0 )
    , M_center_y( 0 )
    , M_field_scale( 0.0 )
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
BallTracePainter::draw( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    if ( ! vconf.isShownBallTrace() )
    {
        return;
    }

    const ViewHolder & holder = M_main_data.viewHolder();

    long start_cycle = vconf.ballTraceStart();
    long end_cycle = vconf.ballTraceEnd();

    const MonitorViewCont & vc = holder.monitorViewCont();

    if ( vc.empty() )
    {
        return;
    }

    if ( vconf.isBallAutoTrace() )
    {
        MonitorViewConstPtr view = M_main_data.getViewData( M_main_data.viewIndex() );
        if ( ! view )
        {
            return;
        }
        start_cycle = std::max( 1L, view->cycle() + vconf.autoTraceStart() );
    }
    else if ( start_cycle >= end_cycle )
    {
        return;
    }

    const std::size_t first = holder.getIndexOf( start_cycle );
    const std::size_t last = ( vconf.isBallAutoTrace()
                               ? std::min( first + vconf.autoTracePeriod(),
                                           vc.size() - 1 )
                               : holder.getIndexOf( end_cycle ) );
    if ( first >= last )
    {
        return;
    }

    const bool line_trace = vconf.isLineTrace();

    painter.setBrush( DrawConfig::instance().transparentBrush() );

    const QPen & bpen = DrawConfig::instance().ballPen();
    QPen black_dot_pen( Qt::black );
    black_dot_pen.setStyle( Qt::DotLine );

    std::size_t i = first;
    int prev_x = vconf.getScreenX( vc[i]->ball().x() );
    int prev_y = vconf.getScreenY( vc[i]->ball().y() );
    ++i;
    for ( ; i <= last; ++i )
    {
        switch ( vc[i]->playmode().mode() ) {
        case rcsc::PM_BeforeKickOff:
        case rcsc::PM_TimeOver:
        case rcsc::PM_KickOff_Left:
        case rcsc::PM_KickOff_Right:
            prev_x = vconf.getScreenX( 0.0 );
            prev_y = vconf.getScreenY( 0.0 );
            continue;
        case rcsc::PM_PlayOn:
        case rcsc::PM_AfterGoal_Left:
        case rcsc::PM_AfterGoal_Right:
            painter.setPen( bpen );
            break;
        default:
            painter.setPen( black_dot_pen );
            break;
        }

        const Ball & b = vc[i]->ball();
        int ix = vconf.getScreenX( b.x() );
        int iy = vconf.getScreenY( b.y() );

        if ( line_trace )
        {
            painter.drawLine( prev_x, prev_y, ix, iy );
        }
        else
        {
            painter.drawLine( prev_x, prev_y, ix, iy );
            painter.drawEllipse( ix - 2, iy - 2, 5, 5 );
        }
        prev_x = ix;
        prev_y = iy;
    }

}
