// -*-c++-*-

/*!
  \file field_canvas.cpp
  \brief main field canvas class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "field_canvas.h"

#include "draw_config.h"
#include "field_painter.h"
#include "score_board_painter.h"
#include "ball_painter.h"
#include "player_painter.h"
#include "offside_line_painter.h"
#include "ball_trace_painter.h"
#include "player_control_painter.h"
#include "player_trace_painter.h"
#include "voronoi_diagram_painter.h"
#include "debug_painter.h"
// model
#include "main_data.h"
#include "app_config.h"

#include <rcsc/common/server_param.h>

#include <algorithm>
#include <iostream>
#include <cmath>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
FieldCanvas::FieldCanvas( MainData & main_data,
                          QWidget * parent )
    : QWidget( parent, 0, Qt::WNoAutoErase )
    , M_main_data( main_data )
    , M_redraw_all( true )
    , M_normal_menu( static_cast< QPopupMenu * >( 0 ) )
    , M_system_menu( static_cast< QPopupMenu * >( 0 ) )
    , M_monitor_menu( static_cast< QPopupMenu * >( 0 ) )
{
    //this->setPalette( M_main_data.drawConfig().fieldBrush().color() );
    //this->setAutoFillBackground( true );

    // need for the MouseMoveEvent
    this->setMouseTracking( true );

    //this->setFocusPolicy( QWidget::StrongFocus );
    this->setFocusPolicy( QWidget::WheelFocus );

    // paint directory
    //this->setAttribute( Qt::WA_PaintOnScreen );
    M_field_painter
        = boost::shared_ptr< FieldPainter >
        ( new FieldPainter( M_main_data.viewConfig() ) );
    M_painters.push_back( boost::shared_ptr< PainterInterface >
                          ( new BallTracePainter( M_main_data ) ) );
    M_painters.push_back( boost::shared_ptr< PainterInterface >
                          ( new PlayerTracePainter( M_main_data ) ) );
    M_painters.push_back( boost::shared_ptr< PainterInterface >
                          ( new PlayerPainter( M_main_data ) ) );
    M_painters.push_back( boost::shared_ptr< PainterInterface >
                          ( new BallPainter( M_main_data ) ) );
    M_painters.push_back( boost::shared_ptr< PainterInterface >
                          ( new OffsideLinePainter( M_main_data ) ) );
    M_painters.push_back( boost::shared_ptr< PainterInterface >
                          ( new PlayerControlPainter( M_main_data ) ) );
    M_painters.push_back( boost::shared_ptr< PainterInterface >
                          ( new VoronoiDiagramPainter( M_main_data ) ) );
    M_painters.push_back( boost::shared_ptr< PainterInterface >
                          ( new DebugPainter( M_main_data ) ) );
    M_painters.push_back( boost::shared_ptr< PainterInterface >
                          ( new ScoreBoardPainter( M_main_data ) ) );

    M_canvas_pixmap.resize( this->size() );
}

/*-------------------------------------------------------------------*/
/*!

*/
FieldCanvas::~FieldCanvas()
{
    //std::cerr << "delete FieldCanvas" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::setNormalMenu( QPopupMenu * menu )
{
    if ( M_normal_menu )
    {
        delete M_normal_menu;
        M_normal_menu = static_cast< QPopupMenu * >( 0 );
    }

    M_normal_menu = menu;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::setSystemMenu( QPopupMenu * menu )
{
    if ( M_system_menu )
    {
        delete M_system_menu;
        M_system_menu = static_cast< QPopupMenu * >( 0 );
    }

    M_system_menu = menu;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::setMonitorMenu( QPopupMenu * menu )
{
    if ( M_monitor_menu )
    {
        delete M_monitor_menu;
        M_monitor_menu = static_cast< QPopupMenu * >( 0 );
    }

    M_monitor_menu = menu;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::setRedrawAllFlag()
{
    M_redraw_all = true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::dropBall()
{
    emit dropBall( M_mouse_state[0].pressedPoint() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::freeKickLeft()
{
    emit freeKickLeft( M_mouse_state[0].pressedPoint() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::freeKickRight()
{
    emit freeKickRight( M_mouse_state[0].pressedPoint() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::paintEvent( QPaintEvent * event )
{
    QWidget::paintEvent( event );
#if 1
    if ( M_canvas_pixmap.isNull()
         || M_canvas_pixmap.size() != this->size() )
    {
        M_canvas_pixmap.resize( this->size() );
    }
#else
    QRect rect = event->rect();
    QSize new_size = rect.size().expandedTo( M_canvas_pixmap.size() );
    M_canvas_pixmap.resize( new_size );
    M_canvas_pixmap.fill( this, rect.topLeft() );
#endif
    M_main_data.update( this->width(),
                        this->height() );

    QPainter painter;
    painter.begin( &M_canvas_pixmap, this );
    //painter.translate( -rect.x(), -rect.y() );

    draw( painter );

    // draw mouse measure
    if ( M_mouse_state[2].isDragged() )
    {
        drawMouseMeasure( painter );
    }

    painter.end();
#if 1
    bitBlt( this, 0, 0,
            &M_canvas_pixmap, 0, 0, this->width(), this->height() );
#else
    bitBlt( this, rect.x(), rect.y(),
            &M_canvas_pixmap, 0, 0, rect.width(), rect.height() );
#endif
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::mouseDoubleClickEvent( QMouseEvent * event )
{
    if ( event->button() == Qt::LeftButton )
    {
        emit focusChanged( event->pos() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::mousePressEvent( QMouseEvent * event )
{
    if ( event->button() == Qt::LeftButton )
    {
        M_mouse_state[1].setMenuFailed( false );
        M_mouse_state[2].setMenuFailed( false );

        M_mouse_state[0].pressed( event->pos() );
        if ( event->state() & Qt::ControlButton )
        {
            emit focusChanged( event->pos() );
        }
        else if ( M_mouse_state[2].isPressed()
                  && M_mouse_state[2].isDragged() )
        {
            M_measure_ball_speed_point = event->pos();
        }
    }
    else if ( event->button() == Qt::MidButton )
    {
        M_mouse_state[0].setMenuFailed( false );
        M_mouse_state[2].setMenuFailed( false );

        M_mouse_state[1].pressed( event->pos() );
    }
    else if ( event->button() == Qt::RightButton )
    {
        M_mouse_state[0].setMenuFailed( false );
        M_mouse_state[1].setMenuFailed( false );

        M_mouse_state[2].pressed( event->pos() );
        M_measure_ball_speed_point = event->pos();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::mouseReleaseEvent( QMouseEvent * event )
{
    if ( event->button() == Qt::LeftButton )
    {
        M_mouse_state[0].released();

        if ( M_mouse_state[0].isMenuFailed() )
        {
            M_mouse_state[0].setMenuFailed( false );
        }
        else if ( M_mouse_state[0].pressedPoint() == event->pos() )
        {
            if ( AppConfig::instance().monitorClientMode() )
            {
                if ( M_monitor_menu
                     && M_monitor_menu->exec( event->globalPos() ) == -1 )
                {
                    M_mouse_state[0].setMenuFailed( true );
                }
            }
        }
    }
    else if ( event->button() == Qt::MidButton )
    {
        M_mouse_state[1].released();
    }
    else if ( event->button() == Qt::RightButton )
    {
        M_mouse_state[2].released();

        if ( M_mouse_state[2].isMenuFailed() )
        {
            M_mouse_state[2].setMenuFailed( false );
        }
        else if ( M_mouse_state[2].pressedPoint() == event->pos() )
        {
            if ( AppConfig::instance().monitorClientMode() )
            {
                if ( M_system_menu
                     && M_system_menu->exec( event->globalPos() ) == -1 )
                {
                    M_mouse_state[2].setMenuFailed( true );
                }
            }
            else
            {
                if ( M_normal_menu
                     && M_normal_menu->exec( event->globalPos() ) == -1 )
                {
                    M_mouse_state[2].setMenuFailed( true );
                }
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::mouseMoveEvent( QMouseEvent * event )
{
    for ( int i = 0; i < 3; ++i )
    {
        M_mouse_state[i].moved( event->pos() );
    }

    if ( M_mouse_state[2].isDragged() )
    {
#if 0
        this->update();
#else
        static QRect s_last_rect;
        QRect new_rect
            = QRect( M_mouse_state[2].pressedPoint(),
                     M_mouse_state[2].draggedPoint() ).normalize();
        new_rect.setLeft( new_rect.left() - 32 );
        new_rect.setTop( new_rect.top() - 32 );
        new_rect.setRight( new_rect.right() + 32 );
        new_rect.setBottom( new_rect.bottom() + 32 );
        if ( new_rect.right() < M_mouse_state[2].draggedPoint().x() + 256 )
        {
            new_rect.setRight( M_mouse_state[2].draggedPoint().x() + 256 );
        }
        // draw mouse measure
        this->update( s_last_rect.unite( new_rect ) );
        s_last_rect = new_rect;
#endif
    }

    emit mouseMoved( event->pos() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::draw( QPainter & painter )
{
    if ( DrawConfig::instance().scoreBoardFont().pointSize()
         != M_main_data.viewConfig().scoreBoardFontSize() )
    {
        DrawConfig::instance().resizeScoreBoardFont
            ( M_main_data.viewConfig().scoreBoardFontSize() );
    }

    M_field_painter->draw( painter );

    if ( ! M_main_data.getViewData( M_main_data.viewIndex() ) )
    {
        return;
    }

    for ( std::vector< boost::shared_ptr< PainterInterface > >::iterator
              it = M_painters.begin();
          it != M_painters.end();
          ++it )
    {
        (*it)->draw( painter );
    }

    M_redraw_all = false;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::drawMouseMeasure( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    QPoint start_point = M_mouse_state[2].pressedPoint();
    QPoint end_point = M_mouse_state[2].draggedPoint();

    // draw straight line
    painter.setPen( DrawConfig::instance().measurePen() );
    painter.setBrush( DrawConfig::instance().transparentBrush() );
    painter.drawLine( start_point, end_point );

    painter.setPen( Qt::red );

    painter.drawEllipse( start_point.x() - 1,
                         start_point.y() - 1,
                         3,
                         3 );
    painter.drawEllipse( end_point.x() - 1,
                         end_point.y() - 1,
                         3,
                         3 );

    rcsc::Vector2D start_real( vconf.getFieldX( start_point.x() ),
                               vconf.getFieldY( start_point.y() ) );
    rcsc::Vector2D end_real( vconf.getFieldX( end_point.x() ),
                             vconf.getFieldY( end_point.y() ) );

    // ball travel marks
    {
        double ball_speed = rcsc::ServerParam::i().ballSpeedMax();

        if ( ! M_measure_ball_speed_point.isNull()
             && M_measure_ball_speed_point != start_point )
        {
            rcsc::Vector2D vel( std::abs( M_measure_ball_speed_point.x()
                                          - start_point.x() )
                                / vconf.fieldScale(),
                                std::abs( M_measure_ball_speed_point.y()
                                          - start_point.y() )
                                / vconf.fieldScale() );
            ball_speed = vel.r();
            if ( ball_speed > rcsc::ServerParam::i().ballSpeedMax() )
            {
                ball_speed = rcsc::ServerParam::i().ballSpeedMax();
            }
        }

        rcsc::Vector2D ball_pos = start_real;
        rcsc::Vector2D ball_vel = end_real - start_real;
        ball_vel.setLength( ball_speed );

        const double max_length = start_real.dist( end_real );
        double total_travel = 0.0;

        ball_pos += ball_vel;
        ball_vel *= rcsc::ServerParam::i().ballDecay();
        total_travel += ball_vel.r();
        QPoint last_pt( start_point.x() - 2,
                        start_point.y() - 2 );
        while ( total_travel < max_length )
        {
            QPoint pt( vconf.getScreenX( ball_pos.x ) - 1,
                       vconf.getScreenY( ball_pos.y ) - 1 );
            if ( std::abs( pt.x() - last_pt.x() ) < 1
                 && std::abs( pt.y() - last_pt.y() ) < 1 )
            {
                break;
            }
            last_pt = pt;
            painter.drawEllipse( pt.x(),
                                 pt.y(),
                                 3,
                                 3 );
            ball_pos += ball_vel;
            ball_vel *= rcsc::ServerParam::i().ballDecay();
            ball_speed *= rcsc::ServerParam::i().ballDecay();
            total_travel += ball_speed;
        }
    }

    // draw distance & angle text
    painter.setFont( DrawConfig::instance().measureFont() );
    painter.setPen( DrawConfig::instance().measureFontPen() );
    //painter.setBackgroundMode( Qt::TransparentMode );

    char buf[64];
    std::snprintf( buf, 64,
                   "(%.2f,%.2f)",
                   start_real.x,
                   start_real.y );

    painter.drawText( start_point,
                      QString::fromAscii( buf ) );

    if ( std::abs( start_point.x() - end_point.x() ) < 1
         && std::abs( start_point.y() - end_point.y() ) < 1 )
    {
        return;
    }

    std::snprintf( buf, 64,
                   "(%.2f,%.2f)",
                   end_real.x,
                   end_real.y );

    painter.drawText( end_point.x(),
                      end_point.y(),
                      QString::fromAscii( buf ) );

    painter.setPen( QColor( 224, 224, 192 ) );
    rcsc::Vector2D rel( end_real - start_real );
    std::snprintf( buf, 64,
                   "rel(%.2f,%.2f) r%.2f th%.1f",
                   rel.x, rel.y, rel.r(), rel.th().degree() );

    int dist_add_y = ( end_point.y() > start_point.y()
                       ? + painter.fontMetrics().height()
                       : - painter.fontMetrics().height() );
    painter.drawText( end_point.x(),
                      end_point.y() + dist_add_y,
                      QString::fromAscii( buf ) );
}
