// -*-c++-*-

/*!
  \file image_save_dialog.cpp
  \brief Image save dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "image_save_dialog.h"

#include "main_data.h"
#include "main_window.h"
#include "field_canvas.h"
#include "view_holder.h"
//#include "dir_selector.h"

#include <algorithm>
#include <iostream>
#include <cassert>
#include <cstdlib>

/*-------------------------------------------------------------------*/
/*!

*/
ImageSaveDialog::ImageSaveDialog( MainWindow * main_window,
                                  FieldCanvas * field_canvas,
                                  MainData & main_data )
    : QDialog( main_window )
    , M_main_window( main_window )
    , M_field_canvas( field_canvas )
    , M_main_data( main_data )

{
    assert( main_window );

    //this->setWindowTitle( tr( "Image Save" ) ); // qt4
    this->setCaption( tr( "Image Save" ) );

    createControls();

}

/*-------------------------------------------------------------------*/
/*!

*/
ImageSaveDialog::~ImageSaveDialog()
{
    //std::cerr << "delete ImageSaveDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::createControls()
{
    QVBoxLayout * layout = new QVBoxLayout( this );
    layout->setMargin( 4 );

    layout->addWidget( createCycleSelectControls(),
                       0, Qt::AlignLeft );

    layout->addWidget( createFileNameControls(),
                       0, Qt::AlignLeft );

    layout->addWidget( createDirSelectControls(),
                       0, Qt::AlignLeft );

    layout->addLayout( createExecuteControls(),
                       0 );

    //this->setLayout( layout ); // qt4
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ImageSaveDialog::createCycleSelectControls()
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Cycle Range" ), this );

    //QHBoxLayout * layout = new QHBoxLayout( group_box );
    //layout->setSpacing( 0 );

    // create input spinctrls & labels
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();
    int min_cycle = 0;
    int max_cycle = 0;
    if ( ! view.empty() )
    {
        min_cycle = static_cast< int >( view.front()->cycle() );
        max_cycle = static_cast< int >( view.back()->cycle() );
    }

    //layout->addWidget( new QLabel( tr( "Start: " ), group_box ),
    //                   0, Qt::AlignVCenter );
    new QLabel( tr( "Start: " ), group_box );

    M_start_cycle = new QSpinBox( group_box );
    M_start_cycle->setRange( min_cycle, max_cycle );
    M_start_cycle->setValue( min_cycle );
    //layout->addWidget( M_start_cycle,
    //                   0, Qt::AlignVCenter );

    //layout->addWidget( new QLabel( tr( " - End: " ), group_box ),
    //                   0, Qt::AlignVCenter );
    new QLabel( tr( " - End: " ), group_box );

    M_end_cycle = new QSpinBox( group_box );
    M_end_cycle->setRange( min_cycle, max_cycle );
    M_end_cycle->setValue( min_cycle );
    //layout->addWidget( M_end_cycle,
    //                   0, Qt::AlignVCenter );

    //layout->addSpacing( 4 );

    QPushButton * select_all_btn = new QPushButton( tr( "Select All" ),
                                                    group_box );
    select_all_btn->setMaximumSize( 80, this->fontMetrics().height() + 12 );
    connect( select_all_btn, SIGNAL( clicked() ),
             this, SLOT( selectAllCycle() ) );
    //layout->addWidget( select_all_btn,
    //                   0, Qt::AlignVCenter );

    //group_box->setLayout( layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ImageSaveDialog::createFileNameControls()
{
    QHGroupBox * group_box = new QHGroupBox( tr( "File" ), this );

    //QHBoxLayout * layout = new QHBoxLayout( group_box );
    //layout->setSpacing( 0 );

    //layout->addWidget( new QLabel( tr( "Name Prefix: " ), group_box ),
    //                   0, Qt::AlignVCenter );
    new QLabel( tr( "Name Prefix: " ), group_box );

    M_name_prefix = new QLineEdit( tr( "image-" ), group_box );
    M_name_prefix->setMaximumSize( 80, 24 );
    //layout->addWidget( M_name_prefix,
    //                   0, Qt::AlignVCenter );

    //layout->addWidget( new QLabel( tr( " Format: " ), group_box ),
    //                   0, Qt::AlignVCenter );
    new QLabel( tr( " Format: " ), group_box );

    M_format_choice = new QComboBox( group_box );
    {
        int i = 0;
        int png_index = 0;
        int max_width = 0;
        for ( unsigned int i = 0; i < QImageIO::outputFormats().count(); ++i )
        {
            QString format( QImageIO::outputFormats().at( i ) );

            int width = this->fontMetrics().width( format );
            if ( max_width < width )
            {
                max_width = width;
            }

            if ( format == "PNG" )
            {
                png_index = i;
            }
            M_format_choice->insertItem( format );
        }
        M_format_choice->setCurrentItem( png_index );
        M_format_choice->setMaximumWidth( max_width + 40 );
    }
    //layout->addWidget( M_format_choice,
    //                   0, Qt::AlignVCenter );

    //group_box->setLayout( layout ); // qt4
    return group_box;;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ImageSaveDialog::createDirSelectControls()
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Save Directry" ), this );

    //QHBoxLayout * layout = new QHBoxLayout( group_box );

    QString dir_str = QDir::current().absPath();
    M_saved_dir = new QLineEdit( dir_str, group_box );
    M_saved_dir->setMinimumWidth( std::max( 360,
                                            M_saved_dir->fontMetrics().width( dir_str )
                                            + 32 ) );
    //layout->addWidget( M_saved_dir,
    //                   0, Qt::AlignVCenter );

    QPushButton * button = new QPushButton( tr( "..." ), group_box );
    button->setMaximumSize( this->fontMetrics().width( tr( "..." ) ) + 24,
                            this->fontMetrics().height() + 12 );
    button->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
    connect( button, SIGNAL( clicked() ),
             this, SLOT( selectSavedDir() ) );
    //layout->addWidget( button );

    //group_box->setLayout( layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QLayout *
ImageSaveDialog::createExecuteControls()
{
    QHBoxLayout * layout = new QHBoxLayout();

    QPushButton * button = new QPushButton( tr( "Create!" ), this );
    connect( button, SIGNAL( clicked() ),
             this, SLOT( executeSave() ) );
    layout->addWidget( button,
                       10, Qt::AlignVCenter );

    QPushButton * cancel = new QPushButton( tr( "Cancel" ), this );
    connect( cancel, SIGNAL( clicked() ),
             this, SLOT( reject() ) );
    layout->addWidget( cancel,
                       2, Qt::AlignVCenter );

    return layout;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::showEvent( QShowEvent * event )
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();

    int min_cycle = 0;
    int max_cycle = 0;
    if ( ! view.empty() )
    {
        min_cycle = static_cast< int >( view.front()->cycle() );
        max_cycle = static_cast< int >( view.back()->cycle() );
    }
    M_start_cycle->setRange( min_cycle, max_cycle );
    M_end_cycle->setRange( min_cycle, max_cycle );

    //event->accept(); // qt4
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::selectAllCycle()
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();
    int min_cycle = 0;
    int max_cycle = 0;
    if ( ! view.empty() )
    {
        min_cycle = static_cast< int >( view.front()->cycle() );
        max_cycle = static_cast< int >( view.back()->cycle() );
    }

    M_start_cycle->setValue( min_cycle );
    M_end_cycle->setValue( max_cycle );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::selectSavedDir()
{
    QString dir
        = QFileDialog::getExistingDirectory( M_saved_dir->text(),
                                             this,
                                             "get existing directory",
                                             tr( "Choose a saved directory" ),
                                             true, // dir only
                                             false ); // dont floow symlink
    if ( ! dir.isEmpty() )
    {
        M_saved_dir->setText( dir );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::executeSave()
{
    /*
    std::cout << "Create button pressed" << std::endl;
    std::cout << "  start cycle = " << M_start_cycle->GetValue()
              << "  end cycle = " << M_end_cycle->GetValue()
              << std::endl;
    std::cout << "  name prefix = " << M_name_prefix->GetValue()
              << std::endl;
    std::cout << "  format = " << format_choices[M_format_choice->GetSelection()]
              << std::endl;
    std::cout << "  save dir = " << M_saved_dir->GetValue()
              << std::endl;
    */
    QString format = M_format_choice->currentText();

    saveImage( M_start_cycle->value(),
               M_end_cycle->value(),
               M_saved_dir->text(),
               M_name_prefix->text(),
               M_format_choice->currentText() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::saveImage( const int start_cycle,
                            const int end_cycle,
                            const QString & saved_dir,
                            const QString & name_prefix,
                            const QString & format_name )
{
    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        QMessageBox::warning( this,
                              tr( "Error" ),
                              tr( "No Monitor View Data!" ) );
        reject();
        return;
    }

    const int backup_index = M_main_data.viewIndex();

    const int first = M_main_data.viewHolder().getIndexOf( start_cycle );
    const int last = M_main_data.viewHolder().getIndexOf( end_cycle );

    if ( first > last )
    {
        QMessageBox::warning( this,
                              tr( "Error" ),
                              tr( "Invalid cycle range!" ) );
        return;
    }

    // create file path base
    QString file_path = saved_dir;

    if ( ! file_path.endsWith( QChar( '/' ) ) )
    {
        file_path += tr( "/" );
    }

    // check directory
    {
        QDir dir( file_path );
        if ( ! dir.exists()
             && ! dir.mkdir( file_path, true ) )
        {
            QMessageBox::warning( this,
                                  tr( "Error" ),
                                  tr( "Failed to create image save directory!" ) );
            return;
        }
    }

    {
        QString name_prefix_trim = name_prefix;
        while ( ! name_prefix_trim.isEmpty()
                && name_prefix_trim.endsWith( QChar( '/' ) ) )
        {
            name_prefix_trim.remove( name_prefix_trim.length() - 1, 1 );
        }

        file_path += name_prefix_trim;
    }

    QString file_ext = tr( "." ) + format_name.lower();

    const QSize image_size = M_field_canvas->size();
    QPixmap pixmap( image_size );

    std::cerr << "Save image size = "
              << image_size.width() << ", "
              << image_size.height() << std::endl;

    QPainter painter( &pixmap );

    // show progress dialog
    QProgressDialog progress_dialog( this );
    progress_dialog.setCaption( tr( "Image Save Progress" ) );
    progress_dialog.setProgress( 0, last - first );
    progress_dialog.setLabelText( file_path + tr( "00000" ) + file_ext );

    bool confirm = true;

    QString format = format_name.upper();

    // main loop
    for ( int i = first; i <= last; ++i )
    {
        // full file path
        char count[16];
        std::snprintf( count, 16, "%05d", i );

        QString file_path_all = file_path;
        file_path_all += QString::fromAscii( count );
        file_path_all += file_ext;

        if ( confirm
             && QFile::exists( file_path_all ) )
        {
            int result
                = QMessageBox::question( this,
                                         tr( "Overwrite?" ),
                                         tr( "There already exists a file called %1.\n Overwrite?")
                                         .arg( file_path_all ),
                                         QMessageBox::No,
                                         QMessageBox::Yes,
                                         QMessageBox::YesAll );
            if ( result == QMessageBox::No )
            {
                progress_dialog.cancel();
                M_main_data.setViewDataIndex( backup_index );
                return;
            }
            else if ( result == QMessageBox::YesAll )
            {
                confirm = false;
            }
        }

        progress_dialog.setProgress( i - first );
        progress_dialog.setLabelText( file_path_all );

        qApp->processEvents();
        if ( progress_dialog.wasCanceled() )
        {
            M_main_data.setViewDataIndex( backup_index );
            return;
        }

        M_main_data.setViewDataIndex( i );
        M_main_data.update( image_size.width(), image_size.height() );

        M_field_canvas->draw( painter );

        if ( ! pixmap.save( file_path_all, format.ascii() ) )
        {
            QMessageBox::critical( this,
                                   tr( "Error" ),
                                   tr( "Failed to save image file " )
                                   + file_path_all );
            M_main_data.setViewDataIndex( backup_index );
            return;
        }
    }

    M_main_data.setViewDataIndex( backup_index );

    accept();
}
