// -*-c++-*-

/*!
  \file voronoi_diagram_painter.cpp
  \brief Voronoi Diagram & Delaunay Triangulation painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "voronoi_diagram_painter.h"

#include "draw_config.h"
// model
#include "main_data.h"

#include <rcsc/common/server_param.h>
#include <rcsc/geom/line_2d.h>
#include <rcsc/geom/rect_2d.h>

/*-------------------------------------------------------------------*/
/*!

*/
void
VoronoiDiagramPainter::draw( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    if ( ! vconf.isShownVoronoiDiagram()
         && ! vconf.isShownDelaunayTrianglation() )
    {
        return;
    }

    static const rcsc::Rect2D PITCH_RECT( - rcsc::ServerParam::i().pitchHalfLength(),
                                          - rcsc::ServerParam::i().pitchHalfWidth(),
                                          rcsc::ServerParam::i().pitchLength(),
                                          rcsc::ServerParam::i().pitchWidth() );
    static const double max_x_abs = 5000.0;
    static const double max_y_abs = 5000.0;
    static const rcsc::Rect2D MAX_RECT( - max_x_abs,
                                        - max_y_abs,
                                        max_x_abs * 2.0,
                                        max_y_abs * 2.0 );

    MonitorViewConstPtr view = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! view )
    {
        return;
    }

    //rcsc::MSecTimer timer;
    //timer.restart();

    std::vector< rcsc::Vector2D > players_pos;
    players_pos.reserve( 22 );

    {
        const std::vector< Player >::const_iterator end = view->players().end();
        for ( std::vector< Player >::const_iterator it = view->players().begin();
              it != end;
              ++it )
        {
            if ( vconf.voronoiTarget() != rcsc::NEUTRAL
                 && it->side() != vconf.voronoiTarget() )
            {
                continue;
            }
            if ( it->isAlive() )
            {
                players_pos.push_back( rcsc::Vector2D( it->x(), it->y() ) );
            }
        }
    }

    const std::size_t MAX_POINT = players_pos.size();
    if ( MAX_POINT <= 2 )
    {
        return;
    }

    std::vector< rcsc::Vector2D > segment_points;
    std::vector< boost::shared_ptr< VoronoiEdge > > edges;
    for( std::size_t i = 0; i < MAX_POINT - 1; ++i )
    {
        for ( std::size_t j = i + 1; j < MAX_POINT; ++j )
        {
            segment_points.clear();

            // make candidate line
            const rcsc::Line2D first_perpend
                = rcsc::Line2D::perpendicular_bisector( players_pos[i],
                                                        players_pos[j] );
            rcsc::Vector2D left_edge, right_edge;
            MAX_RECT.intersection( first_perpend, &left_edge, &right_edge );
            segment_points.push_back( left_edge );
            segment_points.push_back( right_edge );

            // to make the segment, check all other points
            // register segment points
            for ( std::size_t k = 0; k < MAX_POINT; ++k )
            {
                if ( k == i || k == j )
                {
                    continue;
                }

                rcsc::Line2D k_perpend
                    = rcsc::Line2D::perpendicular_bisector( players_pos[i],
                                                            players_pos[k] );
                rcsc::Vector2D perpend_intersect
                    = rcsc::Line2D::intersection( first_perpend, k_perpend );
                if ( ! perpend_intersect )
                {
                    continue; // intersection does not exist
                }
                if ( perpend_intersect.absX() > max_x_abs
                     || perpend_intersect.absY() > max_y_abs )
                {
                    continue; // intersection is out of pitch
                }
                segment_points.push_back( perpend_intersect );
            }
            // sort segment points
            if ( std::fabs( first_perpend.b() ) < 0.001 )
            {
                std::sort( segment_points.begin(), segment_points.end(),
                           rcsc::Vector2D::YCmp() );
            }
            else
            {
                std::sort( segment_points.begin(), segment_points.end(),
                           rcsc::Vector2D::XCmp() );
            }

            const std::vector< rcsc::Vector2D >::iterator end = segment_points.end() - 1;
            for ( std::vector< rcsc::Vector2D >::iterator it = segment_points.begin();
                  it != end;
                  ++it )
            {
                std::vector< rcsc::Vector2D >::iterator it_next = it + 1;
                rcsc::Vector2D segment_center = *it + *it_next;
                segment_center *= 0.5;
                bool is_draw_segment = true;

                const double dist2_to_first = segment_center.dist2( players_pos[i] );
                for ( std::size_t l = 0; l < MAX_POINT; ++l )
                {
                    if ( l == i || l == j )
                    {
                        continue;
                    }
                    if ( segment_center.dist2( players_pos[l] ) < dist2_to_first )
                    {
                        is_draw_segment = false;
                        break;
                    }
                }

                if ( is_draw_segment )
                {
                    boost::shared_ptr< VoronoiEdge > ptr( new VoronoiEdge( i, j,
                                                                           it->x, it->y,
                                                                           it_next->x, it_next->y ) );
                    if ( ptr->point1.x < PITCH_RECT.left() )
                    {
                        ptr->point1.x = PITCH_RECT.left();
                        ptr->point1.y = first_perpend.getY( ptr->point1.x );
                    }
                    else if ( PITCH_RECT.right() < ptr->point1.x )
                    {
                        ptr->point1.x = PITCH_RECT.right();
                        ptr->point1.y = first_perpend.getY( ptr->point1.x );
                    }
                    if ( ptr->point1.y < PITCH_RECT.top() )
                    {
                        ptr->point1.y = PITCH_RECT.top();
                        ptr->point1.x = first_perpend.getX( ptr->point1.y );
                    }
                    else if ( PITCH_RECT.bottom() < ptr->point1.y )
                    {
                        ptr->point1.y = PITCH_RECT.bottom();
                        ptr->point1.x = first_perpend.getX( ptr->point1.y );
                    }

                    if ( ptr->point2.x < PITCH_RECT.left() )
                    {
                        ptr->point2.x = PITCH_RECT.left();
                        ptr->point2.y = first_perpend.getY( ptr->point2.x );
                    }
                    else if ( PITCH_RECT.right() < ptr->point2.x )
                    {
                        ptr->point2.x = PITCH_RECT.right();
                        ptr->point2.y = first_perpend.getY( ptr->point2.x );
                    }
                    if ( ptr->point2.y < PITCH_RECT.top() )
                    {
                        ptr->point2.y = PITCH_RECT.top();
                        ptr->point2.x = first_perpend.getX( ptr->point2.y );
                    }
                    else if ( PITCH_RECT.bottom() < ptr->point2.y )
                    {
                        ptr->point2.y = PITCH_RECT.bottom();
                        ptr->point2.x = first_perpend.getX( ptr->point2.y );
                    }
                    edges.push_back( ptr );
                }
            }
        }
    }

    //std::cerr << "voronoi elapsed " << timer.elapsedReal() << std::endl;
    if ( vconf.isShownVoronoiDiagram() )
    {
        painter.setPen( DrawConfig::instance().measurePen() );
        painter.setBrush( DrawConfig::instance().transparentBrush() );

        const std::vector< boost::shared_ptr< VoronoiEdge > >::iterator end = edges.end();
        for ( std::vector< boost::shared_ptr< VoronoiEdge > >::iterator it = edges.begin();
              it != end;
              ++it )
        {
            painter.drawLine( vconf.getScreenX( (*it)->point1.x ),
                              vconf.getScreenY( (*it)->point1.y ),
                              vconf.getScreenX( (*it)->point2.x ),
                              vconf.getScreenY( (*it)->point2.y ) );
        }
    }

    if ( vconf.isShownDelaunayTrianglation() )
    {
        painter.setPen( DrawConfig::instance().linePen() );
        painter.setBrush( DrawConfig::instance().transparentBrush() );

        const std::vector< boost::shared_ptr< VoronoiEdge > >::iterator end = edges.end();
        for ( std::vector< boost::shared_ptr< VoronoiEdge > >::iterator it = edges.begin();
              it != end;
              ++it )
        {
            painter.drawLine( vconf.getScreenX( players_pos[(*it)->player1].x ),
                              vconf.getScreenY( players_pos[(*it)->player1].y ),
                              vconf.getScreenX( players_pos[(*it)->player2].x ),
                              vconf.getScreenY( players_pos[(*it)->player2].y ) );
        }
    }
}
