// -*-c++-*-

/*!
  \file debug_message_frame.cpp
  \brief debug management frame class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/config.h>   // wxFileConfig
#include <wx/fileconf.h> // wxFileConfig
#include <wx/tglbtn.h>
#include <wx/notebook.h>
#endif

#include "debug_message_frame.h"

#include "soccerwindow_app.h"
#include "main_frame.h"
#include "debug_log_dir_dialog.h"
// model
#include "main_data.h"
#include "debug_log_holder.h"
#include "debug_log_data.h"
#include "app_config.h"
#include "id.h"

#include <cassert>
#include <cstdlib> // getenv
#include <iostream>


#include "xpm/debug_view_all.xpm"
#include "xpm/debug_view_self.xpm"
#include "xpm/debug_view_ball.xpm"
#include "xpm/debug_view_player.xpm"
#include "xpm/debug_view_comment.xpm"
#include "xpm/debug_view_line.xpm"
#include "xpm/debug_view_target.xpm"
#include "xpm/debug_view_message.xpm"

#include "xpm/open_dir.xpm"

#include "xpm/sync.xpm"
#include "xpm/logplayer_one_step_back.xpm"
#include "xpm/logplayer_one_step_forward.xpm"

#include "xpm/num_1.xpm"
#include "xpm/num_2.xpm"
#include "xpm/num_3.xpm"
#include "xpm/num_4.xpm"
#include "xpm/num_5.xpm"
#include "xpm/num_6.xpm"
#include "xpm/num_7.xpm"
#include "xpm/num_8.xpm"
#include "xpm/num_9.xpm"
#include "xpm/num_10.xpm"
#include "xpm/num_11.xpm"
#include "xpm/num_12.xpm"
#include "xpm/num_13.xpm"
#include "xpm/num_14.xpm"
#include "xpm/num_15.xpm"
#include "xpm/num_16.xpm"

//#include "xpm/left.xpm"
//#include "xpm/right.xpm"

/*-------------------------------------------------------------------*/
/*!

*/
DebugMessageFrame::DebugMessageFrame( wxWindow * parent,
                                      MainData & main_data )
    : wxFrame( parent,
               SWID_FRAME_DEBUG_MESSAGE,
               wxT( "Debug Message" ),
               wxDefaultPosition, //wxPoint( 32, 32 ),
               wxSize( 640, 480 ),
               wxDEFAULT_FRAME_STYLE // | wxFRAME_FLOAT_ON_PARENT
               )
    , M_main_data( main_data )
    , M_debug_holder( new DebugLogHolder )
    , M_level( 0xFFFF )
{
    createWindows();
    connectEvents();

    loadConfig();
}

/*-------------------------------------------------------------------*/
/*!

*/
DebugMessageFrame::~DebugMessageFrame()
{
    saveConfig();
    //std::cerr << "delete DebugMessageFrame" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::loadConfig()
{
    wxFileConfig & config_file = SoccerWindowApp::get_config_file();

    wxString old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/DebugMessageFrame" ) );

    if ( this->GetToolBar() )
    {
        wxToolBar * tbar = this->GetToolBar();

        for ( int i = 0; i < 16; ++i )
        {
            wxString key;
            key.Printf( wxT( "debug%02d" ), i + 1 );

            long value = 1;
            if ( config_file.Read( key, &value ) )
            {
                int level = DebugLogData::LEVEL_1 << i;
                if ( value )
                {
                    M_level |= level;
                }
                else
                {
                    M_level &= ~level;
                }

                tbar->ToggleTool( SWID_DEBUG_LEVEL_1 + i, value );
            }

        }
    }

    config_file.SetPath( old_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::saveConfig()
{
    wxFileConfig & config_file = SoccerWindowApp::get_config_file();

    wxString old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/DebugMessageFrame" ) );

    if ( this->GetToolBar() )
    {
        wxToolBar * tbar = this->GetToolBar();

        for ( int i = 0; i < 16; ++i )
        {
            wxString key;
            key.Printf( wxT( "debug%02d" ), i + 1 );

            config_file.Write( key,
                               tbar->GetToolState( SWID_DEBUG_LEVEL_1 + i ) ? 1L : 0L );
        }
    }

    config_file.SetPath( old_path );
}

/*-------------------------------------------------------------------*/
/*!
  \brief create all windows that has this mainframe as a parent
*/
void
DebugMessageFrame::createWindows()
{
    createMenuBar();
    createToolBar();

    M_notebook = new wxNotebook( this, SWID_NOTEBOOK_DEBUG_MESSAGE );


    for ( int i = 0; i < 11; ++i )
    {
        M_message[i] = new wxTextCtrl( M_notebook,
                                       -1,
                                       wxT( "" ),
                                       wxDefaultPosition, wxDefaultSize,
                                       wxTE_READONLY | wxTE_MULTILINE );
        wxString number;
        number.Printf( wxT( "   %2d " ), i + 1 );
        M_notebook->AddPage( M_message[i], number );
    }

    /*
#if wxCHECK_VERSION(2, 6, 0)
    // wxNotebookSizer is removed if wxWidgets version is higher than 2.6.0,
    // because wxNotebookSizer is deprecated.
#else
    wxNotebookSizer * sizer = new wxNotebookSizer( M_notebook );
    this->SetSizer( sizer );
    this->SetAutoLayout( true );
#endif
    */

    // set minimal window size
    this->SetSizeHints( 280, 220 );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::createMenuBar()
{
    // create menu
    //wxMenuBar * menubar = new wxMenuBar;
    // append menubar to MainFrame.
    //this->SetMenuBar( menubar );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::createToolBar()
{
    wxToolBar * tbar = this->CreateToolBar( wxTB_HORIZONTAL
                                            | wxTB_FLAT,
                                            SWID_DEBUG_TOOLBAR );
    const ViewConfig & vconf = M_main_data.viewConfig();

    ////////////////////////////////////////////////////////
    // visual debugger switches
    tbar->AddCheckTool( SWID_CANVAS_DRAW_DEBUG_VIEW,
                        _( "Switch" ),
                        wxBitmap( debug_view_all_xpm ),
                        wxNullBitmap,
                        _( "Show/Hide all debug view" ) );
    tbar->ToggleTool( SWID_CANVAS_DRAW_DEBUG_VIEW,
                      vconf.isShownDebugView() );
    tbar->AddCheckTool( SWID_CANVAS_DRAW_DEBUG_VIEW_SELF,
                        wxT( "Self" ),
                        wxBitmap( debug_view_self_xpm ),
                        wxNullBitmap,
                        _( "Show/Hide self" ) );
    tbar->ToggleTool( SWID_CANVAS_DRAW_DEBUG_VIEW_SELF,
                      vconf.isShownDebugViewSelf() );
    tbar->AddCheckTool( SWID_CANVAS_DRAW_DEBUG_VIEW_BALL,
                        wxT( "Ball" ),
                        wxBitmap( debug_view_ball_xpm ),
                        wxNullBitmap,
                        _( "Show/Hide ball" ) );
    tbar->ToggleTool( SWID_CANVAS_DRAW_DEBUG_VIEW_BALL,
                      vconf.isShownDebugViewBall() );
    tbar->AddCheckTool( SWID_CANVAS_DRAW_DEBUG_VIEW_PLAYERS,
                        wxT( "Player" ),
                        wxBitmap( debug_view_player_xpm ),
                        wxNullBitmap,
                        _( "Show/Hide other players" ) );
    tbar->ToggleTool( SWID_CANVAS_DRAW_DEBUG_VIEW_PLAYERS,
                      vconf.isShownDebugViewPlayers() );
    tbar->AddCheckTool( SWID_CANVAS_DRAW_DEBUG_VIEW_COMMENT,
                        wxT( "Comment" ),
                        wxBitmap( debug_view_comment_xpm ),
                        wxNullBitmap,
                        _( "Show/Hide comment" ) );
    tbar->ToggleTool( SWID_CANVAS_DRAW_DEBUG_VIEW_COMMENT,
                      vconf.isShownDebugViewComment() );
    tbar->AddCheckTool( SWID_CANVAS_DRAW_DEBUG_VIEW_FIGURE,
                        wxT( "Line" ),
                        wxBitmap( debug_view_line_xpm ),
                        wxNullBitmap,
                        _( "Show/Hide debug line" ) );
    tbar->ToggleTool( SWID_CANVAS_DRAW_DEBUG_VIEW_FIGURE,
                      vconf.isShownDebugViewFigure() );
    tbar->AddCheckTool( SWID_CANVAS_DRAW_DEBUG_VIEW_TARGET,
                        wxT( "Target" ),
                        wxBitmap( debug_view_target_xpm ),
                        wxNullBitmap,
                        _( "Show/Hide target" ) );
    tbar->ToggleTool( SWID_CANVAS_DRAW_DEBUG_VIEW_TARGET,
                      vconf.isShownDebugViewTarget() );
    tbar->AddCheckTool( SWID_CANVAS_DRAW_DEBUG_VIEW_MESSAGE,
                        wxT( "Message" ),
                        wxBitmap( debug_view_message_xpm ),
                        wxNullBitmap,
                        _( "Show/Hide debug message" ) );
    tbar->ToggleTool( SWID_CANVAS_DRAW_DEBUG_VIEW_MESSAGE,
                      vconf.isShownDebugViewMessage() );
    tbar->AddSeparator();
    ////////////////////////////////////////////////////////
    // dir selector
    tbar->AddTool( SWID_SHOW_DEBUG_LOG_DIR_DIALOG,
                   wxT( "Open Dir" ),
                   wxBitmap( open_dir_xpm ),
                   _( "Open debug log directory" ) );
    tbar->AddSeparator();
    ////////////////////////////////////////////////////////
    // log message switch

    tbar->AddTool( SWID_DEBUG_LOG_SYNC,
                   wxT( "Sync" ),
                   wxBitmap( sync_xpm ),
                   _( "Synchronize with field canvas" ) );
    tbar->AddTool( SWID_DEBUG_LOG_DECREMENT,
                   wxT( "Decrement" ),
                   wxBitmap( logplayer_one_step_back_xpm ),
                   _( "Decremet message cycle" ) );
    tbar->AddTool( SWID_DEBUG_LOG_INCREMENT,
                   wxT( "Increment" ),
                   wxBitmap( logplayer_one_step_forward_xpm ),
                   _( "Increment message cycle" ) );

    tbar->AddSeparator();
    ////////////////////////////////////////////////////////

    // log level selector
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_1, _( "Level 1" ),
                        wxBitmap( num_1_xpm ), wxNullBitmap,
                        _( "Debug Level 1 System" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_1, M_level & DebugLogData::LEVEL_1 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_2, _( "Level 2" ),
                        wxBitmap( num_2_xpm ), wxNullBitmap,
                        _( "Debug Level 2 Sensor" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_2, M_level & DebugLogData::LEVEL_2 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_3, _( "Level 3" ),
                        wxBitmap( num_3_xpm ), wxNullBitmap,
                        _( "Debug Level 3 WorldModel" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_3, M_level & DebugLogData::LEVEL_3 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_4, _( "Level 4" ),
                        wxBitmap( num_4_xpm ), wxNullBitmap,
                        _( "Debug Level 4 BasicAction" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_4, M_level & DebugLogData::LEVEL_4 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_5, _( "Level 5" ),
                        wxBitmap( num_5_xpm ), wxNullBitmap,
                        _( "Debug Level 5 Intercept" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_5, M_level & DebugLogData::LEVEL_5 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_6, _( "Level 6" ),
                        wxBitmap( num_6_xpm ), wxNullBitmap,
                        _( "Debug Level 6 Kick" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_6, M_level & DebugLogData::LEVEL_6 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_7, _( "Level 7" ),
                        wxBitmap( num_7_xpm ), wxNullBitmap,
                        _( "Debug Level 7 Dribble" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_7, M_level & DebugLogData::LEVEL_7 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_8, _( "Level 8" ),
                        wxBitmap( num_8_xpm ), wxNullBitmap,
                        _( "Debug Level 8 Pass" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_8, M_level & DebugLogData::LEVEL_8 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_9, _( "Level 9" ),
                        wxBitmap( num_9_xpm ), wxNullBitmap,
                        _( "Debug Level 9 Cross" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_9, M_level & DebugLogData::LEVEL_9 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_10, _( "Level 10" ),
                        wxBitmap( num_10_xpm ), wxNullBitmap,
                        _( "Debug Level 10 Shoot" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_10, M_level & DebugLogData::LEVEL_10 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_11, _( "Level 11" ),
                        wxBitmap( num_11_xpm ), wxNullBitmap,
                        _( "Debug Level 11 Clear" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_11, M_level & DebugLogData::LEVEL_11 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_12, _( "Level 12" ),
                        wxBitmap( num_12_xpm ), wxNullBitmap,
                        _( "Debug Level 12 Team" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_12, M_level & DebugLogData::LEVEL_12 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_13, _( "Level 13" ),
                        wxBitmap( num_13_xpm ), wxNullBitmap,
                        _( "Debug Level 13 Role" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_13, M_level & DebugLogData::LEVEL_13 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_14, _( "Level 14" ),
                        wxBitmap( num_14_xpm ), wxNullBitmap,
                        _( "Debug Level 14" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_14, M_level & DebugLogData::LEVEL_14 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_15, _( "Level 15" ),
                        wxBitmap( num_15_xpm ), wxNullBitmap,
                        _( "Debug Level 15" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_15, M_level & DebugLogData::LEVEL_15 );
    tbar->AddCheckTool( SWID_DEBUG_LEVEL_16, _( "Level 16" ),
                        wxBitmap( num_16_xpm ), wxNullBitmap,
                        _( "Debug Level 16" ) );
    tbar->ToggleTool( SWID_DEBUG_LEVEL_16, M_level & DebugLogData::LEVEL_16 );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::connectEvents()
{
    // ---------------------------------------------------
    // close frame
    Connect( SWID_FRAME_DEBUG_MESSAGE, wxEVT_CLOSE_WINDOW,
             wxCloseEventHandler( DebugMessageFrame::handleCloseEvent ) );

    // tab page change
    Connect( SWID_NOTEBOOK_DEBUG_MESSAGE, wxEVT_COMMAND_NOTEBOOK_PAGE_CHANGED,
             wxNotebookEventHandler( DebugMessageFrame::changeCurrentTab ) );

    // ---------------------------------------------------
    // tool menus
    // show/hide
    Connect(  SWID_CANVAS_DRAW_DEBUG_VIEW, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleShowDebugViewAll ) );
    Connect( SWID_CANVAS_DRAW_DEBUG_VIEW_SELF, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleShowDebugViewSelf ) );
    Connect( SWID_CANVAS_DRAW_DEBUG_VIEW_BALL, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleShowDebugViewBall ) );
    Connect( SWID_CANVAS_DRAW_DEBUG_VIEW_PLAYERS, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleShowDebugViewPlayers ) );
    Connect( SWID_CANVAS_DRAW_DEBUG_VIEW_COMMENT, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleShowDebugViewComment ) );
    Connect( SWID_CANVAS_DRAW_DEBUG_VIEW_FIGURE, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleShowDebugViewFigure ) );
    Connect( SWID_CANVAS_DRAW_DEBUG_VIEW_TARGET, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleShowDebugViewTarget ) );
    Connect( SWID_CANVAS_DRAW_DEBUG_VIEW_MESSAGE, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleShowDebugViewMessage ) );
    // dir selector
    Connect( SWID_SHOW_DEBUG_LOG_DIR_DIALOG, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::showDebugLogDirDialog ) );
    // cycle control
    Connect( SWID_DEBUG_LOG_SYNC, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::syncCycle ) );
    Connect( SWID_DEBUG_LOG_DECREMENT, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::decrementCycle ) );
    Connect( SWID_DEBUG_LOG_INCREMENT, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::incrementCycle ) );
    // debug level
    Connect( SWID_DEBUG_LEVEL_1, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_1 ) );
    Connect( SWID_DEBUG_LEVEL_2, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_2 ) );
    Connect( SWID_DEBUG_LEVEL_3, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_3 ) );
    Connect( SWID_DEBUG_LEVEL_4, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_4 ) );
    Connect( SWID_DEBUG_LEVEL_5, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_5 ) );
    Connect( SWID_DEBUG_LEVEL_6, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_6 ) );
    Connect( SWID_DEBUG_LEVEL_7, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_7 ) );
    Connect( SWID_DEBUG_LEVEL_8, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_8 ) );
    Connect( SWID_DEBUG_LEVEL_9, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_9 ) );
    Connect( SWID_DEBUG_LEVEL_10, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_10 ) );
    Connect( SWID_DEBUG_LEVEL_11, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_11 ) );
    Connect( SWID_DEBUG_LEVEL_12, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_12 ) );
    Connect( SWID_DEBUG_LEVEL_13, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_13 ) );
    Connect( SWID_DEBUG_LEVEL_14, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_14 ) );
    Connect( SWID_DEBUG_LEVEL_15, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_15 ) );
    Connect( SWID_DEBUG_LEVEL_16, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( DebugMessageFrame::toggleDebugLevel_16 ) );
}


/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::loadDebugLogDir( const rcsc::SideID side,
                                    const std::string & dir_path )
{
    MonitorViewConstPtr ptr = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! ptr )
    {
        std::cerr << "No valid data!" << std::endl;
        return;
    }

    if ( ptr->leftTeam().name().empty() )
    {
        std::cerr << "Empty team name!" << std::endl;
        return;
    }

    if ( side == rcsc::LEFT )
    {
        if ( M_debug_holder->setDir( ptr->leftTeam().name(),
                                     M_notebook->GetSelection() + 1,
                                     dir_path ) )
        {
            syncCycleOf( M_notebook->GetSelection() );
        }
    }
    else if ( side == rcsc::RIGHT )
    {
        if ( M_debug_holder->setDir( ptr->rightTeam().name(),
                                     M_notebook->GetSelection() + 1,
                                     dir_path ) )
        {
            syncCycleOf( M_notebook->GetSelection() );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::handleCloseEvent( wxCloseEvent & WXUNUSED( event ) )
{
    if ( this->GetParent() )
    {
        this->Hide();
    }
    else
    {
        this->Destroy();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::changeCurrentTab( wxNotebookEvent & event )
{
    /*
      std::cerr << "Notebook page changed. from = "
      << event.GetOldSelection()
      << " to " << event.GetSelection() << std::endl;
    */
    syncCycleOf( event.GetSelection() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleShowDebugViewAll( wxCommandEvent & event )
{
    if ( M_main_data.viewConfig().isShownDebugView() != event.IsChecked() )
    {
        M_main_data.getViewConfig().toggleShowDebugView();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleShowDebugViewSelf( wxCommandEvent & event )
{
    if ( M_main_data.viewConfig().isShownDebugViewSelf() != event.IsChecked() )
    {
        M_main_data.getViewConfig().toggleShowDebugViewSelf();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleShowDebugViewBall( wxCommandEvent & event )
{
    if ( M_main_data.viewConfig().isShownDebugViewBall() != event.IsChecked() )
    {
        M_main_data.getViewConfig().toggleShowDebugViewBall();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleShowDebugViewPlayers( wxCommandEvent & event )
{
    if ( M_main_data.viewConfig().isShownDebugViewPlayers() != event.IsChecked() )
    {
        M_main_data.getViewConfig().toggleShowDebugViewPlayers();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleShowDebugViewComment( wxCommandEvent & event )
{
    if ( M_main_data.viewConfig().isShownDebugViewComment() != event.IsChecked() )
    {
        M_main_data.getViewConfig().toggleShowDebugViewComment();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleShowDebugViewFigure( wxCommandEvent & event )
{
    if ( M_main_data.viewConfig().isShownDebugViewFigure() != event.IsChecked() )
    {
        M_main_data.getViewConfig().toggleShowDebugViewFigure();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleShowDebugViewTarget( wxCommandEvent & event )
{
    if ( M_main_data.viewConfig().isShownDebugViewTarget() != event.IsChecked() )
    {
        M_main_data.getViewConfig().toggleShowDebugViewTarget();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleShowDebugViewMessage( wxCommandEvent & event )
{
    if ( M_main_data.viewConfig().isShownDebugViewMessage() != event.IsChecked() )
    {
        M_main_data.getViewConfig().toggleShowDebugViewMessage();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::showDebugLogDirDialog( wxCommandEvent & WXUNUSED( event ) )
{
    DebugLogDirDialog dlg( this );

    if ( dlg.ShowModal() != wxID_OK )
    {
        //std::cerr << "DebugLogDirDialog is not ok." << std::endl;
        return;
    }

    rcsc::SideID side = dlg.getSide();
    std::string dir_path = dlg.getDirPath();

    std::cerr << "Select side = "
              << ( side == rcsc::LEFT ? "left" : "right" )
              << "  debug log dir = [" << dir_path << "]"
              << std::endl;
    AppConfig::instance().setDebugLogDir( dir_path );

    loadDebugLogDir( side, dir_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::syncCycle( wxCommandEvent & WXUNUSED( event ) )
{
    syncCycleOf( M_notebook->GetSelection() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::syncCycleOf( const int selection )
{
    if ( selection < 0
         || static_cast< int >( M_notebook->GetPageCount() ) - 1 < selection  )
    {
        std::cerr << "sync debug cycle. selection overflow." << selection
                  << std::endl;
        return;
    }

    MonitorViewConstPtr ptr = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! ptr )
    {
        std::cerr << "sync debug cycle. no view data." << std::endl;
        return;
    }

    std::cerr << "sync debug data cycle" << std::endl;
    // seek current cycle data
    if ( ! M_debug_holder->seekData( selection + 1, ptr->cycle() ) )
    {
        std::cerr << "No data! number = "
                  <<  selection + 1
                  << std::endl;
        return;
    }

    updateMessage( selection );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::decrementCycle( wxCommandEvent & WXUNUSED( event ) )
{
    if ( ! M_debug_holder->decDataCycle( M_notebook->GetSelection() + 1 ) )
    {
        return;
    }

    updateMessage();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::incrementCycle( wxCommandEvent & event )
{
    if ( ! M_debug_holder->incDataCycle( M_notebook->GetSelection() + 1 ) )
    {
        std::cerr << "Failed increment debug log" << std::endl;
        return;
    }

    updateMessage();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_1( wxCommandEvent & event )
{
    //M_level ^= DebugLogData::LEVEL_1; // reverse only 1 bit

    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_1;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_1;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_2( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_2;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_2;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_3( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_3;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_3;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_4( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_4;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_4;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_5( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_5;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_5;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_6( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_6;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_6;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_7( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_7;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_7;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_8( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_8;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_9;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_9( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_9;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_9;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_10( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_10;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_10;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_11( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_11;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_11;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_12( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_12;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_12;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_13( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_13;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_13;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_14( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_14;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_14;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_15( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_15;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_15;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::toggleDebugLevel_16( wxCommandEvent & event )
{
    if ( event.IsChecked() )
    {
        M_level |= DebugLogData::LEVEL_16;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_16;
    }

    updateMessage();
    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::clearAll()
{
    M_debug_holder = boost::shared_ptr< DebugLogHolder >( new DebugLogHolder );

    for ( int i = 0; i < 11; ++i )
    {
        M_message[ i ]->Clear();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::updateMessage()
{
    updateMessage( M_notebook->GetSelection() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageFrame::updateMessage( const int selection )
{
    if ( selection < 0
         || static_cast< int >( M_notebook->GetPageCount() ) - 1 < selection )
    {
        return;
    }

    const DebugLogHolder::DataList * data_list
        = M_debug_holder->getDataList( selection + 1 );

    if ( ! data_list )
    {
        return;
    }

    if ( M_level == 0 )
    {
        return;
    }

    const long cur_cycle = M_debug_holder->getCurrentCycle();

    wxTextCtrl * textctrl = M_message[ selection ];

    // clear current data
    textctrl->Clear();

    wxString main_buffer;
    main_buffer.Alloc( 1000 * 256 );
    int n_line = 0;

    long show_pos = 0;
    const DebugLogHolder::DataList::const_iterator data_end = data_list->end();
    for ( DebugLogHolder::DataList::const_iterator data_it = data_list->begin();
          data_it != data_end;
          ++data_it )
    {
        if ( ! (*data_it) )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Null data found!" << std::endl;
            continue;
        }

        if ( (*data_it)->cycle() <= cur_cycle )
        {
            show_pos = textctrl->GetLastPosition();
        }

        wxString cycle_line;
        cycle_line.Printf( wxT( "CYCLE %ld ------------------------------\n" ),
                           (*data_it)->cycle() );
        textctrl->AppendText( cycle_line );

        const DebugLogData::TextList::const_iterator text_end = (*data_it)->textList().end();
        for ( DebugLogData::TextList::const_iterator text_it = (*data_it)->textList().begin();
              text_it != text_end;
              ++text_it )
        {
            // level check
            //if ( lfirst <= text_it->first
            //&& text_it->first <= llast )
            if ( M_level & text_it->first )
            {
                main_buffer += wxString( text_it->second.c_str(),
                                         *wxConvCurrent );
                if ( ++n_line > 1000 )
                {
                    // append message to text control
                    //textctrl->AppendText( text_it->second );
                    textctrl->AppendText( main_buffer );
                    main_buffer.Empty();
                    n_line = 0;
                }
            }
        }
        if ( ! main_buffer.IsEmpty() )
        {
            textctrl->AppendText( main_buffer );
            main_buffer.Empty();
        }
    }

    textctrl->ShowPosition( show_pos );
}
