// -*-c++-*-

/*!
  \file debug_painter.cpp
  \brief debug info painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include "debug_painter.h"

#include "draw_config.h"
// model
#include "main_data.h"

#include <rcsc/common/server_param.h>

#include <iostream>

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::draw( wxDC & dc ) const
{
    int number = M_main_data.viewConfig().selectedNumber();
    if ( number == 0 )
    {
        return;
    }

    const rcsc::SideID player_side = ( number < 0 ? rcsc::RIGHT : rcsc::LEFT );

    const std::map< long, DebugViewCont > & view_map
        = ( player_side == rcsc::RIGHT
            ? M_main_data.viewHolder().rightDebugView()
            : M_main_data.viewHolder().leftDebugView() );

    MonitorViewConstPtr monitor_view
        = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! monitor_view )
    {
        return;
    }

    std::map< long, DebugViewCont >::const_iterator it
        = view_map.find( monitor_view->cycle() );

    if ( it == view_map.end() )
    {
        return;
    }

    number = std::abs( number );

    if ( static_cast< int >( it->second.size() ) < number )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error debug view container size is small."
                  << std::endl;
        return;
    }

    const DebugViewPtr view = it->second[number - 1];
    if ( ! view )
    {
        // no data!
        //std::cerr << __FILE__ << ":" << __LINE__
        //          << " Error. No debug view data of " << number
        //          << std::endl;
        return;
    }

    // draw self
    if ( M_main_data.viewConfig().isShownDebugViewSelf() )
    {
        drawSelf( dc, player_side, *view );
    }

    const DrawConfig & dconf = DrawConfig::instance();
    // draw players
    if ( M_main_data.viewConfig().isShownDebugViewPlayers() )
    {
        drawPlayers( dc,
                     player_side,
                     view->teammates(),
                     ( M_main_data.viewConfig().isShownDebugViewTarget()
                       ? view->targetTeammate()
                       : -1 ),
                     dconf.debugTeammateBrush() );
        drawPlayers( dc,
                     player_side,
                     view->opponents(),
                     -1,
                     dconf.debugOpponentBrush() );
        drawPlayers( dc,
                     player_side,
                     view->unknownTeammates(),
                     -1,
                     dconf.debugUnknownTeammateBrush() );
        drawPlayers( dc,
                     player_side,
                     view->unknownOpponents(),
                     -1,
                     dconf.debugUnknownOpponentBrush() );
        drawPlayers( dc,
                     player_side,
                     view->unknownPlayers(),
                     -1,
                     dconf.debugUnknownPlayerBrush() );
    }

    // draw ball
    if ( M_main_data.viewConfig().isShownDebugViewBall()
         && view->ball() )
    {
        drawBall( dc, player_side, *view );
    }

    // draw lines
    if ( M_main_data.viewConfig().isShownDebugViewFigure() )
    {
        drawLines( dc, player_side, *view );
        drawTriangles( dc, player_side, *view );
        drawRectangles( dc, player_side, *view );
        drawCircles( dc, player_side, *view );
    }

    if ( M_main_data.viewConfig().isShownDebugViewMessage()
         && ! view->message().empty() )
    {
        drawMessage( dc, view->message() );
    }
}

/*-------------------------------------------------------------------*/
/*!
  pos, vel, body, neck, comment
*/
void
DebugPainter::drawSelf( wxDC & dc,
                        const rcsc::SideID player_side,
                        const DebugViewData & view ) const
{
    const boost::shared_ptr< DebugViewData::SelfT > self = view.self();
    if ( ! self )
    {
        std::cerr << "DebugPainter::drawSelf. No self data!" << std::endl;
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    const int ix = vconf.getScreenX( (double)self->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    const int iy = vconf.getScreenY( (double)self->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    const int r = std::max( 2,
                            ( vconf.isEnlarged()
                              ? vconf.scale( 1.0 )
                              : vconf.scale( 0.3 ) ) );
    double body_start_dir = - ( (double)self->body_ + 90.0 );
    if ( player_side == rcsc::RIGHT ) body_start_dir -= 180.0;
    double body_end_dir = body_start_dir + 180.0;

    // draw targets
    if ( vconf.isShownDebugViewTarget()
         &&  view.targetPoint() )
    {
        int tx = vconf.getScreenX( view.targetPoint()->x * reverse );
        int ty = vconf.getScreenY( view.targetPoint()->y * reverse );

        dc.SetPen( dconf.debugTargetPen() );
        //dc.DrawLine( ix, iy, tx, ty );
        dc.DrawLine( tx - r, ty - r,
                     tx + r, ty + r );
        dc.DrawLine( tx - r, ty + r,
                     tx + r, ty - r );
    }

    // draw body half circle
    dc.SetPen( *wxTRANSPARENT_PEN );
    dc.SetBrush( dconf.debugSelfBrush() );
    dc.DrawEllipticArc( ix - r, iy - r,
                        r * 2, r * 2,
                        body_start_dir, body_end_dir );

    // draw edge
    dc.SetPen( dconf.debugPlayerPen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );
    dc.DrawCircle( ix, iy, r );

    // draw comment
    if ( vconf.isShownDebugViewComment()
         && ! self->comment_.empty() )
    {
        dc.SetFont( dconf.debugCommentFont() );
        dc.SetTextForeground( dconf.debugCommentFontColor() );
        dc.SetBackgroundMode( wxTRANSPARENT );

        wxString msg( self->comment_.c_str(),
                      *wxConvCurrent );
        dc.DrawText( msg,
                     ix - r, iy + r );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawBall( wxDC & dc,
                        const rcsc::SideID player_side,
                        const DebugViewData & view ) const
{
    const boost::shared_ptr< DebugViewData::BallT > & ball = view.ball();

    if ( ! ball )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    rcsc::Vector2D bpos( (double)ball->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse,
                         (double)ball->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    rcsc::Vector2D bvel( (double)ball->vx_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse,
                         (double)ball->vy_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

    const double ball_decay = rcsc::ServerParam::i().ballDecay();
    const int ix = vconf.getScreenX( bpos.x );
    const int iy = vconf.getScreenY( bpos.y );
    const int r = std::max( 2,
                            ( vconf.isEnlarged()
                              ? vconf.scale( 0.25 )
                              : vconf.scale( 0.07 ) ) );

    dc.SetPen( *wxTRANSPARENT_PEN );
    dc.SetBrush( dconf.debugBallBrush() );

    dc.DrawCircle( ix, iy, r );

    dc.SetPen( dconf.debugBallTracePen() );
    dc.SetBrush( dconf.debugBallTraceBrush() );

    int bx = ix;
    int by = iy;
    for ( int i = 0; i < 10; ++i )
    {
        bpos += bvel;
        bvel *= ball_decay;
        bx = vconf.getScreenX( bpos.x );
        by = vconf.getScreenY( bpos.y );
        dc.DrawCircle( bx, by, 2 );
    }
    dc.DrawLine( ix, iy, bx, by );

    // draw comment
    if ( vconf.isShownDebugViewComment()
         && ! ball->comment_.empty() )
    {
        dc.SetFont( dconf.debugCommentFont() );
        dc.SetTextForeground( dconf.debugCommentFontColor() );
        dc.SetBackgroundMode( wxTRANSPARENT );

        wxString msg( ball->comment_.c_str(),
                      *wxConvCurrent );
        dc.DrawText( msg,
                     ix - r, iy + r );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawPlayers( wxDC & dc,
                           const rcsc::SideID player_side,
                           const DebugViewData::PlayerCont & players,
                           const int target_number,
                           const wxBrush & body_brush ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    const bool comment = vconf.isShownDebugViewComment();
    const int r = std::max( 2,
                            ( vconf.isEnlarged()
                              ? vconf.scale( 1.0 )
                              : vconf.scale( 0.3 ) ) );

    const DebugViewData::PlayerCont::const_iterator end = players.end();
    for ( DebugViewData::PlayerCont::const_iterator it = players.begin();
          it != end;
          ++it )
    {
        const int ix = vconf.getScreenX( (double)(*it)->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
        const int iy = vconf.getScreenY( (double)(*it)->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

        if ( (*it)->body_ != -360 )
        {
            double body_start_dir = - ( (double)(*it)->body_ + 90.0 );
            if ( player_side == rcsc::RIGHT ) body_start_dir -= 180.0;
            double body_end_dir = body_start_dir + 180.0;
            // draw body half circle
            dc.SetPen( *wxTRANSPARENT_PEN );
            dc.SetBrush( body_brush );
            dc.DrawEllipticArc( ix - r, iy - r,
                                r * 2, r * 2,
                                body_start_dir, body_end_dir );
            // draw edge
            dc.SetPen( dconf.debugPlayerPen() );
            dc.SetBrush( *wxTRANSPARENT_BRUSH );
            dc.DrawCircle( ix, iy, r );
        }
        else
        {
            // draw simple circle
            dc.SetPen( *wxTRANSPARENT_PEN );
            dc.SetBrush( body_brush );
            dc.DrawCircle( ix, iy, r );
        }

        if ( (*it)->unum_ > 0 )
        {
            dc.SetFont( dconf.debugCommentFont() );
            dc.SetTextForeground( dconf.debugCommentFontColor() );
            dc.SetBackgroundMode( wxTRANSPARENT );
            wxString num;
            num.Printf( wxT( "%2d" ), (*it)->unum_ );
            dc.DrawText( num, ix + r, iy - r );
        }

        if ( target_number == (*it)->unum_ )
        {
            dc.SetPen( dconf.debugTargetPen() );
            dc.SetBrush( *wxTRANSPARENT_BRUSH );
            dc.DrawCircle( ix, iy, r + 2 );
        }

        if ( comment
             && ! (*it)->comment_.empty() )
        {
            dc.SetFont( dconf.debugCommentFont() );
            dc.SetTextForeground( dconf.debugCommentFontColor() );
            dc.SetBackgroundMode( wxTRANSPARENT );

            wxString msg( (*it)->comment_.c_str(),
                          *wxConvCurrent );
            dc.DrawText( msg, ix - r, iy + r );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawLines( wxDC & dc,
                         const rcsc::SideID player_side,
                         const DebugViewData & view ) const
{
    if ( view.lines().empty() )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    dc.SetPen( dconf.debugFigurePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    const std::list< DebugViewData::LineT >::const_iterator end = view.lines().end();
    for ( std::list< DebugViewData::LineT >::const_iterator it = view.lines().begin();
          it != end;
          ++it )
    {
        dc.DrawLine( vconf.getScreenX( it->x1_ * reverse ),
                     vconf.getScreenY( it->y1_ * reverse ),
                     vconf.getScreenX( it->x2_ * reverse ),
                     vconf.getScreenY( it->y2_ * reverse ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawTriangles( wxDC & dc,
                             const rcsc::SideID player_side,
                             const DebugViewData & view ) const
{
    if ( view.triangles().empty() )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    dc.SetPen( dconf.debugFigurePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    const std::list< DebugViewData::TriangleT >::const_iterator end = view.triangles().end();
    for ( std::list< DebugViewData::TriangleT >::const_iterator it = view.triangles().begin();
          it != end;
          ++it )
    {
        int x1 = vconf.getScreenX( it->x1_ * reverse );
        int y1 = vconf.getScreenY( it->y1_ * reverse );
        int x2 = vconf.getScreenX( it->x2_ * reverse );
        int y2 = vconf.getScreenY( it->y2_ * reverse );
        int x3 = vconf.getScreenX( it->x3_ * reverse );
        int y3 = vconf.getScreenY( it->y3_ * reverse );

        dc.DrawLine( x1, y1, x2, y2 );
        dc.DrawLine( x2, y2, x3, y3 );
        dc.DrawLine( x3, y3, x1, y1 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawRectangles( wxDC & dc,
                              const rcsc::SideID player_side,
                              const DebugViewData & view ) const
{
    if ( view.rectangles().empty() )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    dc.SetPen( dconf.debugFigurePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    const std::list< DebugViewData::RectT >::const_iterator end = view.rectangles().end();
    for ( std::list< DebugViewData::RectT >::const_iterator it = view.rectangles().begin();
          it != end;
          ++it )
    {
        int left_x = vconf.getScreenX( it->left_x_ * reverse );
        int top_y = vconf.getScreenY( it->top_y_ * reverse );
        int right_x = vconf.getScreenX( it->right_x_ * reverse );
        int bottom_y = vconf.getScreenY( it->bottom_y_ * reverse );

        dc.DrawLine( left_x, top_y, right_x, top_y );
        dc.DrawLine( right_x, top_y, right_x, bottom_y );
        dc.DrawLine( right_x, bottom_y, left_x, bottom_y );
        dc.DrawLine( left_x, bottom_y, left_x, top_y );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawCircles( wxDC & dc,
                           const rcsc::SideID player_side,
                           const DebugViewData & view ) const
{
    if ( view.circles().empty() )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    dc.SetPen( dconf.debugFigurePen() );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    const std::list< DebugViewData::CircleT >::const_iterator end = view.circles().end();
    for ( std::list< DebugViewData::CircleT >::const_iterator it = view.circles().begin();
          it != end;
          ++it )
    {
        dc.DrawCircle( vconf.getScreenX( it->center_x_ * reverse ),
                       vconf.getScreenY( it->center_y_ * reverse ),
                       vconf.scale( it->radius_ ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawMessage( wxDC & dc,
                           const std::string & message ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    dc.SetFont( dconf.debugMessageFont() );
    dc.SetTextForeground( dconf.debugMessageFontColor() );
    dc.SetBackgroundMode( wxTRANSPARENT );

    wxString msg( message.c_str(),
                  *wxConvCurrent );
    dc.DrawText( msg,
                 10,
                 vconf.scoreBoardHeight() + 2 );
}
