// -*-c++-*-

/*!
  \file field_canvas.cpp
  \brief main field canvas class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include "field_canvas.h"

#include "draw_config.h"

#include "ball_painter.h"
#include "ball_trace_painter.h"
#include "debug_painter.h"
#include "field_painter.h"
#include "offside_line_painter.h"
#include "player_control_painter.h"
#include "player_painter.h"
#include "player_trace_painter.h"
#include "score_board_painter.h"
#include "voronoi_diagram_painter.h"

// model
#include "id.h"
#include "main_data.h"
#include "view_holder.h"
#include "monitor_view_data.h"
#include "view_config.h"
#include "app_config.h"

#include <rcsc/geom/vector_2d.h>
#include <rcsc/common/server_param.h>

#include <iostream>
#include <cassert>
#include <cmath>

/*-------------------------------------------------------------------*/
/*!

*/
FieldCanvas::FieldCanvas( wxWindow * parent,
                          MainData & main_data )
    : wxWindow( parent, -1, wxDefaultPosition, wxDefaultSize,
                wxSIMPLE_BORDER /*| wxNO_FULL_REPAINT_ON_RESIZE*/ )
    , M_main_data( main_data )
    , M_normal_menu( new wxMenu )
    , M_system_menu( new wxMenu )
    , M_monitor_menu( new wxMenu )
{
    assert( parent );

    createWindows();
    connectEvents();

    wxSize cli_size = this->GetClientSize();
    M_canvas_bmp.Create( cli_size.GetWidth(), cli_size.GetHeight() );
}

/*-------------------------------------------------------------------*/
/*!
  destructor
*/
FieldCanvas::~FieldCanvas()
{
    //std::cerr << "delete FieldCanvas" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::createWindows()
{
    //----------------------------------------------------
    M_normal_menu->Append( SWID_SHOW_OPEN_RCG_DIALOG,
                           _( "Open\tctrl-o" ) );
    M_normal_menu->Append( SWID_MONITOR_CONNECT,
                           _( "Connect\tctrl-c" ) );
    M_normal_menu->Append( SWID_REQUEST_RESTART_SERVER,
                           _( "Start server" ) );
    //----------------------------------------------------
    M_system_menu->Append( SWID_SHOW_OPEN_RCG_DIALOG,
                           _( "Open\tctrl-o" ) );
    M_system_menu->Append( SWID_MONITOR_CONNECT,
                           _( "Connect\tctrl-c" ) );
    M_system_menu->AppendSeparator();
    M_system_menu->Append( SWID_REQUEST_KILL_SERVER,
                           _( "Kill server" ) );
    M_system_menu->Append( SWID_REQUEST_RESTART_SERVER,
                           _( "Restart server" ) );
    //----------------------------------------------------
    M_monitor_menu->Append( SWID_MONITOR_KICKOFF,
                            _( "Kick Off\tctrl-k" ) );
    M_monitor_menu->AppendSeparator();
    M_monitor_menu->Append( SWID_MENU_MONITOR_DROPBALL,
                            _( "Drop Ball" ) );
    M_monitor_menu->Append( SWID_MENU_MONITOR_FREEKICK_LEFT,
                            _( "Free Kick Left" ) );
    M_monitor_menu->Append( SWID_MENU_MONITOR_FREEKICK_RIGHT,
                            _( "Free Kick Right" ) );
    M_monitor_menu->AppendSeparator();
    M_monitor_menu->Append( SWID_MENU_MONITOR_DROPBALL_THERE,
                            _( "Drop Ball There" ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::connectEvents()
{
    //////////////////////////////////////////////////////////////
    // paint event
    Connect( wxID_ANY, wxEVT_PAINT,
             wxPaintEventHandler( FieldCanvas::handlePaint ) );
    // mouse event
    Connect( wxID_ANY, wxEVT_LEFT_DOWN,
             wxMouseEventHandler( FieldCanvas::handleLeftDown ) );
    Connect( wxID_ANY, wxEVT_LEFT_UP,
             wxMouseEventHandler( FieldCanvas::handleLeftUp ) );
    Connect( wxID_ANY, wxEVT_LEFT_DCLICK,
             wxMouseEventHandler( FieldCanvas::handleLeftDClick ) );
    Connect( wxID_ANY, wxEVT_MIDDLE_DOWN,
             wxMouseEventHandler( FieldCanvas::handleMiddleDown ) );
    Connect( wxID_ANY, wxEVT_MIDDLE_UP,
             wxMouseEventHandler( FieldCanvas::handleMiddleUp ) );
    Connect( wxID_ANY, wxEVT_RIGHT_DOWN,
             wxMouseEventHandler( FieldCanvas::handleRightDown ) );
    Connect( wxID_ANY, wxEVT_RIGHT_UP,
             wxMouseEventHandler( FieldCanvas::handleRightUp ) );
    Connect( wxID_ANY, wxEVT_MOTION,
             wxMouseEventHandler( FieldCanvas::handleMouseMotion ) );
    Connect( wxID_ANY, wxEVT_MOUSEWHEEL,
             wxMouseEventHandler( FieldCanvas::handleMouseWheel ) );

    // key event
    Connect( wxID_ANY, wxEVT_KEY_DOWN,
             wxKeyEventHandler( FieldCanvas::handleKeyDown ) );
    Connect( wxID_ANY, wxEVT_CHAR,
             wxCharEventHandler( FieldCanvas::handleChar ) );

    // menu event: normal
    Connect( SWID_SHOW_OPEN_RCG_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FieldCanvas::handleOpenRCG ) );
    Connect( SWID_MONITOR_CONNECT, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FieldCanvas::handleConnectMonitor ) );
    Connect( SWID_REQUEST_RESTART_SERVER, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FieldCanvas::handleRestartServer ) );
    // menu event: system
    Connect( SWID_REQUEST_KILL_SERVER, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FieldCanvas::handleKillServer ) );

    // menu event: monitor
    Connect( SWID_MENU_MONITOR_DROPBALL, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FieldCanvas::dropBall ) );
    Connect( SWID_MENU_MONITOR_FREEKICK_LEFT, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FieldCanvas::freeKickLeft ) );
    Connect( SWID_MENU_MONITOR_FREEKICK_RIGHT, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FieldCanvas::freeKickRight ) );
    Connect( SWID_MENU_MONITOR_DROPBALL_THERE, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FieldCanvas::dropBallThere ) );
}

/*-------------------------------------------------------------------*/
/*!
  event handler for EVT_PAINT.
*/
void
FieldCanvas::handlePaint( wxPaintEvent & WXUNUSED( event ) )
{
    wxPaintDC dc( this );
    draw( dc );
}

/*-------------------------------------------------------------------*/
/*!
  redraw all.
  called from external event handlers.
*/
void
FieldCanvas::draw()
{
    wxClientDC dc( this );
    draw( dc );
}

/*-------------------------------------------------------------------*/
/*!
  draw all data.
  create memory DC, draw the data to it and blit memoryDC to 'dc'.
*/
void
FieldCanvas::draw( wxDC & dc )
{
    // check canvas size
    //const wxSize cur_size = this->GetClientSize();
    const wxSize cur_size = this->GetSize();
    if ( M_canvas_bmp.GetWidth() != cur_size.GetWidth()
         || M_canvas_bmp.GetHeight() != cur_size.GetHeight() )
    {
        //std::cerr << "recreate field canbas bitmap memory" << std::endl;
        //M_measure_bmp.Create( cur_size.GetWidth(), cur_size.GetHeight() );
        //M_measure_bmp.SetMask( new wxMask( M_measure_bmp, *wxBLACK ) );
        M_canvas_bmp.Create( cur_size.GetWidth(), cur_size.GetHeight() );
    }

    M_main_data.update( cur_size.GetWidth(), cur_size.GetHeight() );

    if ( DrawConfig::instance().scoreBoardFont().GetPointSize()
         != M_main_data.viewConfig().scoreBoardFontSize() )
    {
        DrawConfig::instance()
            .resizeScoreBoardFont( M_main_data.viewConfig().scoreBoardFontSize() );
    }

    // create buffer DC
    wxMemoryDC mem_dc;
    mem_dc.SelectObject( M_canvas_bmp );

    drawAll( mem_dc );

    if ( M_mouse_state[2].isPressed()
         && M_mouse_state[2].isDragged() )
    {
        drawMouseMeasure( mem_dc );
    }

    // blit buffer DC to 'dc'
    dc.Blit( 0, 0,
             cur_size.GetWidth(), cur_size.GetHeight(),
             &mem_dc, 0, 0 );

    mem_dc.SelectObject( wxNullBitmap );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::drawAll( wxDC & mem_dc )
{
    // draw view data to buffer DC.
    FieldPainter( M_main_data.viewConfig() ).draw( mem_dc );

    if ( ! M_main_data.getViewData( M_main_data.viewIndex() ) )
    {
        return;
    }

    if ( M_main_data.viewConfig().isShownBallTrace() )
    {
        BallTracePainter( M_main_data ).draw( mem_dc );
    }

    if ( M_main_data.viewConfig().isShownPlayerTrace() )
    {
        PlayerTracePainter( M_main_data ).draw( mem_dc );
    }

    if ( M_main_data.viewConfig().isShownPlayers() )
    {
        PlayerPainter( M_main_data ).draw( mem_dc );
    }

    if ( M_main_data.viewConfig().isShownBall() )
    {
        BallPainter( M_main_data ).draw( mem_dc );
    }

    if ( M_main_data.viewConfig().isShownOffsideLine() )
    {
        OffsideLinePainter( M_main_data ).draw( mem_dc );
    }

    if ( M_main_data.viewConfig().isShownPlayers() )
    {
        PlayerControlPainter( M_main_data ).draw( mem_dc );
    }

    if ( M_main_data.viewConfig().isShownVoronoiDiagram()
         || M_main_data.viewConfig().isShownDelaunayTrianglation() )
    {
        VoronoiDiagramPainter( M_main_data ).draw( mem_dc );
    }

    if ( M_main_data.viewConfig().isShownDebugView() )
    {
        DebugPainter( M_main_data ).draw( mem_dc );
    }

    if ( M_main_data.viewConfig().isShownScoreBoard() )
    {
        ScoreBoardPainter( M_main_data ).draw( mem_dc );
    }
}

/*-------------------------------------------------------------------*/
/*!
  draw the line of mouse measure on 'mem_dc'.
*/
void
FieldCanvas::drawMouseMeasure( wxDC & mem_dc )
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    wxPoint start_point = M_mouse_state[2].pressedPoint();
    wxPoint end_point = M_mouse_state[2].draggedPoint();

    // draw straight line
    mem_dc.SetPen( dconf.measurePen() );
    mem_dc.SetBrush( *wxTRANSPARENT_BRUSH );
    mem_dc.DrawLine( start_point, end_point );

    mem_dc.SetPen( *wxRED_PEN );
    mem_dc.DrawCircle( start_point, 1 );
    mem_dc.DrawCircle( end_point, 1 );

    rcsc::Vector2D start_real( vconf.getFieldX( start_point.x ),
                               vconf.getFieldY( start_point.y ) );
    rcsc::Vector2D end_real( vconf.getFieldX( end_point.x ),
                             vconf.getFieldY( end_point.y ) );

    // ball travel marks
    {
        rcsc::Vector2D ball_pos = start_real;
        double ball_speed = rcsc::ServerParam::i().ballSpeedMax();
        rcsc::Vector2D ball_vel = end_real - start_real;
        ball_vel.setLength( ball_speed );

        const double max_length = start_real.dist( end_real );
        double total_travel = 0.0;

        ball_pos += ball_vel;
        ball_vel *= rcsc::ServerParam::i().ballDecay();
        total_travel += ball_vel.r();
        while ( total_travel < max_length )
        {
            mem_dc.DrawCircle( vconf.getScreenX( ball_pos.x ),
                               vconf.getScreenY( ball_pos.y ),
                               1 );
            ball_pos += ball_vel;
            ball_vel *= rcsc::ServerParam::i().ballDecay();
            ball_speed *= rcsc::ServerParam::i().ballDecay();
            total_travel += ball_speed;
            if ( ball_speed < 0.05 ) break;
        }
    }

    // draw distance & angle text
    mem_dc.SetFont( dconf.measureFont() );
    mem_dc.SetTextForeground( dconf.measureFontColor() );
    mem_dc.SetBackgroundMode( wxTRANSPARENT );

    wxString temp_str;
    temp_str.Printf( wxT( "(%.2f,%.2f)" ), start_real.x, start_real.y );
    mem_dc.DrawText( temp_str,
                     start_point.x,
                     start_point.y );

    if ( std::abs( start_point.x - end_point.x ) < 1
         && std::abs( start_point.y - end_point.y ) < 1 )
    {
        return;
    }

    temp_str.Printf( wxT( "(%.2f,%.2f)" ), end_real.x, end_real.y );
    // get the dimension of the string with current font
    wxCoord string_w, string_h;
    mem_dc.GetTextExtent( temp_str, &string_w, &string_h );
    mem_dc.DrawText( temp_str,
                     end_point.x,
                     end_point.y - string_h );

    mem_dc.SetTextForeground( wxColor( 224, 224, 192 ) );
    rcsc::Vector2D rel( end_real - start_real );
    temp_str.Printf( wxT( "rel(%.2f,%.2f) r%.2f th%.1f" ),
                     rel.x, rel.y, rel.r(), rel.th().degree() );

    mem_dc.DrawText( temp_str,
                     end_point.x,
                     end_point.y - string_h * 2 - 5 );

}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleLeftDown( wxMouseEvent & event )
{
    M_mouse_state[0].pressed( event.GetPosition() );

    if ( event.ControlDown() )
    {
        //std::cerr << "Ctrl down" << std::endl;
        emit_focusChanged( event.GetPosition() );
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleLeftUp( wxMouseEvent & event )
{
    M_mouse_state[0].released();

    if ( AppConfig::instance().monitorClientMode() )
    {
        PopupMenu( M_monitor_menu.get(), event.GetPosition() );
    }
}


/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleLeftDClick( wxMouseEvent & event )
{
    //std::cerr << "Ctrl down" << std::endl;
    emit_focusChanged( event.GetPosition() );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleMiddleDown( wxMouseEvent & event )
{
    M_mouse_state[1].pressed( event.GetPosition() );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleMiddleUp( wxMouseEvent & event )
{
    M_mouse_state[1].released();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleRightDown( wxMouseEvent & event )
{
    M_mouse_state[2].pressed( event.GetPosition() );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleRightUp( wxMouseEvent & event )
{
    wxPoint up_point = event.GetPosition();

    if ( M_mouse_state[2].isDragged() )
    {

    }
    else
    {
        if ( AppConfig::instance().monitorClientMode() )
        {
            PopupMenu( M_system_menu.get(), up_point );
        }
        else
        {
            PopupMenu( M_normal_menu.get(), up_point );
        }
    }

    M_mouse_state[2].released();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleMouseMotion( wxMouseEvent & event )
{
    wxPoint motion_point = event.GetPosition();

    for ( int i = 0; i < 3; ++i )
    {
        M_mouse_state[i].moved( motion_point );
    }

    if ( M_mouse_state[2].isDragged() )
    {
        // redraw measure
        //this->Refresh(); // invoke paint event
        draw();
    }

    emit_mouseMoved( motion_point );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleMouseWheel( wxMouseEvent & event )
{
    if ( event.GetWheelRotation() < 0 )
    {
        emit_wheelUp();
    }
    else
    {
        emit_wheelDown();
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleChar( wxKeyEvent & event )
{
#if 0
    std::cerr << "handleChar: key code="
              << event.GetKeyCode()
              << std::endl;
#endif

    if ( doHandleKeyCode( event ) )
    {
        return;
    }
    event.Skip();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleKeyDown( wxKeyEvent & event )
{
#if 0
    std::cerr << "handleKeyDown: key code="
              << event.GetKeyCode()
              << std::endl;
#endif
    if ( doHandleKeyCode( event ) )
    {
        return;
    }
    event.Skip();
}

/*-------------------------------------------------------------------*/
/*!
  internal key code handling method.
*/
bool
FieldCanvas::doHandleKeyCode( wxKeyEvent & event )
{
    //std::cerr << "handleKeyCod: key code="
    //          << event.GetKeyCode()
    //          << std::endl;

    return ( emit_keyPressed( event.GetKeyCode(),
                              event.ControlDown(),
                              event.AltDown(),
                              event.ShiftDown() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::handleOpenRCG( wxCommandEvent & WXUNUSED( event ) )
{
    std::cerr << "FieldCanvas handle openRCG" << std::endl;
    emit_openRCG();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::handleConnectMonitor( wxCommandEvent & WXUNUSED( event ) )
{
    std::cerr << "FieldCanvas handle connectMonitor" << std::endl;
    emit_connectMonitor();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::handleRestartServer( wxCommandEvent & WXUNUSED( event ) )
{
    std::cerr << "FieldCanvas handle restartServer" << std::endl;
    emit_restartServer();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::handleKillServer( wxCommandEvent & WXUNUSED( event ) )
{
    std::cerr << "FieldCanvas handle killServer" << std::endl;
    emit_killServer();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::dropBall( wxCommandEvent & WXUNUSED( event ) )
{
    emit_dropBall( M_mouse_state[0].pressedPoint() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::freeKickLeft( wxCommandEvent & WXUNUSED( event ) )
{
    emit_freeKickLeft( M_mouse_state[0].pressedPoint() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::freeKickRight( wxCommandEvent & WXUNUSED( event ) )
{
    emit_freeKickRight( M_mouse_state[0].pressedPoint() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::dropBallThere( wxCommandEvent & WXUNUSED( event ) )
{
    emit_dropBallThere();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::updateView()
{
    //this->Update(); // invoke paint event
    //this->Refresh(); // too heavy
    draw();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::setRedrawAllFlag()
{


}
