// -*-c++-*-

/*!
  \file monitor_client.cpp
  \brief Monitor Client class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <cassert>
#include <iostream>
#include <sstream>

#include <rcsc/net/udp_socket.h>
#include <rcsc/rcg/types.h>
#include <rcsc/monitor/monitor_command.h>

#include "id.h"
#include "app_config.h"
#include "main_frame.h"
#include "view_holder.h"

#include "monitor_client.h"

namespace {
const int POLL_INTERVAL_MS = 20;
}

/*-------------------------------------------------------------------*/
/*!
  constructor.
*/
MonitorClient::MonitorClient( const int version,
                              const char * hostname,
                              const int port,
                              MainFrame * main_frame,
                              ViewHolder & view_holder )

    : M_version( version )
    , M_socket( new rcsc::UDPSocket( hostname, port ) )
    , M_timer( this, SWID_TIMER_MONITOR )
    , M_main_frame( main_frame )
    , M_view_holder( view_holder )
{
    assert( main_frame );
    assert( M_socket );

    // check protocl versin range
    if ( M_version < 1 )
    {
        M_version = 1;
    }
    else if ( 2 < M_version )
    {
        M_version = 2;
    }

    if ( M_socket->fd() == -1 )
    {
        return;
    }

    Connect( SWID_TIMER_MONITOR, wxEVT_TIMER,
             wxTimerEventHandler( MonitorClient::handleTimer ) );

}

/*-------------------------------------------------------------------*/
/*!
  destructor.
*/
MonitorClient::~MonitorClient()
{
    disconnect();
    //std::cerr << "delete MonitorClient" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MonitorClient::disconnect()
{
    M_timer.Stop();
    if ( isConnected() )
    {
        sendDispBye();
        M_socket->close();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MonitorClient::isConnected() const
{
    return ( M_socket->fd() != -1 );
}

/*-------------------------------------------------------------------*/
/*!
  timer event handler.
*/
void
MonitorClient::handleTimer( wxTimerEvent & event )
{
    static int s_wait_counter = 0;
    s_wait_counter += 1;

    if ( M_version == 2 )
    {
        rcsc::rcg::dispinfo_t2 disp2;
        while ( M_socket->receive( reinterpret_cast< char * >( &disp2 ),
                                   sizeof( disp2 ) ) > 0 )
        {
            s_wait_counter = 0;
            M_view_holder.addDispInfo2( disp2 );
        }
    }
    else
    {
        rcsc::rcg::dispinfo_t disp1;
        while ( M_socket->receive( reinterpret_cast< char * >( &disp1 ),
                                   sizeof( disp1 ) ) > 0 )
        {
            //std::cerr << "receive dispinfo_t1" << std::endl;
            s_wait_counter = 0;
            M_view_holder.addDispInfo( disp1 );
        }
    }

    const long & wait_sec = AppConfig::instance().waitSeconds();
    if ( wait_sec > 0
         && s_wait_counter * POLL_INTERVAL_MS >= wait_sec * 1000 )
    {
        std::cerr << "MonitorClient waited "
                  << s_wait_counter * POLL_INTERVAL_MS
                  << " [ms].\n"
                  << "No response from rcssserver..delete the connection."
                  << std::endl;
        s_wait_counter = 0;
        M_timer.Stop();
        M_socket->close();

        if ( AppConfig::instance().autoQuitMode() )
        {
            ::wxExit();
        }
    }

    // received data
    if ( s_wait_counter == 0 )
    {
        M_main_frame->receiveMonitorPacket();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MonitorClient::sendCommand( const rcsc::MonitorCommand & com )
{
    if ( M_socket->fd() == -1 )
    {
        return;
    }

    std::ostringstream ostr;
    com.toStr( ostr );

    std::string com_str = ostr.str();
    M_socket->send( com_str.c_str(), com_str.length() + 1 );
    std::cerr << "send: " << com_str << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MonitorClient::sendDispInit()
{
    if ( M_socket->fd() == -1 )
    {
        return;
    }

    rcsc::MonitorInitCommand com( M_version );

    sendCommand( com );

    M_timer.Start( POLL_INTERVAL_MS );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MonitorClient::sendDispBye()
{
    if ( M_socket->fd() == -1 )
    {
        return;
    }

    rcsc::MonitorByeCommand com;

    sendCommand( com );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MonitorClient::sendKickOff()
{
    if ( M_socket->fd() == -1 )
    {
        return;
    }

    rcsc::MonitorKickOffCommand com;

    sendCommand( com );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MonitorClient::sendFreeKick( const double & x,
                             const double & y,
                             const rcsc::SideID side )
{
    if ( M_socket->fd() == -1 )
    {
        return;
    }

    rcsc::MonitorFreeKickCommand com( x, y, side );

    sendCommand( com );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MonitorClient::sendMove( const rcsc::SideID side,
                         const int unum,
                         const double & x, const double & y,
                         const double & angle )
{
    if ( M_socket->fd() == -1 )
    {
        return;
    }

    if ( side == rcsc::NEUTRAL )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " dispplayer invalie side" << std::endl;
        return;
    }

    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " dispplayer invalie unum " << unum << std::endl;
        return;
    }

    rcsc::MonitorMovePlayerCommand com( side, unum, x, y, angle );

    sendCommand( com );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MonitorClient::sendDiscard( const rcsc::SideID side,
                            const int unum )
{
    if ( M_socket->fd() == -1 )
    {
        return;
    }

    if ( side == rcsc::NEUTRAL )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " dispdiscard Invalie side" << std::endl;
        return;
    }

    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " dispdiscard invalie unum " << unum << std::endl;
        return;
    }

    rcsc::MonitorDiscardPlayerCommand com( side, unum );

    sendCommand( com );
}
