// -*-c++-*-

/*!
  \file plaeyr_painter.cpp
  \brief player painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include "player_painter.h"

#include "draw_config.h"
// model
#include "main_data.h"

#include <rcsc/common/player_type.h>
#include <rcsc/common/server_param.h>

#include <cmath>

namespace {

const double DEG2RAD = M_PI / 180.0;

}

/*-------------------------------------------------------------------*/
/*

*/
PlayerPainter::Param::Param( const Player & p,
                             const Ball & ball,
                             const ViewConfig & view_conf,
                             const rcsc::PlayerType & ptype )
    : x_( view_conf.getScreenX( p.x() ) )
    , y_( view_conf.getScreenY( p.y() ) )
    , body_radius_( view_conf.scale( ptype.playerSize() ) )
    , kick_radius_( view_conf.scale( ptype.kickableArea() ) )
    , have_full_effort_( p.hasFullEffort( ptype.effortMax() ) )
    , player_( p )
    , ball_( ball )
    , player_type_( ptype )
{

    if ( body_radius_ < 1 ) body_radius_ = 1;
    if ( kick_radius_ < 5 ) kick_radius_ = 5;

    draw_radius_ = ( view_conf.isEnlarged()
                     ? kick_radius_
                     : body_radius_ );
}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::draw( wxDC & dc ) const
{
    MonitorViewConstPtr view = M_main_data.getViewData( M_main_data.viewIndex() );

    if ( ! view )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();

    const Ball & ball = view->ball();

    const std::vector< Player >::const_iterator it_end = view->players().end();
    for ( std::vector< Player >::const_iterator it = view->players().begin();
          it != it_end;
          ++it )
    {
        const Param param( *it,
                           ball,
                           vconf,
                           M_main_data.viewHolder().playerType( it->type() ) );

        drawBody( dc, param );
        drawShadow( dc, param );
        drawEdge( dc, param );

        if ( vconf.isSelectedPlayer( it->side(), it->unum() )
             && vconf.playerFutureCycle() > 0
             && it->hasDelta() )
        {
            drawFuture( dc, param );
        }

        if ( it->hasView()
             && vconf.isShownViewCone() )
        {
            drawViewCone( dc, param );
        }

        if ( vconf.isShownControlArea() )
        {
            drawControlArea( dc, param );
        }

        drawText( dc, param );
    }

}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::drawBody( wxDC & dc,
                         const PlayerPainter::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // decide base color
    dc.SetPen( dconf.playerPen() );
    //dc.SetPen( *wxTRANSPARENT_PEN );

    //dc.SetBrush( dconf.playerBodyBrush( player.side(), player.isGoalie() ) );
    switch ( param.player_.side() ) {
    case rcsc::LEFT:
        if ( param.player_.isGoalie() )
        {
            dc.SetBrush( dconf.leftGoalieBrush() );
        }
        else
        {
            dc.SetBrush( dconf.leftTeamBrush() );
        }
        break;
    case rcsc::RIGHT:
        if ( param.player_.isGoalie() )
        {
            dc.SetBrush( dconf.rightGoalieBrush() );
        }
        else
        {
            dc.SetBrush( dconf.rightTeamBrush() );
        }
        break;
    case rcsc::NEUTRAL:
        dc.SetBrush( dconf.shadowBrush() );
        break;
    default:
        dc.SetBrush( dconf.shadowBrush() );
        break;
    }


    // decide status color
    if ( ! param.player_.isAlive() )
    {
        dc.SetBrush( *wxBLACK_BRUSH );
    }
    if ( param.player_.isKick() )
    {
        dc.SetPen( dconf.kickPen() );
    }
    if ( param.player_.isKickFault() )
    {
        dc.SetPen( dconf.kickFaultPen() );
        dc.SetBrush( dconf.kickFaultBrush() );
    }
    if ( param.player_.isCatch() )
    {
        dc.SetBrush( dconf.catchBrush() );
    }
    if ( param.player_.isCatchFault() )
    {
        dc.SetBrush( dconf.catchFaultBrush() );
    }
    if ( param.player_.isTackle() )
    {
        dc.SetPen( dconf.tacklePen() );
        dc.SetBrush( dconf.tackleBrush() );
    }
    if ( param.player_.isTackleFault() )
    {
        dc.SetPen( dconf.tacklePen() );
        dc.SetBrush( dconf.tackleFaultBrush() );
    }
    if ( param.player_.isCollidedBall() )
    {
        dc.SetBrush( dconf.collideBallBrush() );
    }
    if ( param.player_.isCollidedPlayer() )
    {
        dc.SetBrush( dconf.collidePlayerBrush() );
    }

    // draw to DC
    dc.DrawCircle( param.x_, param.y_,
                   param.draw_radius_ );
}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::drawShadow( wxDC & dc,
                           const PlayerPainter::Param & param ) const
{
    // set size
    int shadow_radius = static_cast< int >( param.draw_radius_ * 0.9 );
    // set angle
    double shadow_start_dir = -( param.player_.body() - 90.0 ); // reverse for EllipticArc
    double shadow_end_dir = shadow_start_dir + 180.0;

    // decide shadow color
#if 0
    double stamina_rate = ( param.player_.hasStamina()
                            ? ( param.player_.stamina()
                                / param.server_param_.staminaMax() )
                            : 1.0 );
    const wxColour & base_col = dconf.playerShadowBrush().GetColour();
    int r = 255 - static_cast< int >( (255 - base_col.Red()) * stamina_rate );
    int g = 255 - static_cast< int >( (255 - base_col.Green()) * stamina_rate );
    int b = 255 - static_cast< int >( (255 - base_col.Blue()) * stamina_rate );
    wxBrush tmp_brush( wxColour( r, g, b ), wxSOLID );
    dc.SetPen( *wxTRANSPARENT_PEN );
    dc.SetBrush( tmp_brush );
#else
    // create temporary brush
    int col = 255
        - static_cast< int >( 255
                              * ( param.player_.hasStamina()
                                  ? ( param.player_.stamina()
                                      / rcsc::ServerParam::i().staminaMax() )
                                  : 1.0 ) );
    wxBrush tmp_brush( wxColour( col, col, col ), wxSOLID );
    dc.SetPen( *wxTRANSPARENT_PEN );
    dc.SetBrush( tmp_brush );
#endif
    // draw half circle of shadow
    dc.DrawEllipticArc( param.x_ - shadow_radius, // left x
                        param.y_ - shadow_radius, // top y
                        shadow_radius * 2, // width
                        shadow_radius * 2, // height
                        shadow_start_dir, shadow_end_dir );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawEdge( wxDC & dc,
                         const PlayerPainter::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // draw body edge or kickable area edge
    if ( M_main_data.viewConfig().isEnlarged() )
    {
        // draw real body edge
        dc.SetPen( dconf.realBodyPen() );
        dc.SetBrush( *wxTRANSPARENT_BRUSH );
        dc.DrawCircle( param.x_, param.y_,
                       param.body_radius_ );
    }
    else
    {
        // draw kickable area edge
        if ( param.player_.side() == rcsc::LEFT )
        {
            dc.SetPen( dconf.leftTeamPen() );
        }
        else
        {
            dc.SetPen( dconf.rightTeamPen() );
        }
        dc.SetBrush( *wxTRANSPARENT_BRUSH );
        dc.DrawCircle( param.x_, param.y_,
                       param.kick_radius_ );
    }

    // draw stamina status if effort or recovery is decayed.
    if ( param.player_.hasStamina() )
    {
        if ( ! param.have_full_effort_ )
        {
            dc.SetPen( dconf.effortDecayedPen() );
            dc.SetBrush( *wxTRANSPARENT_BRUSH );
            dc.DrawCircle( param.x_, param.y_,
                           param.draw_radius_ + 2 );
        }
        else if ( ! param.player_.hasFullRecovery() )
        {
            dc.SetPen( dconf.recoveryDecayedPen() );
            dc.SetBrush( *wxTRANSPARENT_BRUSH );
            dc.DrawCircle( param.x_, param.y_,
                           param.draw_radius_ + 2 );
        }
    }

}


/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawFuture( wxDC & dc,
                           const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    // draw future state
    rcsc::Vector2D ppos( param.player_.x(), param.player_.y() );
    rcsc::Vector2D pvel( param.player_.deltaX(), param.player_.deltaY() );

    const int last = vconf.playerFutureCycle();

    wxPoint first_point( vconf.getScreenX( ppos.x ),
                         vconf.getScreenY( ppos.y ) ) ;
    wxPoint last_point = first_point;

    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    // draw kickable area edge
    if ( param.player_.side() == rcsc::LEFT )
    {
        dc.SetPen( DrawConfig::instance().rightTeamPen() );
    }
    else
    {
        dc.SetPen( DrawConfig::instance().leftTeamPen() );
    }

    for ( int i = 0; i < last; ++i )
    {
        ppos += pvel;
        pvel *= param.player_type_.playerDecay();

        wxPoint pt( vconf.getScreenX( ppos.x ),
                    vconf.getScreenY( ppos.y ) );
        if ( std::abs( last_point.x - pt.x ) < 1
             && std::abs( last_point.y - pt.y ) < 1 )
        {
            break;
        }

        dc.DrawCircle( pt, 2 );
        dc.DrawCircle( pt, param.kick_radius_ );

        last_point = pt;
    }

    // draw move line
    dc.SetPen( DrawConfig::instance().debugTargetPen() );
    dc.DrawLine( first_point, last_point );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawViewCone( wxDC & dc,
                             const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    const double head_dir = param.player_.head();
    // degree of view cone angle width
    const double view_width = param.player_.viewWidth();

    // screen pixel of visible distance
    const double visible_dist = rcsc::ServerParam::i().visibleDistance();
    const int visible_pixel = vconf.scale( visible_dist );
    // view cone start angle
    double view_start = head_dir - view_width * 0.5;
    // view cone end angle
    double view_end = view_start + view_width;
    // converted to radian
    view_start *= DEG2RAD;
    view_end *= DEG2RAD;

    // draw enlarged view cone and feel area
    if ( vconf.isSelectedPlayer( param.player_.side(), param.player_.unum() ) )
    {
        dc.SetPen( *wxWHITE_PEN );
        dc.SetBrush( *wxTRANSPARENT_BRUSH );

        // left side view cone end point x
        int lx = param.x_ + vconf.scale( 60.0 * std::cos( view_start ) );
        // left side view cone end point y
        int ly = param.y_ + vconf.scale( 60.0 * std::sin( view_start ) );
        // right side view cone end point x
        int rx = param.x_ + vconf.scale( 60.0 * std::cos( view_end ) );
        // right side view cone end point y
        int ry = param.y_ + vconf.scale( 60.0 * std::sin( view_end ) );

        // draw edge line
        dc.DrawLine( lx, ly, param.x_, param.y_ );
        dc.DrawLine( param.x_, param.y_, rx, ry );

        // draw arc
        view_start = (view_width * 0.5) - head_dir; // -(head - vw*0.5)
        view_end = view_start - view_width;
        const int UNUM_FAR = vconf.scale( 20.0 );
        const int TEAM_FAR = vconf.scale( 40.0 );
        const int TEAM_TOOFAR = vconf.scale( 60.0 );
        dc.DrawEllipticArc( param.x_ - UNUM_FAR, // left x
                            param.y_ - UNUM_FAR, // toop y
                            UNUM_FAR * 2, // width
                            UNUM_FAR * 2, // height
                            view_start, view_end );
        dc.DrawEllipticArc( param.x_ - TEAM_FAR, // left x
                            param.y_ - TEAM_FAR, // toop y
                            TEAM_FAR * 2, // width
                            TEAM_FAR * 2, // height
                            view_start, view_end );
        dc.DrawEllipticArc( param.x_ - TEAM_TOOFAR, // left x
                            param.y_ - TEAM_TOOFAR, // toop y
                            TEAM_TOOFAR * 2, // width
                            TEAM_TOOFAR * 2, // height
                            view_start, view_end );
        // draw feeling area circle
        dc.DrawCircle( param.x_, param.y_,
                       visible_pixel );
        return;
    }

    // draw normal view cone
    {
        dc.SetPen( DrawConfig::instance().viewConePen() );
        dc.SetBrush( *wxTRANSPARENT_BRUSH );
        // left side view cone end point x
        int lx = param.x_
            + vconf.scale( visible_dist * std::cos( view_start ) );
        // left side view cone end point y
        int ly = param.y_
            + vconf.scale( visible_dist  * std::sin( view_start ) );
        // right side view cone end point x
        int rx = param.x_
            + vconf.scale( visible_dist * std::cos( view_end ) );
        // right side view cone end point y
        int ry = param.y_
            + vconf.scale( visible_dist * std::sin( view_end ) );

        dc.DrawArc( rx, ry,
                    lx, ly,
                    param.x_, param.y_ );

    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawControlArea( wxDC & dc,
                                const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    if ( param.player_.isGoalie() )
    {
        int catchable = vconf.scale
            ( std::sqrt( std::pow( rcsc::ServerParam::i().catchAreaLength(), 2.0 )
                         + std::pow( rcsc::ServerParam::i().catchAreaWidth() * 0.5, 2.0 ) ) );
        dc.SetPen( ( param.player_.side() == rcsc::LEFT )
                   ? dconf.leftGoaliePen()
                   : dconf.rightGoaliePen() );
        dc.SetBrush( *wxTRANSPARENT_BRUSH );
        dc.DrawCircle( param.x_, param.y_, catchable );
    }

    rcsc::Vector2D ppos( param.player_.x(), param.player_.y() );
    rcsc::Vector2D bpos( param.ball_.x(), param.ball_.y() );
    rcsc::Vector2D player_to_ball = bpos - ppos;
    player_to_ball.rotate( - param.player_.body() );

    // draw tackle area & probability
    double tackle_dist = ( player_to_ball.x > 0.0
                           ? rcsc::ServerParam::i().tackleDist()
                           : rcsc::ServerParam::i().tackleBackDist() );
    double tackle_prob = ( std::pow( player_to_ball.absX() / tackle_dist,
                                     rcsc::ServerParam::i().tackleExponent() )
                           + std::pow( player_to_ball.absY() / rcsc::ServerParam::i().tackleWidth(),
                                       rcsc::ServerParam::i().tackleExponent() ) );
    if ( tackle_prob < 1.0 )
    {
        rcsc::AngleDeg body_ang = param.player_.body();
        rcsc::AngleDeg body_ang_side = body_ang + 90.0;
        double body_x = body_ang.cos();
        double body_y = body_ang.sin();
        double forward_x = rcsc::ServerParam::i().tackleDist() * body_x;
        double forward_y = rcsc::ServerParam::i().tackleDist() * body_y;
        double back_x = rcsc::ServerParam::i().tackleBackDist() * -body_x;
        double back_y = rcsc::ServerParam::i().tackleBackDist() * -body_y;
        double right_x = rcsc::ServerParam::i().tackleWidth() * body_ang_side.cos();
        double right_y = rcsc::ServerParam::i().tackleWidth() * body_ang_side.sin();

        wxPoint pts[4];
        pts[0].x = vconf.getScreenX( ppos.x + forward_x + right_x );
        pts[0].y = vconf.getScreenY( ppos.y + forward_y + right_y );
        pts[1].x = vconf.getScreenX( ppos.x + forward_x - right_x );
        pts[1].y = vconf.getScreenY( ppos.y + forward_y - right_y );
        pts[2].x = vconf.getScreenX( ppos.x + back_x - right_x );
        pts[2].y = vconf.getScreenY( ppos.y + back_y - right_y );
        pts[3].x = vconf.getScreenX( ppos.x + back_x + right_x );
        pts[3].y = vconf.getScreenY( ppos.y + back_y + right_y );

        //dc.SetPen( *wxMEDIUM_GREY_PEN );
        dc.SetPen( dconf.tackleAreaPen() );
        dc.SetBrush( *wxTRANSPARENT_BRUSH );

        dc.DrawPolygon( 4, pts );

        dc.SetTextForeground( dconf.tacklePen().GetColour() );
        dc.SetBackgroundMode( wxTRANSPARENT );
        wxString msg;
        msg.Printf( wxT( "TakleProb=%.3f" ), 1.0 - tackle_prob );

        int text_radius = param.draw_radius_;
        if ( text_radius > 40 )
        {
            text_radius = 40;
        }
        dc.DrawText( msg,
                     param.x_ + text_radius,
                     param.y_ - text_radius + dc.GetCharHeight() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawText( wxDC & dc,
                         const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const int text_radius = std::min( 40, param.draw_radius_ );

    dc.SetFont( dconf.playerFont() );
    dc.SetBackgroundMode( wxTRANSPARENT );

    wxString player_str;

    if ( vconf.isShownPlayerNumber()
         && vconf.isShownHeteroNumber() )
    {
        player_str.Printf( wxT( "%2d,t%d" ),
                           param.player_.unum(), param.player_.type() );
    }
    else if ( vconf.isShownPlayerNumber() )
    {
        player_str.Printf( wxT( "%2d" ), param.player_.unum() );
    }
    else if ( vconf.isShownHeteroNumber() )
    {
        player_str.Printf( wxT( "t%d" ), param.player_.type() );
    }

    if ( ! player_str.IsEmpty() )
    {
        dc.SetTextForeground( dconf.playerNumberFontColor() );
        dc.DrawText( player_str,
                     param.x_ + text_radius,
                     param.y_ - text_radius );
    }

    if ( param.player_.hasStamina()
         && vconf.isShownStamina() )
    {
        // this player is selected
        if ( vconf.isSelectedPlayer( param.player_.side(),
                                         param.player_.unum() ) )
        {
            player_str.Printf( wxT( "%4.0f,e%.3f,r%.3f" ),
                               param.player_.stamina(),
                               param.player_.effort(),
                               param.player_.recovery() );
            dc.SetTextForeground( dconf.playerNumberFontColor() );
        }
        else
        {
            player_str.Printf( wxT( "%4.0f" ), param.player_.stamina() );
#if 0
            if ( ! param.have_full_effort_ )
            {
                dc.SetTextForeground( dconf.getEffortDecayedPen().GetColour() );
            }
            else if ( ! param.player_.hasFullRecovery() )
            {
                dc.SetTextForeground( dconf.recoveryDecayedPen().GetColour() );
            }
            else
#endif
            {
                dc.SetTextForeground( dconf.playerStaminaFontColor() );
            }
        }
        dc.DrawText( player_str,
                     param.x_ - text_radius,
                     param.y_ - text_radius - dc.GetCharHeight() );
    }
}
