// -*-c++-*-

/*!
  \file audio_memory.cpp
  \brief communication message data holder Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "audio_memory.h"

#include "logger.h"

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

*/
AudioMemory::AudioMemory()
    : M_pass_receiver_number( 0 )
    , M_pass_receive_pos( 0.0, 0.0 )
    , M_pass_time( -1, 0 )
    , M_our_intercept_number( Unum_Unknown )
    , M_our_intercept_cycle( 1000 )
    , M_our_intercept_time( -1, 0 )
    , M_opp_intercept_number( Unum_Unknown )
    , M_opp_intercept_cycle( 1000 )
    , M_opp_intercept_time( -1, 0 )
    , M_goalie_pos( 50.0, 0.0 )
    , M_goalie_time( -1, 0 )
    , M_offside_line_x( 100.0 )
    , M_offside_line_time( -100, 0 )
    , M_defense_line_x( 100.0 )
    , M_defense_line_time( -100, 0 )
    , M_wait_request_time( -100, 0 )
    , M_pass_request_sender_number( Unum_Unknown )
    , M_pass_request_pos( 50.0, 0.0 )
    , M_pass_request_time( -100, 0 )
    , M_dribble_sender_number( Unum_Unknown )
    , M_dribble_target_pos( 50.0, 0.0 )
    , M_dribble_queue_count( 0 )
    , M_dribble_target_time( -100, 0 )
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setBall( const Vector2D & pos,
                      const Vector2D & vel,
                      const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard ball: pos=(%.3f, %.3f) vel=(%.2f, %.2f)",
                  pos.x, pos.y, vel.x, vel.y );

    M_ball_pos = pos;
    M_ball_vel = vel;
    M_ball_time = current;
}


/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setPass( const int receiver,
                      const Vector2D & pos,
                      const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard pass: receiver=%d, pos=(%.2f, %.2f)",
                  receiver, pos.x, pos.y );

    M_pass_receiver_number = receiver;
    M_pass_receive_pos = pos;
    M_pass_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setIntercept( const bool our,
                           const int interceptor_number,
                           const int cycle,
                           const GameTime & current )
{
    if ( our )
    {
        dlog.addText( Logger::WORLD,
                      __FILE__": set heard teammate intercept: unum=%d cycle=%d",
                      interceptor_number, cycle );

        M_our_intercept_number = interceptor_number;
        // -1 because the heard value was estimated in the previous cycle
        M_our_intercept_cycle = std::max( 0, cycle - 1 );
        M_our_intercept_time = current;
    }
    else
    {
        dlog.addText( Logger::WORLD,
                      __FILE__": set heard opponent intercept: unum=%d cycle=%d",
                      interceptor_number, cycle );

        M_opp_intercept_number = interceptor_number;
        // -1 because the heard value was estimated in the previous cycle
        M_opp_intercept_cycle = std::max( 0, cycle - 1 );
        M_opp_intercept_time = current;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setOpponentGoalie( const Vector2D & pos,
                                const AngleDeg & body,
                                const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard goalie: pos=(%.2f, %.2f) body=%.1f",
                  pos.x, pos.y, body.degree() );

    M_goalie_pos = pos;
    M_goalie_body = body;
    M_goalie_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setPlayer( const int unum,
                        const Vector2D & pos,
                        const AngleDeg & body,
                        const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard player. unum=%d pos=(%.2f, %.2f) body=%.1f",
                  unum,
                  pos.x, pos.y, body.degree() );

    M_player_unum = unum;
    M_player_pos = pos;
    M_player_body = body;
    M_player_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setOffsideLine( const double & offside_line_x,
                             const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard offside line = %.1f",
                  offside_line_x );

    M_offside_line_x = offside_line_x;
    M_offside_line_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setDefenseLine( const double & defense_line_x,
                             const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard defense line = %.1f",
                  defense_line_x );

    M_defense_line_x = defense_line_x;
    M_defense_line_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setWaitRequest( const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard wait request" );

    M_wait_request_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setPassRequest( const int sender_number,
                             const Vector2D & request_pos,
                             const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard pass request. unum=%d pos=(%.1f %.1f)",
                  sender_number, request_pos.x, request_pos.y );

    M_pass_request_sender_number = sender_number;
    M_pass_request_pos = request_pos;
    M_pass_request_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setStamina( const int sender_number,
                         const double & rate,
                         const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard stamina. unum=%d rate=%.3f",
                  sender_number, rate );

    M_stamina_sender_number = sender_number;
    M_stamina_rate = rate;
    M_stamina_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setRecovery( const int sender_number,
                          const double & rate,
                          const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard recovery. unum=%d rate=%.3f",
                  sender_number, rate );

    M_recovery_sender_number = sender_number;
    M_recovery_rate = rate;
    M_recovery_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioMemory::setDribbleTarget( const int sender_number,
                               const Vector2D & pos,
                               const int queue_count,
                               const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  __FILE__": set heard dribble target. unum=%d"
                  " target=(%.1f %.1f) count=%d",
                  sender_number,
                  pos.x, pos.y,
                  queue_count );

    M_dribble_sender_number = sender_number;
    M_dribble_target_pos = pos;
    M_dribble_queue_count = queue_count;
    M_dribble_target_time = current;
}

}
