// -*-c++-*-

/*!
  \file audio_memory.h
  \brief communication message data holder Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_AUDIO_MEMORY_H
#define RCSC_PLAYER_AUDIO_MEMORY_H

#include <rcsc/geom/vector_2d.h>
#include <rcsc/game_time.h>
#include <rcsc/types.h>

namespace rcsc {

/*!
  \class AudioMemory
  \brief the memory of heard data
*/
class AudioMemory {
protected:
    // ball info
    Vector2D M_ball_pos; //!< heard ball position
    Vector2D M_ball_vel; //!< heard ball velocity
    GameTime M_ball_time; //!< ball info heard time

    // pass info
    int M_pass_receiver_number; //!< heard pass receiver number
    Vector2D M_pass_receive_pos; //!< heard pass receive pos
    GameTime M_pass_time; //!< pass info heard time

    // intercept info
    int M_our_intercept_number; //!< heard interceptor number
    int M_our_intercept_cycle; //!< heard intercept cycle
    GameTime M_our_intercept_time; //!< our intercept info heard time

    int M_opp_intercept_number; //!< heard interceptor number
    int M_opp_intercept_cycle; //!< heard intercept cycle
    GameTime M_opp_intercept_time; //!< opp intercept info heard time

    // opponent goalie info
    Vector2D M_goalie_pos; //!< heard goalie positon
    AngleDeg M_goalie_body; //!< heard goalie's body angle
    GameTime M_goalie_time; //!< goalie info heard time

    // player info
    int M_player_unum; //!< heard player unum
    Vector2D M_player_pos; //!< heard player position
    AngleDeg M_player_body; //!< heard player's body angle
    GameTime M_player_time; //!< player info heard time


    // offside line
    double M_offside_line_x; //!< heard offside line value
    GameTime M_offside_line_time; //!< offside line info heard time

    // defense line
    double M_defense_line_x; //!< heard defense line value
    GameTime M_defense_line_time; //!< defense line info heard time

    // wait request
    GameTime M_wait_request_time; //!< wait request info heard time

    // pass request
    int M_pass_request_sender_number; //!< pass request sendeer number
    Vector2D M_pass_request_pos; //!< heard pass request position
    GameTime M_pass_request_time; //!< game time when attack request line info is received

    // stamina
    int M_stamina_sender_number; //!< stamina sender number
    double M_stamina_rate; //!< heard stamina rate
    GameTime M_stamina_time; //!< stamina info heard time

    // recovery
    int M_recovery_sender_number; //!< recovery sender number
    double M_recovery_rate; //!< heard recovery rate
    GameTime M_recovery_time; //!< recovery info heard time

    // dribble info
    int M_dribble_sender_number; //!< heard dribble info sendeer number
    Vector2D M_dribble_target_pos; //!< heard dribble target point
    int M_dribble_queue_count; //!< the size of dribble action queue
    GameTime M_dribble_target_time; //!< dribble info heard time

private:
    // not used
    AudioMemory( const AudioMemory & );
    AudioMemory & operator=( const AudioMemory & );

public:

    /*!
      \brief initialize member variables
    */
    AudioMemory();

    /*!
      \brief virtual destructor
    */
    virtual
    ~AudioMemory()
      { }


    // accessor methods

    /*!
      \brief get heard ball pos
      \return ball pos
    */
    const
    Vector2D & ballPos() const
      {
          return M_ball_pos;
      }

    /*!
      \brief get heard ball vel
      \return heard ball pos
    */
    const
    Vector2D & ballVel() const
      {
          return M_ball_vel;
      }

    /*!
      \brief ball info heard time
      \return time value
    */
    const
    GameTime & ballTime() const
      {
          return M_ball_time;
      }

    /*!
      \brief get heard pass receiver number
      \return heard pass receiver number
    */
    int passReceiverNumber() const
      {
          return M_pass_receiver_number;
      }

    /*!
      \brief get heard pass receive pos
      \return heard pass receive pos
    */
    const
    Vector2D & passReceivePos() const
      {
          return M_pass_receive_pos;
      }

    /*!
      \brief get pas info heard time
      \return time value
    */
    const
    GameTime & passTime() const
      {
          return M_pass_time;
      }

    /*!
      \brief get heard our interceptor number
      \return heard our interceptor number
    */
    int ourInterceptNumber() const
      {
          return M_our_intercept_number;
      }

    /*!
      \brief get heard our intercept cycle
      \return heard our intercept cycle
    */
    int ourInterceptCycle() const
      {
          return M_our_intercept_cycle;
      }

    /*!
      \brief get our intercept info heard time
      \return time value
    */
    const
    GameTime & ourInterceptTime() const
      {
          return M_our_intercept_time;
      }

    /*!
      \brief get opp interceptor number
      \return opp interceptor number
    */
    int oppInterceptNumber() const
      {
          return M_opp_intercept_number;
      }

    /*!
      \brief get opp intercept cycle
      \return opp intercept cycle
    */
    int oppInterceptCycle() const
      {
          return M_opp_intercept_cycle;
      }

    /*!
      \brief get opp intercept info heard time
      \return time value
    */
    const
    GameTime & oppInterceptTime() const
      {
          return M_opp_intercept_time;
      }

    /*!
      \brief get heard goalie pos
      \return heard goalie pos
    */
    const
    Vector2D & goaliePos() const
      {
          return M_goalie_pos;
      }

    /*!
      \brief get heard goalie body angle
      \return heard goalie body angle
    */
    const
    AngleDeg & goalieBody() const
      {
          return M_goalie_body;
      }

    /*!
      \brief get goalie info heard time
      \return time value
    */
    const
    GameTime & goalieTime() const
      {
          return M_goalie_time;
      }

    /*!
      \brief get heard player unum
      \return heard player unum
    */
    int playerUnum() const
      {
          return M_player_unum;
      }

    /*!
      \brief get heard player pos
      \return heard player pos
    */
    const
    Vector2D & playerPos() const
      {
          return M_player_pos;
      }

    /*!
      \brief get heard player body angle
      \return heard player body angle
    */
    const
    AngleDeg & playerBody() const
      {
          return M_player_body;
      }

    /*!
      \brief get player info heard time
      \return time value
    */
    const
    GameTime & playerTime() const
      {
          return M_player_time;
      }

    /*!
      \brief get heard offside line
      \return heard offside line
    */
    const
    double & offsideLineX() const
      {
          return M_offside_line_x;
      }

    /*!
      \brief get offside line heard time
      \return time value
    */
    const
    GameTime & offsideLineTime() const
      {
          return M_offside_line_time;
      }

    /*!
      \brief get heard defense line
      \return heard defense line
    */
    const
    double & defenseLineX() const
      {
          return M_defense_line_x;
      }

    /*!
      \brief get defense line heard time
      \return time value
    */
    const
    GameTime & defenseLineTime() const
      {
          return M_defense_line_time;
      }

    /*!
      \brief get wait request heard time
      \return time value
    */
    const
    GameTime & waitRequestTime() const
      {
          return M_wait_request_time;
      }

    /*!
      \brief get pass request sender number
      \return pass request sender number
    */
    int passRequestSenderNumber() const
      {
          return M_pass_request_sender_number;
      }

    /*!
      \brief get pass request point
      \return pass request point
    */
    const
    Vector2D & passRequestPos() const
      {
          return M_pass_request_pos;
      }

    /*!
      \brief get pass request heard time
      \return time value
    */
    const
    GameTime & passRequestTime() const
      {
          return M_pass_request_time;
      }

    /*!
      \brief get heard stamina sender number
      \return stamina sender number
    */
    int staminaSenderNumber() const
      {
          return M_stamina_sender_number;
      }

    /*!
      \brief get heard stamina rate
      \return heard stamina rate
    */
    const
    double & staminaRate() const
      {
          return M_stamina_rate;
      }

    /*!
      \brief get stamina info heard time
      \return time value
    */
    const
    GameTime & staminaTime() const
      {
          return M_stamina_time;
      }

    /*!
      \brief get heard recovery sender number
      \return recovery sender number
    */
    int recoverySenderNumber() const
      {
          return M_recovery_sender_number;
      }

    /*!
      \brief get heard recovery rate
      \return heard recovery rate
    */
    const
    double & recoveryRate() const
      {
          return M_recovery_rate;
      }

    /*!
      \brief get recovery info heard time
      \return time value
    */
    const
    GameTime & recoveryTime() const
      {
          return M_recovery_time;
      }

    /*!
      \brief get dribble info sender number
      \return dribble info sender number
    */
    int dribbleSenderNumber() const
      {
          return M_dribble_sender_number;
      }

    /*!
      \brief get heard dribble target point
      \return heard dribble target point
    */
    const
    Vector2D & dribbleTartgetPos() const
      {
          return M_dribble_target_pos;
      }

    /*!
      \brief get heard dribble queue count
      \return heard dribble queue count
     */
    int dribbleQueueCount() const
      {
          return M_dribble_queue_count;
      }

    /*!
      \brief get dribble info heard time
      \return time value
    */
    const
    GameTime & dribbleTargetTime() const
      {
          return M_dribble_target_time;
      }


    // setter methods

    /*!
      \brief set heard ball status
      \param pos heard ball posision
      \param vel heard ball velocity
      \param current current game time
    */
    void setBall( const Vector2D & pos,
                  const Vector2D & vel,
                  const GameTime & current );

    /*!
      \brief set heard pass info
      \param receiver receiver number
      \param pos pass receive point
      \param current current game time
    */
    void setPass( const int receiver,
                  const Vector2D & pos,
                  const GameTime & current );

    /*!
      \brief set heard pass request info
      \param our true if teammate info
      \param interceptor_number interceptor's uniform number.
      \param cycle estimated intercept cycle
    */
    void setIntercept( const bool our,
                       const int interceptor_number,
                       const int cycle,
                       const GameTime & current );

    /*!
      \brief set heard opponent goalie info
      \param pos heard opponent goalie pos
      \param body heard opponent goalie's body angle
      \param current current game time
    */
    void setOpponentGoalie( const Vector2D & pos,
                            const AngleDeg & body,
                            const GameTime & current );

    /*!
      \brief set heard player info
      \param unum heard player unum
      \param pos heard player pos
      \param body heard player body angle
      \param current current game time
    */
    void setPlayer( const int unum,
                    const Vector2D & pos,
                    const AngleDeg & body,
                    const GameTime & current );

    /*!
      \brief set heard offside line info
      \param offside_line_x heard value
      \param current current game time
    */
    void setOffsideLine( const double & offside_line_x,
                         const GameTime & current );

    /*!
      \brief set heard defense line info
      \param defense_line_x heard value
      \param current current game time
    */
    void setDefenseLine( const double & defense_line_x,
                         const GameTime & current );

    /*!
      \brief set heard wait request info
    */
    void setWaitRequest( const GameTime & current );

    /*!
      \brief set heard pass request info
      \param sender_number message sender's uniform number
      \param request_pos request position
      \param current current game time
    */
    void setPassRequest( const int sender_number,
                         const Vector2D & request_pos,
                         const GameTime & current );

    /*!
      \brief set heard stamina info
      \param sender_number message sender's uniform number
      \param rate stamina value rate
      \param current current game time
    */
    void setStamina( const int sender_number,
                     const double & rate,
                     const GameTime & current );

    /*!
      \brief set heard recovery info
      \param sender_number message sender's uniform number
      \param rate recovery value rate
      \param current current game time
    */
    void setRecovery( const int sender_number,
                      const double & rate,
                      const GameTime & current );

    /*!
      \brief set heard dribble target point
      \param sender_number message sender's uniform number
      \param pos target position
      \param queue_count dribble queue count
      \param current current game time
    */
    void setDribbleTarget( const int sender_number,
                           const Vector2D & pos,
                           const int queue_count,
                           const GameTime & current );
};

}

#endif
