// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_defender_basic_block_move.h"

#include "bhv_basic_move.h"
#include "bhv_defender_get_ball.h"
#include "bhv_danger_area_tackle.h"

#include "body_intercept2008.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_DefenderBasicBlockMove::execute( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        __FILE__": Bhv_DefenderBasicBlock" );
    //////////////////////////////////////////////////////////////////
    // tackle
    if ( Bhv_DangerAreaTackle().execute( agent ) )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": tackle" );
        return true;
    }

    //////////////////////////////////////////////////////////////////
    // get ball
    if ( Bhv_DefenderGetBall( M_home_pos ).execute( agent ) )
    {
        return true;
    }

    //////////////////////////////////////////////////////////////////
    // positioning

    const double ball_xdiff
        = agent->world().ball().pos().x - agent->world().self().pos().x;

    if ( ball_xdiff > 10.0
         && agent->world().interceptTable()->isOurTeamBallPossessor() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": ball is front and our team keep ball" );
        Bhv_BasicMove( M_home_pos ).execute( agent );
        return true;
    }

    double dash_power = get_dash_power( agent, M_home_pos );

    double dist_thr = agent->world().ball().distFromSelf() * 0.1;
    if ( dist_thr < 1.5 ) dist_thr = 1.5;

    agent->debugClient().addMessage( "DefMove%.0f", dash_power );
    agent->debugClient().setTarget( M_home_pos );
    agent->debugClient().addCircle( M_home_pos, dist_thr );
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        __FILE__": go home. power=%.1f",
                        dash_power );

    if ( ! rcsc::Body_GoToPoint( M_home_pos,
                                 dist_thr,
                                 dash_power
                                 ).execute( agent ) )
    {
        rcsc::AngleDeg body_angle = 180.0;
        if ( agent->world().ball().angleFromSelf().abs() < 80.0 )
        {
            body_angle = 0.0;
        }
        rcsc::Body_TurnToAngle( body_angle ).execute( agent );
    }

    agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  static method
*/
double
Bhv_DefenderBasicBlockMove::get_dash_power( const rcsc::PlayerAgent * agent,
                                            const rcsc::Vector2D & home_pos )
{
    static bool S_recover_mode = false;

    const rcsc::WorldModel & wm = agent->world();

    if ( wm.self().stamina() < rcsc::ServerParam::i().staminaMax() * 0.5 )
    {
        S_recover_mode = true;
    }
    else if ( wm.self().stamina()
              > rcsc::ServerParam::i().staminaMax() * 0.85 )
    {
        S_recover_mode = false;
    }

    const double ball_xdiff
        //= wm.ball().pos().x - home_pos.x;
        = wm.ball().pos().x - wm.self().pos().x;

    const rcsc::PlayerType & mytype = wm.self().playerType();
    const double my_inc = mytype.staminaIncMax() * wm.self().recovery();

    double dash_power;
    if ( S_recover_mode )
    {
        if ( wm.defenseLineX() > wm.self().pos().x )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__": get_dash_power. correct DF line & recover" );
            dash_power = my_inc;
        }
        else if ( ball_xdiff < 5.0 )
        {
            dash_power = rcsc::ServerParam::i().maxPower();
        }
        else if ( ball_xdiff < 10.0 )
        {
            dash_power = rcsc::ServerParam::i().maxPower();
            dash_power *= 0.7;
            //dash_power
            //    = mytype.getDashPowerToKeepSpeed( 0.7, wm.self().effort() );
        }
        else if ( ball_xdiff < 20.0 )
        {
            dash_power = std::max( 0.0, my_inc - 10.0 );
        }
        else // >= 20.0
        {
            dash_power = std::max( 0.0, my_inc - 20.0 );
        }

        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": get_dash_power. recover mode dash_power= %.1f",
                            dash_power );

        return dash_power;
    }

    // normal case

#if 1
    // added 2006/06/11 03:34
    if ( wm.ball().pos().x > 0.0
         && wm.self().pos().x < home_pos.x )
    {
        double power_for_max_speed = mytype.getDashPowerToKeepMaxSpeed( rcsc::ServerParam::i(),
                                                                        wm.self().effort() );
        double defense_dash_dist = wm.self().pos().dist( rcsc::Vector2D( -48.0, 0.0 ) );
        int cycles_to_reach = mytype.cyclesToReachDistance( defense_dash_dist );
        int available_dash_cycles = mytype.getMaxDashCyclesSavingStamina( rcsc::ServerParam::i(),
                                                                          power_for_max_speed,
                                                                          wm.self().stamina(),
                                                                          wm.self().recovery() );
        if ( available_dash_cycles < cycles_to_reach )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__": get_dash_power. keep stamina for defense back dash,"
                                " power_for_max=%.1f"
                                " dash_dist=%.1f, reach_cycle=%d, dashable_cycle=%d",
                                power_for_max_speed,
                                defense_dash_dist, cycles_to_reach, available_dash_cycles );
            dash_power = std::max( 0.0, my_inc - 20.0 );
            return dash_power;
        }
    }
#endif

    if ( wm.self().pos().x < -30.0
         && wm.defenseLineX() > wm.self().pos().x )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": get_dash_power. correct dash power for the defense line" );
        dash_power = rcsc::ServerParam::i().maxPower();
    }
    else if ( home_pos.x < wm.self().pos().x )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": get_dash_power. max power to go to the behind home position" );
        dash_power = rcsc::ServerParam::i().maxPower();
    }
    else if ( ball_xdiff > 20.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": get_dash_power. correct dash power to save stamina(1)" );
        dash_power = my_inc;
    }
    else if ( ball_xdiff > 10.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": get_dash_power. correct dash power to save stamina(2)" );
        dash_power = rcsc::ServerParam::i().maxPower();
        dash_power *= 0.6;
        //dash_power = mytype.getDashPowerToKeepSpeed( 0.6, wm.self().effort() );
    }
    else if ( ball_xdiff > 5.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": get_dash_power. correct dash power to save stamina(3)" );
        dash_power = rcsc::ServerParam::i().maxPower();
        dash_power *= 0.85;
        //dash_power = mytype.getDashPowerToKeepSpeed( 0.85, wm.self().effort() );
    }
    else
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": get_dash_power. max power" );
        dash_power = rcsc::ServerParam::i().maxPower();
    }

    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        __FILE__": get_dash_power. nomal mode dash_power=%.1f",
                        dash_power );

    return dash_power;
}


/*-------------------------------------------------------------------*/
/*!
  static method
*/
double
Bhv_DefenderBasicBlockMove::get_line_defense_x( const rcsc::PlayerAgent * agent,
                                                const rcsc::Vector2D & home_pos )
{
    const rcsc::WorldModel & wm = agent->world();

    if ( wm.ball().pos().x > -30.0
         && wm.self().pos().x > home_pos.x
         && wm.ball().pos().x > wm.self().pos().x + 3.0 )
    {
        double tmpx;
        if ( wm.ball().pos().x > 0.0 ) tmpx = -3.5;
        else if ( wm.ball().pos().x > -8.0 ) tmpx = -11.5;
        else if ( wm.ball().pos().x > -16.0 ) tmpx = -19.5;
        else if ( wm.ball().pos().x > -24.0 ) tmpx = -27.5;
        else if ( wm.ball().pos().x > -30.0 ) tmpx = -33.5;
        else tmpx = home_pos.x;

        if ( tmpx > 0.0 ) tmpx = 0.0;
        if ( tmpx > home_pos.x + 5.0 ) tmpx = home_pos.x + 5.0;

        if ( wm.defenseLineX() < tmpx - 1.0 )
        {
            tmpx = wm.defenseLineX() + 1.0;
        }

        return tmpx;
    }

    return home_pos.x;
}
