// -*-c++-*-

/*!
  \file shoot_table.h
  \brief shoot plan search and holder class
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_ACTION_SHOOT_TABLE_H
#define RCSC_ACTION_SHOOT_TABLE_H

#include <rcsc/geom/line_2d.h>
#include <rcsc/geom/vector_2d.h>
#include <rcsc/game_time.h>

#include <functional>
#include <vector>

namespace rcsc {

class PlayerAgent;
class PlayerObject;
class WorldModel;

/*!
  \class ShootTable
  \brief shoot plan search and holder table
 */
class ShootTable {
public:
    struct Shot {
        Vector2D point_;
        Vector2D vel_;
        double speed_;
        AngleDeg angle_;
        bool goalie_never_reach_;
        int score_;

        Shot( const Vector2D & point,
              const double & speed,
              const AngleDeg & angle )
            : point_( point )
            , vel_( Vector2D::polar2vector( speed, angle ) )
            , speed_( speed )
            , angle_( angle )
            , goalie_never_reach_( true )
            , score_( 0 )
          { }
    };

    typedef std::vector< Shot > ShotCont;


    struct ScoreCmp
        : public std::binary_function< Shot,
                                       Shot,
                                       bool > {
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return lhs.score_ > rhs.score_;
          }
    };

private:
    //! last calculated time
    GameTime M_time;
    //! cached calculated shoot pathes
    ShotCont M_shots;

public:
    /*!
      \brief accessible from global.
     */
    ShootTable()
      { }

    const
    ShotCont & getShots( const PlayerAgent * agent )
      {
          search( agent );
          return M_shots;
      }

private:

    /*!
      \brief search shoot route patterns. goal mouth is divided to several
      segment and several ball speed for each segment are calculated.
      \param agent const pointer to the agent itself
     */
    void search( const PlayerAgent * agent );

    void calculateShotPoint( const WorldModel & wm,
                             const Vector2D & shot_point,
                             const PlayerObject * goalie );
    bool canScore( const WorldModel & wm,
                   Shot * shot );
    bool maybeGoalieCatch( const WorldModel & wm,
                           const PlayerObject * goalie,
                           Shot * shot );

};

}

#endif
