// -*-c++-*-

/*!
  \file triangle_2d.h
  \brief 2D triangle class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_GEOM_TRIANGLE2D_H
#define RCSC_GEOM_TRIANGLE2D_H

#include <rcsc/geom/vector_2d.h>

namespace rcsc {

/*!
  \class Triangle2D
  \brief 2D triangle class
*/
class Triangle2D {
private:
    Vector2D M_a; //!< first vertex point
    Vector2D M_b; //!< second vertex point
    Vector2D M_c; //!< third vertex point

    //! not used
    Triangle2D();
public:
    /*!
      \brief constructor with variables
      \param v1 first vertex point
      \param v2 second vertex point
      \param v3 third vertex point
    */
    Triangle2D( const Vector2D & v1,
                const Vector2D & v2,
                const Vector2D & v3 )
        : M_a( v1 )
        , M_b( v2 )
        , M_c( v3 )
      { }

    /*!
      \brief assign new vertex points
      \param v1 first vertex point
      \param v2 second vertex point
      \param v3 third vertex point
      \return const reference to itself
    */
    const
    Triangle2D & assign( const Vector2D & v1,
                         const Vector2D & v2,
                         const Vector2D & v3 )
      {
          M_a = v1;
          M_b = v2;
          M_c = v3;
          return *this;
      }

    /*!
      \brief get 1st point
      \return const reference to the member variable
     */
    const
    Vector2D & a() const
      {
          return M_a;
      }

    /*!
      \brief get 2nd point
      \return const reference to the member variable
     */
    const
    Vector2D & b() const
      {
          return M_b;
      }

    /*!
      \brief get 3rd point
      \return const reference to the member variable
     */
    const
    Vector2D & c() const
      {
          return M_c;
      }

    /*!
      \brief get the area of this region
      \return value of the area
     */
    double area() const
      {
          // outer product == area of parallelogram(Heikou Shihenkei)
          // triangle area is a half of parallelogram area
          return std::fabs( ( b() - a() ).outerProduct( c() - a() ) ) * 0.5;
      }

    /*!
      \brief check if this triangle contains 'point'.
      \param point considerd point
      \return true or false
    */
    bool contains( const Vector2D & point ) const;

    /*!
      \brief get the center of gravity(centroid, JUU-SIN)
      \return coordinates of gravity center
     */
    Vector2D getCentroid() const
      {
          return centroid( a(), b(), c() );
      }

    /*!
      \brief get the center of inscribed circle(NAI-SIN)
      \return coordinates of inner center
    */
    Vector2D getIncenter() const
      {
          return incenter( a(), b(), c() );
      }

    /*!
      \brief get the center of circumscribed circle(GAI-SIN)
      \return coordinates of outer center
    */
    Vector2D getCircumcenter() const
      {
          return circumcenter( a(), b(), c() );
      }

    /*!
      \brief get the orthocenter(SUI-SIN)
      \return coordinates of ortho center
    */
    Vector2D getOrthocenter() const
      {
          return orthocenter( a(), b(), c() );
      }

    //brief get the excenter coordinates(BOU-SIN)
    //return coordinates of excenter
    //Vector2D getExcenter() const;


    // static utility

    /*!
      \brief get the center of gravity(JUU-SIN)
      \param a triangle's 1st vertex
      \param b triangle's 2nd vertex
      \param c triangle's 3rd vertex
      \return coordinates of gravity center

      centroid = (a + b + c) / 3
     */
    static
    Vector2D centroid( const Vector2D & a,
                       const Vector2D & b,
                       const Vector2D & c )
      {
          Vector2D g( a );
          g += b;
          g += c;
          g /= 3.0;
          return g;
      }

    /*!
      \brief get the incenter point(NAI-SIN)
      \param a triangle's 1st vertex
      \param b triangle's 2nd vertex
      \param c triangle's 3rd vertex
      \return coordinates of incenter
     */
    static
    Vector2D incenter( const Vector2D & a,
                       const Vector2D & b,
                       const Vector2D & c );

    /*!
      \brief get the circumcenter point(GAI-SIN)
      \param a triangle's 1st vertex
      \param b triangle's 2nd vertex
      \param c triangle's 3rd vertex
      \return coordinates of circumcenter
     */
    static
    Vector2D circumcenter( const Vector2D & a,
                           const Vector2D & b,
                           const Vector2D & c );

    /*!
      \brief get the orthomcenter point(SUI-SIN)
      \param a triangle's 1st vertex
      \param b triangle's 2nd vertex
      \param c triangle's 3rd vertex
      \return coordinates of orthocenter

      orthocenter = a + b + c - 2 * circumcenter
     */
    static
    Vector2D orthocenter( const Vector2D & a,
                          const Vector2D & b,
                          const Vector2D & c );
};

}

#endif
