// -*-c++-*-

/*!
  \file logger.h
  \brief Logger class Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_LOGGER_H
#define RCSC_PLAYER_LOGGER_H

#include <boost/cstdint.hpp>

#include <cstdio>

namespace rcsc {

class PlayerAgent;

/*!
  \class Logger
  \brief log output manager
*/
class Logger {
public:

    static const boost::int32_t LEVEL_00 = 0x00000000;
    static const boost::int32_t LEVEL_01 = 0x00000001;
    static const boost::int32_t LEVEL_02 = 0x00000002;
    static const boost::int32_t LEVEL_03 = 0x00000004;
    static const boost::int32_t LEVEL_04 = 0x00000008;
    static const boost::int32_t LEVEL_05 = 0x00000010;
    static const boost::int32_t LEVEL_06 = 0x00000020;
    static const boost::int32_t LEVEL_07 = 0x00000040;
    static const boost::int32_t LEVEL_08 = 0x00000080;
    static const boost::int32_t LEVEL_09 = 0x00000100;
    static const boost::int32_t LEVEL_10 = 0x00000200;
    static const boost::int32_t LEVEL_11 = 0x00000400;
    static const boost::int32_t LEVEL_12 = 0x00000800;
    static const boost::int32_t LEVEL_13 = 0x00001000;
    static const boost::int32_t LEVEL_14 = 0x00002000;
    static const boost::int32_t LEVEL_15 = 0x00004000;
    static const boost::int32_t LEVEL_16 = 0x00008000;
    static const boost::int32_t LEVEL_17 = 0x00010000;
    static const boost::int32_t LEVEL_18 = 0x00020000;
    static const boost::int32_t LEVEL_19 = 0x00040000;
    static const boost::int32_t LEVEL_20 = 0x00080000;
    static const boost::int32_t LEVEL_21 = 0x00100000;
    static const boost::int32_t LEVEL_22 = 0x00200000;
    static const boost::int32_t LEVEL_23 = 0x00400000;
    static const boost::int32_t LEVEL_24 = 0x00800000;
    static const boost::int32_t LEVEL_25 = 0x01000000;
    static const boost::int32_t LEVEL_26 = 0x02000000;
    static const boost::int32_t LEVEL_27 = 0x04000000;
    static const boost::int32_t LEVEL_28 = 0x08000000;
    static const boost::int32_t LEVEL_29 = 0x10000000;
    static const boost::int32_t LEVEL_30 = 0x20000000;
    static const boost::int32_t LEVEL_31 = 0x40000000;
    static const boost::int32_t LEVEL_32 = 0x80000000;
    static const boost::int32_t LEVEL_ANY = 0xffffffff;

    /*************************************************
    Log Message Line Format:
    Line := <Time> <Level> <Type> <Content>
    Time := integer value
    Level := integer value
    Type :=  T | P | L | C | M
        T: text
        P: point
        L: line
        C: circle
        M: message
    Content := <Text> | <Point> | <Line> | <Circle> | <Message>
    Text := <Str>
    Point := <x:Real> <y:Real>
    Line := <x1:Real> <y1:Real> <x2:Real> <y2:Real>
    Circle := <x:Real> <y:Real> <r:Real>
    Message := <x:Real> <y:Real> <Str>
    **************************************************/

    static const boost::int32_t SYSTEM    = LEVEL_01;
    static const boost::int32_t SENSOR    = LEVEL_02;
    static const boost::int32_t WORLD     = LEVEL_03;
    static const boost::int32_t ACTION    = LEVEL_04;
    static const boost::int32_t INTERCEPT = LEVEL_05;
    static const boost::int32_t KICK      = LEVEL_06;
    static const boost::int32_t DRIBBLE   = LEVEL_07;
    static const boost::int32_t PASS      = LEVEL_08;
    static const boost::int32_t CROSS     = LEVEL_09;
    static const boost::int32_t SHOOT     = LEVEL_10;
    static const boost::int32_t CLEAR     = LEVEL_11;
    static const boost::int32_t TEAM      = LEVEL_12;
    static const boost::int32_t ROLE      = LEVEL_13;

private:

    //! const pointer to PlayerAgent
    const PlayerAgent * M_agent;

    //! output file stream
    FILE* M_fout;

    //! log level flag
    boost::int32_t M_flags;

public:
    /*!
      \brief allocate message buffer memory
     */
    Logger();

    /*!
      \brief if file is opened, flush buffer and close file.
     */
    ~Logger();

    /*!
      \brief set new flag Id
      \param agent const pointer to player agent instance
      \param id new flag Id
      \param on if true, set flag for id
     */
    void setLogFlag( const PlayerAgent * agent,
                     const boost::int32_t id,
                     const bool on = true );

    /*!
      \brief check if level is included
      \param id checked flag Id
      \return true if id is included in flags
     */
    bool isLogFlag( const boost::int32_t id ) const
      {
#ifdef _MSC_VER
		  return BOOL( M_flags & id ) == TRUE;
#else
		  return ( M_flags & id );
#endif
      }

    /*!
      \brief open file to record
      \param file_path file path to open
     */
    void open( const char * file_path );

    /*!
      \brief check if file is opend
      \return true if file is opened
     */
    bool isOpen()
      {
          return ( M_fout != NULL );
      }

    /*!
      \brief put message to file directry. no flush.
      \param msg message
     */
    void print( const char * msg );

    /*!
      \brief add free message to buffer with cycle, level & message tag 'T'
      \param id debug flag id
      \param msg message
     */
    void addText( const boost::int32_t id,
                  char * msg, ... );
#if 0
    /*!
      \brief add point info to buffer with cycle, level & message tag 'P'
      \param x point coordinate x
      \param y point coordinate y
     */
    void addPoint( const boost::int32_t id,
                   const double & x,
                   const double & y );

    /*!
      \brief add point info to buffer with cycle, level & message tag 'C'
      \param x circle center point coordinate x
      \param y circle center point coordinate y
      \param r circle radius
     */
    void addCircle( const boost::int32_t id,
                    const double & x,
                    const double & y,
                    const double & r );

    /*!
      \brief add line info to buffer with cycle, level & message tag 'L'
      \param x1 line start point coordinate x
      \param y1 line start point coordinate y
      \param x2 line end point coordinate x
      \param y2 line end point coordinate y
     */
    //! put text with cycle, level & message tag 'L'
    void addLine( const boost::int32_t id,
                  const double & x1,
                  const double & y1,
                  const double & x2,
                  const double & y2 );

    /*!
      \brief add point info to buffer with cycle, level & message tag 'M'
      \param x text drawed point coordinate x
      \param y text drawed point coordinate y
      \param msg drawd text
     */
    void addMessage( const boost::int32_t id,
                     const double & x,
                     const double & y,
                     char * msg, ... );
#endif
    /*!
      \brief flush stored message
    */
    void flush();

    /*!
      \brief clear buffer without flush
    */
    void clear();
};

//! global variable
extern Logger dlog;

}

#endif
