// -*-c++-*-

/*!
  \file serializer.h
  \brief rcg serializer class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_RCG_SERIALIZER_H
#define RCSC_RCG_SERIALIZER_H

#include <rcsc/rcg/types.h>
#include <rcsc/types.h>

#include <boost/shared_ptr.hpp>

#include <string>
#include <ostream>

namespace rcsc {
namespace rcg {

/*!
  \class Serializer
  \brief rcg data serializer interface class
*/
class Serializer {
private:
    char M_playmode;
    team_t M_teams[2];


protected:

    /*!
      \brief initialize member variables.
      constructor is accessible only from the derived classes.
     */
    Serializer();

    /*!
      \brief virtual destructor
    */
    virtual
    ~Serializer()
      { }

    /////////////////////////////////////////////////////////////
    // implementations

    /*!
      \brief write header
      \param os reference to the output stream
      \param version log version
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const int version );

    /*!
      \brief write server param
      \param os reference to the output stream
      \param param server_params_t variable
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const server_params_t & param );

    /*!
      \brief write player param
      \param os reference to the output stream
      \param pparam plyaer_params_t variable
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const player_params_t & pparam );

    /*!
      \brief write player type param
      \param os reference to the output stream
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const player_type_t & type );

    /*!
      \brief write team info
      \param os reference to the output stream
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const team_t & team_l,
                                  const team_t & team_r );
    /*!
      \brief write playmode
      \param os reference to the output stream
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const char pmode );

    /*!
      \brief write playmode
      \param os reference to the output stream
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const PlayMode pmode );

    /*!
      \brief write dispinfo (version 1 protocol)
      \param os reference to the output stream
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const dispinfo_t & disp );

    /*!
      \brief write showinfo (version 2 protocol)
      \param os reference to the output stream
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const showinfo_t & show );

    /*!
      \brief write showinfo_t2 (version 3 protocol).
      data is converted to short_showinfo_t2.
      \param os reference to the output stream
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const showinfo_t2 & show2 );

    /*!
      \brief write short_showinfo (version 3 protocol)
      \param os reference to the output stream
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const short_showinfo_t2 & show2 );

    /*!
      \brief write message info
      \param os reference to the output stream
      \return reference to the output stream
    */
    std::ostream & serializeImpl( std::ostream & os,
                                  const msginfo_t & msg );


    /*!
      \brief write dispinfo_t2, but data is converted.
      \param os reference to the output stream
      \return reference to the output stream
     */
    std::ostream & serializeImpl( std::ostream & os,
                                  const dispinfo_t2 & disp2 );



public:
    /////////////////////////////////////////////////////////////
    // utility

    /*!
      \brief convert team inf to team_t
      \param name source team name string
      \param score source team score
      \param to destination team_t variable
     */
    static
    void convert( const std::string & name,
                  const int score,
                  team_t & to );

    /*!
      \brief convert showinfo_t2 to showinfo_t
      \param from source showinfo_t2 variable
      \param to destination showinfo_t variable
     */
    static
    void convert( const showinfo_t2 & from,
                  showinfo_t & to );

    /*!
      \brief make msginfo_t from string
      \param from source message string
      \param to destination msginfo_t variable
    */
    static
    void convert( const std::string & from,
                  msginfo_t & to );

    /////////////////////////////////////////////////////////////
    // interfaces

    /*!
      \brief write header
      \param os reference to the output stream
      \return reference to the output stream
    */
    virtual
    std::ostream & serializeHeader( std::ostream & os ) = 0;

    /*!
      \brief write header
      \param os reference to the output stream
      \param param server_params_t variable by network byte order
      \return reference to the output stream
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const server_params_t & param ) = 0;

    /*!
      \brief write header
      \param os reference to the output stream
      \param pparam player_params_t variable by network byte order
      \return reference to the output stream
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const player_params_t & pparam ) = 0;

    /*!
      \brief write header
      \param os reference to the output stream
      \param type player_type_t variable by network byte order
      \return reference to the output stream
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const player_type_t & type ) = 0;

    /*!
      \brief write showinfo_t2, but data should be converted.
      \param os reference to the output stream
      \param show2 showinfo_t2 variable by network byte order
      \return reference to the output stream
     */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const showinfo_t2 & show2 ) = 0;

    /*!
      \brief write message info
      \param os reference to the output stream
      =param msg msginfo_t variable by network byte order
      \return serialization result
    */
    virtual
    std::ostream & serialize( std::ostream & os,
                              const msginfo_t & msg ) = 0;

};


typedef boost::shared_ptr< Serializer > SerializerPtr;

} // end of namespace rcg
} // end of namespace rcsc

#endif
