#include "StepDebugManager.h"
#include <process.h>
#include "StepDebug_Define.h"
#include <afxdlgs.h>
#include "IndexSelectDialog.h"

auto_ptr<CStepDebugManager> CStepDebugManager::_instance;

// _thread_event use between main thread and Thread_Online_Write thread
CEvent	_thread_event(FALSE, 
					  FALSE,
					  "Step Debug Thread",
					  NULL);


CStepDebugManager::CStepDebugManager(void)
: _type(DISABLE)
, _pAgent( NULL )
, _pIndexFile( 0 )
, _pDataFile( 0 )
, _pData( 0 )
, _iCurrentIndex( -1 )
, _bCloseThread( false )
, _bOutputing( false )
, _pDecisionFile( 0 )
, _iCommandList_PlayerIndex( -1 )
{
}

CStepDebugManager::~CStepDebugManager(void)
{
	CloseThread();
	
	if( _pIndexFile != 0 )
		fclose( _pIndexFile );

	if( _pDataFile != 0 )
		fclose( _pDataFile );

	delete _pData;
}


CStepDebugManager* CStepDebugManager::Instance()
{
	if( 0== _instance.get())
	{
		_instance.reset( new CStepDebugManager);
	}
	return _instance.get();
}

bool CStepDebugManager::Initialize( rcsc::PlayerAgent*	pAgent, TYPE t )
{
	_pAgent = pAgent;
	if( _pAgent == NULL )
		return false;

	switch( t )
	{
	case ONLINE_WRITE:	// initialize file, event and thread
		{
			_thread_event.ResetEvent();

			const rcsc::WorldModel& wm = _pAgent->world();

			// TODO
			CString	szDataFileName;
			szDataFileName.Format("StepDebug_%d.dlog", wm.self().unum() );
			if( (_pDataFile = fopen( (LPCTSTR)szDataFileName, "wb+" )) == NULL )
			{
				std::cerr << __FILE__ << ": " << __LINE__
						  << "[StepDebug] " 
						  << "error: " 
						  << "cannot open StepDebug file"
						  << "(" << (LPCTSTR)szDataFileName << ")"
						  << std::endl;
				return false;
			}

			CString	szIndexFileName;
			szIndexFileName.Format("StepDebug_%d.ilog", wm.self().unum() );
			if( (_pIndexFile = fopen( (LPCTSTR)szIndexFileName, "wb+" )) == NULL )
			{
				std::cerr << __FILE__ << ": " << __LINE__
						  << "[StepDebug] " 
						  << "error: " 
						  << "cannot open StepDebug file"
						  << "(" << (LPCTSTR)szIndexFileName << ")"
						  << std::endl;
				return false;
			}

			_beginthread( Thread_Online_Write, 0 , this );
		}
		break;

	case OFFLINE_READ:	// initialize files
		{
			// Open dialog to select index or data file
			CString szDataFileName, szIndexFileName;
			{
				char szFilters[]=
				  "Index Log Files (*.ilog)|*.ilog|Data Log Files (*.dlog)|*.dlog|All Files (*.*)|*.*||";

				CFileDialog fileDlg (TRUE, "ilog", "*.ilog",
				  OFN_FILEMUSTEXIST| OFN_HIDEREADONLY, szFilters, NULL);

				if( fileDlg.DoModal ()==IDOK )
				{
				  CString pathName = fileDlg.GetPathName();

				  if( pathName.Right(5) == ".ilog" )
				  {
					  szDataFileName = szIndexFileName = pathName;
					  szDataFileName.Replace( ".ilog", ".dlog" );
				  }
				  else if( pathName.Right(5) == ".dlog" )
				  {
					  szDataFileName = szIndexFileName = pathName;
					  szIndexFileName.Replace( ".dlog", ".ilog" );
				  }
				  else
				  {
					  AfxMessageBox("ERROR File");
					  return false;
				  }

				}
				else
					return false;
			}

			// open file handle
			{
				if( (_pDataFile = fopen( (LPCTSTR)szDataFileName, "rb" )) == NULL )
				{
					std::cerr << __FILE__ << ": " << __LINE__
							  << "[StepDebug] " 
							  << "error: " 
							  << "cannot open StepDebug file"
							  << "(" << (LPCTSTR)szDataFileName << ")"
							  << std::endl;
					return false;
				}

				if( (_pIndexFile = fopen( (LPCTSTR)szIndexFileName, "rb" )) == NULL )
				{
					std::cerr << __FILE__ << ": " << __LINE__
							  << "[StepDebug] " 
							  << "error: " 
							  << "cannot open StepDebug file"
							  << "(" << (LPCTSTR)szIndexFileName << ")"
							  << std::endl;
					return false;
				}

				_logfilepath = szIndexFileName;
			}

			// read _IndexLoopMap from index file
			{
				while( !feof( _pIndexFile ) )
				{
					STEPDEBUG_INDEX si;
					if( 1 != fread( &si, sizeof( STEPDEBUG_INDEX ), 1, _pIndexFile ) )
						continue;

					// save to map
					_IndexLoopMap[ si._dwIndex ] = si;

					if( ferror( _pIndexFile ) )
						break;
				}

			}

			// open index select dialog
			{
				CIndexSelectDialog dlg;
				dlg.DoModal();
			}

		}
		break;

	case DISABLE:
		break;

	default:
		return false;
	}

	_type = t;
	return true;
}

void CStepDebugManager::Thread_Online_Write(void *p)
{
	CStepDebugManager *pManager = (CStepDebugManager*)p;

	while( !pManager->IsCloseThread() )
	{
		if( WaitForSingleObject( _thread_event.m_hObject, INFINITE ) == WAIT_FAILED )
		{
			std::cerr	<< __FILE__ << ": " << __LINE__
						<< "[StepDebug] " 
						<< "error: " 
						<< "WaitForSingleObject failed"
						<< std::endl;
			break;
		}

		pManager->Lock();

		// write current loop to log file
		pManager->Write2StepDebugFile();

		pManager->Unlock();

//		_thread_event.ResetEvent();
	}

	/* _endthread given to terminate */
    _endthread();
}

CStepDebugManager::TYPE CStepDebugManager::GetType()
{
	return _type;
}

void CStepDebugManager::SetType( CStepDebugManager::TYPE t )
{
	_type = t;
}

bool CStepDebugManager::Write2StepDebugFile()
{
//	HTRACE_TIME( "Write2StepDebugFile" );

	if( _pAgent == 0 ||
		_pIndexFile == 0 ||
		_pDataFile == 0 )
		return false;

	const rcsc::WorldModel& wm = _pAgent->world();
	

	// write index
	{
		// check pointer
		if( fseek(_pDataFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( _pDataFile );
		if( lStart == -1 )
			return false;

		// format sturcture
		static size_t	nIndex = 0;

		STEPDEBUG_INDEX	si;
		si._dwPosition	= lStart;
		si._dwIndex		= nIndex;
		si._dwLoop		= wm.time().cycle();

		// write index file
		if( fseek(_pIndexFile,0L,SEEK_END) != 0 )
			return false;
		nIndex++;
		fwrite( &si, sizeof( STEPDEBUG_INDEX ), 1, _pIndexFile );
		fflush( _pIndexFile );

	}

	// write data
	{
		if( !wm.Write2StepDebugFile(_pDataFile) )
			return false;

		if( !_pAgent->config().Write2StepDebugFile(_pDataFile) )
			return false;

		if( !_pAgent->effector().Write2StepDebugFile(_pDataFile) )
			return false;

		// PlayerAgentImpl
		if( !_pAgent->Write2StepDebugFile_PlayerAgentImpl(_pDataFile) )
			return false;
	}

	return true;
}
const STEPDEBUG_INDEX* CStepDebugManager::GetIndexBlock( int iIndex )
{
	// no data file
	if( _pDataFile == 0 || _IndexLoopMap.find( iIndex ) == _IndexLoopMap.end() )
		return NULL;

	return &(_IndexLoopMap[ iIndex ]);
}

const char* CStepDebugManager::Load_By_Index( int iIndex )
{
	// the data has alreadly been load.
	if( _pData != 0 && _iCurrentIndex == iIndex )
	{
		return _pData;
	}
	else
	{
		delete _pData;
		_pData = 0;
		_iCurrentIndex = -1;
	}

	// no data file
	if( _pDataFile == 0 || _IndexLoopMap.find( iIndex ) == _IndexLoopMap.end() )
		return NULL;


	// check size
	long lSize( 0 );
	if( _IndexLoopMap.find( iIndex + 1 ) != _IndexLoopMap.end() )
	{
		lSize = _IndexLoopMap[ iIndex + 1 ]._dwPosition - _IndexLoopMap[ iIndex ]._dwPosition;
	}
	else
	{
		if( fseek(_pDataFile,0L,SEEK_END) != 0 )
			return 0;

		long lEnd		= ftell( _pDataFile );
		if( lEnd == -1 )
			return 0;

		lSize = lEnd - _IndexLoopMap[ iIndex ]._dwPosition;
	}

	if( lSize <= 0 )
		return 0;

	// read data from data file
	{
		if( fseek( _pDataFile, _IndexLoopMap[ iIndex ]._dwPosition, SEEK_SET ) != 0 )
			return false;

		_pData = new char[lSize];
		if( fread( _pData, lSize, 1, _pDataFile ) != 1 )
		{
			delete _pData;
			_pData = 0;
			return 0;
		}

		_iCurrentIndex = iIndex;
	}

	return _pData;
}

bool CStepDebugManager::GetIndexLoopMap( std::map< int, STEPDEBUG_INDEX >  **ppIndexLoopMap )
{
	*ppIndexLoopMap = &_IndexLoopMap;

	return true;
}

void CStepDebugManager::agent_action( const char* data )
{
	if( data == 0 )
	{
		ASSERT(false);
		return;
	}

	const char* pCurrent = data;

	// load data to world model
	if( !_pAgent->world().ReadFromDataFile( pCurrent ) )
	{
		::MessageBox(NULL, "Fail to ReadFromDataFile", "Fatal Error", MB_OK );
		return;
	}
	pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

	// load data to M_config
	if( !_pAgent->config().ReadFromDataFile( pCurrent ) )
	{
		::MessageBox(NULL, "Fail to ReadFromDataFile", "Fatal Error", MB_OK );
		return;
	}
	pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

	// load data to M_effector
	if( !_pAgent->effector().ReadFromDataFile( pCurrent ) )
	{
		::MessageBox(NULL, "Fail to ReadFromDataFile", "Fatal Error", MB_OK );
		return;
	}
	pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

	// load data to PlayerAgentImpl
	if( !_pAgent->ReadFromDataFile_PlayerAgentImpl( pCurrent ) )
	{
		::MessageBox(NULL, "Fail to ReadFromDataFile", "Fatal Error", MB_OK );
		return;
	}
	pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

	// do action
	_pAgent->action();
}

void CStepDebugManager::CloseThread()
{
	_bCloseThread = true;
//	_thread_event.SetEvent();
}

bool CStepDebugManager::IsCloseThread()
{
	return _bCloseThread;
}

bool CStepDebugManager::Online_Write()
{
	return _thread_event.SetEvent() == TRUE;
}

void CStepDebugManager::ShowDecision( const std::string& str )
{
	if( _bOutputing )
	{
		if( _pDecisionFile )
		{
			fwrite( str.c_str(), str.size(), 1, _pDecisionFile );
			fwrite( "\r\n", 2, 1, _pDecisionFile );
		}
	}
	else
	{
		_strDecision = str;
//		::MessageBox( NULL, str.c_str(), "Decision", MB_OK );
	}
}

void CStepDebugManager::OutputAllDecision()
{
	CWaitCursor	wc;

	_bOutputing = true;

	// open file
	CString strPath = _logfilepath.c_str();
	strPath.Replace(".ilog", "_decision.txt");
	_pDecisionFile = fopen( (LPCTSTR)strPath, "wb+" );

	if( _pDecisionFile )
	{

		std::map< int, STEPDEBUG_INDEX >::const_iterator	itr;
		for( itr = _IndexLoopMap.begin(); itr != _IndexLoopMap.end(); itr++ )
		{
			const char* pData = Load_By_Index( itr->first );
			if( pData )
			{
				CString str;
//				str.Format( "Loop %d (%d) :  ", itr->second._dwLoop, itr->second._dwIndex );
				str.Format( "Loop %d :  ", itr->second._dwLoop );
				fwrite( (LPCTSTR)str, str.GetLength(), 1, _pDecisionFile );


				agent_action( pData );
			}
		}
		

		// close file
		fclose( _pDecisionFile );
		_pDecisionFile = 0;

		// open this file
		::ShellExecute(NULL,"open", strPath,"","", SW_SHOW );
	}
	else
	{
		ASSERT(false);
	}

	_bOutputing = false;
}