// -*-c++-*-

/*!
  \file stamina_model.h
  \brief player's stamina model Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_STAMINA_MODEL_H
#define RCSC_PLAYER_STAMINA_MODEL_H

#include "fullstate_sensor.h"

namespace rcsc {

class BodySensor;
class PlayerType;
class GameTime;

/*!
  \class StaminaModel
  \brief stamina management class
*/
class StaminaModel {
private:
    //! simulator's default max recovery value
    static const double DEFAULT_RECOVERY_MAX;

    //! stamina value
    double M_stamina;
    //! recover rate
    double M_recovery;
    //! effort rate
    double M_effort;
    //! remained stamina capacity
    double M_stamina_capacity;
public:
    /*!
      \brief init members by built-in values
    */
    StaminaModel();

    /*!
      \brief initialize internal variables with server parameters
      \param stamina_max maximal stamina parameter
      \param effort_max maxmal effort parameter

      initial recovery parameter is always DEFAULT_RECOVERY_MAX
    */
    void init( const double & stamina_max,
               const double & effort_max );

    /*!
      \brief get current stamina value
      \return const reference to the current stamina value
    */
    const
    double & stamina() const
      {
          return M_stamina;
      }

    /*!
      \brief get current recovery value
      \return const reference to the current recovery value
    */
    const
    double & recovery() const
      {
          return M_recovery;
      }

    /*!
      \brief get current effort value
      \return const reference to the current effort value
    */
    const
    double & effort() const
      {
          return M_effort;
      }

    /*!
      \brief get the current remained stamina capacity
      \return stamina capacity value
     */
    const
    double & staminaCapacity() const
      {
          return M_stamina_capacity;
      }

    /*!
      \brief update stamina related variables
      \param player_type PlayerType parameter
      \param dashpower previous dash command parameter

      update without sense_body message.
      this means that this method try to update variables
      only with internal values and estimated previous action effect,
      but we can update the stamina information very safety.
    */
    void update( const PlayerType & player_type,
                 const double & dashpower );

    /*!
      \brief update with sense_body message
      \param player_type PlayerType parameter
      \param body_sensor reference to the body sensor instance
      \param current game time that sense_body was received
    */
    void updateAfterSense( const PlayerType & player_type,
                           const BodySensor & body_sensor,
                           const GameTime & current );

    /*!
      \brief update with fullstate message
      \param state reference to the fullstate sensor instance
     */
    void updateAfterFullstate( const FullstateSensor::PlayerT & state );

};

}

#endif
