// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "actgen_simple_through_pass.h"

#include "pass.h"
#include "simple_pass_checker.h"
#include "field_analyzer.h"

#include "action_state_pair.h"
#include "predict_state.h"

#include <rcsc/player/world_model.h>
#include <rcsc/common/logger.h>
#include <rcsc/timer.h>

#include <limits>

// #define DEBUG_PROFILE
// #define DEBUG_PRINT

using namespace rcsc;

namespace {

int g_generated = 0;

}

/*-------------------------------------------------------------------*/
/*!

 */
void
ActGen_SimpleThroughPass::generate( std::vector< ActionStatePair > * result,
                                    const PredictState & state,
                                    const WorldModel & wm,
                                    const std::vector< ActionStatePair > & path ) const
{
    if ( path.empty() )
    {
        return;
    }

#ifdef DEBUG_PROFILE
    static GameTime s_profile_time( 0, 0 );
    static int s_call_counter = 0;
    static double s_cumulative_msec = 0.0;

    if ( s_profile_time != wm.time() )
    {
        g_generated = 0;
        s_call_counter = 0;
        s_cumulative_msec = 0.0;
    }
    s_profile_time = wm.time();

    ++s_call_counter;

    Timer timer;
#endif

    if ( ( state.ball().pos().x < state.offsideLineX() - 15.0
           && state.ball().pos().x < state.ourOffensePlayerLineX() - 15.0
           && state.ball().pos().x < ServerParam::i().theirPenaltyAreaLineX() - 10.0 ) )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "xxx (through) first point(%.2f 5.2f) too back.",
                      state.ball().pos().x, state.ball().pos().y );
#endif
        return;
    }


    const PredictPlayerPtrCont::const_iterator end = state.allTeammates().end();
    for ( PredictPlayerPtrCont::const_iterator it = state.allTeammates().begin();
          it != end;
          ++it )
    {
        if ( ! (*it)->isValid() )
        {
            continue;
        }

        if ( (*it)->unum() == state.ballHolderUnum() )
        {
            continue;
        }

#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "========== SimpleThroughPass receiver unum:%d ==========",
                      receiver->unum() );
#endif

        createThroughPass( result, state, wm, **it );
    }

#ifdef DEBUG_PROFILE
    double msec = timer.elapsedReal();
    s_cumulative_msec += msec;
    dlog.addText( Logger::ACTION_CHAIN,
                  __FILE__": PROFILE path=%d: call=%d: gen=%d: elapsed %f / %f [ms] ",
                  s_call_counter,
                  g_generated,
                  path.size(),
                  msec, s_cumulative_msec );
#endif
}

/*-------------------------------------------------------------------*/
/*!

 */
void
ActGen_SimpleThroughPass::createThroughPass( std::vector< ActionStatePair > * result,
                                             const PredictState & state,
                                             const WorldModel &,
                                             const AbstractPlayerObject & receiver ) const
{
    // static const int MIN_RECEIVE_STEP = 6;
    //static const int MAX_RECEIVE_STEP = 35;

    static const double MIN_THROUGH_PASS_DIST = 5.0;
    static const double MAX_THROUGH_PASS_DIST
        = 0.9 * inertia_final_distance( ServerParam::i().ballSpeedMax(),
                                        ServerParam::i().ballDecay() );
    // static const double MAX_RECEIVE_BALL_SPEED
    //     = ServerParam::i().ballSpeedMax()
    //     * std::pow( ServerParam::i().ballDecay(), MIN_RECEIVE_STEP );

    static const int ANGLE_DIVS = 8; // 12
    static const double MIN_ANGLE = -40.0;
    static const double MAX_ANGLE = +40.0;
    static const double ANGLE_STEP = ( MAX_ANGLE - MIN_ANGLE ) / ANGLE_DIVS;

    static const double MIN_MOVE_DIST = 6.0;
    static const double MAX_MOVE_DIST = 30.0 + 0.001;
    static const double MOVE_DIST_STEP = 4.0; //2.0;

    const ServerParam & SP = ServerParam::i();
    const PlayerType * ptype = receiver.playerTypePtr();

    const double min_receive_x = std::min( state.offsideLineX() + 15.0,
                                           SP.theirPenaltyAreaLineX() + 5.0 );

    if ( receiver.pos().x < min_receive_x - MAX_MOVE_DIST
         || receiver.pos().x < 1.0
         || ( receiver.pos().x < state.offsideLineX() - 5.0
              && receiver.pos().x < state.ourOffensePlayerLineX() - 5.0
              && receiver.pos().x < SP.theirPenaltyAreaLineX() ) )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "xxx (through) unum=%d too back.",
                      receiver.unum() );
#endif
        return;
    }

    // const double min_ball_speed = SP.defaultRealSpeedMax();
    // const double max_ball_speed = SP.ballSpeedMax();

    // const double min_receive_ball_speed = 0.001;
    // const double max_receive_ball_speed
    //     = std::min( MAX_RECEIVE_BALL_SPEED,
    //                 ptype->kickableArea() + ( SP.maxDashPower()
    //                                           * ptype->dashPowerRate()
    //                                           * ptype->effortMax() ) * 1.5 );

    const SimplePassChecker pass_checker;

    //
    // angle loop
    //
    for ( int a = 0; a <= ANGLE_DIVS; ++a )
    {
        const AngleDeg angle = MIN_ANGLE + ( ANGLE_STEP * a );
        const Vector2D unit_rvec = Vector2D::from_polar( 1.0, angle );

        //
        // distance loop
        //
        for ( double move_dist = MIN_MOVE_DIST;
              move_dist < MAX_MOVE_DIST;
              move_dist += MOVE_DIST_STEP )
        {
            const Vector2D receive_point
                = receiver.pos()
                + unit_rvec * move_dist;
#ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          ">>> (through) from[%d](%.2f %.2f) to[%d](%.2f %.2f)",
                          state.ballHolderUnum(),
                          state.ball().pos().x, state.ball().pos().y,
                          receiver.unum(),
                          receive_point.x, receive_point.y );
#endif

            if ( receive_point.x < min_receive_x )
            {
#ifdef DEBUG_PRINT
                dlog.addText( Logger::ACTION_CHAIN,
                              "xxx receive point is back",
                              state.ballHolderUnum(),
                              state.ball().pos().x, state.ball().pos().y,
                              receiver.unum(),
                              receive_point.x, receive_point.y );
#endif
                continue;
            }

            if ( receive_point.x > SP.pitchHalfLength() - 1.5
                 || receive_point.absY() > SP.pitchHalfWidth() - 1.5 )
            {
#ifdef DEBUG_PRINT
                dlog.addText( Logger::ACTION_CHAIN,
                              "xxx out of pitch" );
#endif
                break;
            }

            const double ball_move_dist = state.ball().pos().dist( receive_point );

            if ( ball_move_dist < MIN_THROUGH_PASS_DIST
                 || MAX_THROUGH_PASS_DIST < ball_move_dist )
            {
#ifdef DEBUG_PRINT
                dlog.addText( Logger::ACTION_CHAIN,
                              "xxx over ball_move_dist=%.3f min=%.3f max=%.3f",
                              ball_move_dist,
                              MIN_THROUGH_PASS_DIST, MAX_THROUGH_PASS_DIST );
#endif
                continue;
            }

            {
                int nearest_receiver_unum = getNearestTeammateUnum( state, receive_point );
                if ( nearest_receiver_unum != receiver.unum() )
                {
#ifdef DEBUG_PRINT
                    dlog.addText( Logger::ACTION_CHAIN,
                                  "xxx other receiver=%d",
                                  nearest_receiver_unum );
#endif
                    break;
                }
            }

            const double receiver_move_dist2 = receiver.pos().dist2( receive_point );

            bool exist_opponent = false;
            for ( PlayerCont::const_iterator o = state.opponents().begin();
                  o != state.opponents().end();
                  ++o )
            {
                if ( o->pos().dist2( receive_point ) < receiver_move_dist2 * std::pow( 0.97, 2 ) )
                {
                    exist_opponent = true;
                    break;
                }
            }

            if ( exist_opponent )
            {
#ifdef DEBUG_PRINT
                dlog.addText( Logger::ACTION_CHAIN,
                              "xxx (through) exist opponent near to receive pos" );
#endif
                continue;
            }

            const int receiver_step = ptype->cyclesToReachDistance( std::sqrt( receiver_move_dist2 ) );
            double first_ball_speed =  calc_first_term_geom_series( ball_move_dist,
                                                                    SP.ballDecay(),
                                                                    receiver_step );
            if ( first_ball_speed > SP.ballSpeedMax() )
            {
                first_ball_speed = SP.ballSpeedMax();
            }

            if ( ! pass_checker( state,
                                 *(state.ballHolder()),
                                 receiver,
                                 receive_point,
                                 first_ball_speed ) )
            {
#ifdef DEBUG_PRINT
                dlog.addText( Logger::ACTION_CHAIN,
                              "xxx (through) pass checker." );
#endif
                continue;
            }

            int n_kick = 2;
            int duration_step = calc_length_geom_series( first_ball_speed,
                                                         ball_move_dist,
                                                         SP.ballDecay() )
                + n_kick;

            PredictState::ConstPtr result_state( new PredictState( state,
                                                                   duration_step,
                                                                   receiver.unum(),
                                                                   receive_point ) );

            CooperativeAction::Ptr action( new Pass( state.ballHolderUnum(),
                                                     receiver.unum(),
                                                     receive_point,
                                                     first_ball_speed,
                                                     duration_step,
                                                     n_kick,
                                                     FieldAnalyzer::to_be_final_action( state ),
                                                     "simpleThrough" ) );
            ++g_generated;
            result->push_back( ActionStatePair( action, result_state ) );
        }
    }

}

/*-------------------------------------------------------------------*/
/*!

 */
int
ActGen_SimpleThroughPass::getNearestTeammateUnum( const PredictState & state,
                                                  const Vector2D & pos ) const
{
    int unum = Unum_Unknown;
    double min_dist2 = std::numeric_limits< double >::max();

    for ( PredictPlayerPtrCont::const_iterator p = state.allTeammates().begin();
          p != state.allTeammates().end();
          ++p )
    {
        double d2 = (*p)->pos().dist2( pos );
        if ( d2 < min_dist2 )
        {
            min_dist2 = d2;
            unum = (*p)->unum();
        }
    }

    return unum;
}
