// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "actgen_voronoi_pass.h"

#include "pass.h"
#include "simple_pass_checker.h"

#include "field_analyzer.h"

#include "predict_state.h"
#include "action_state_pair.h"

#include <rcsc/player/world_model.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <rcsc/geom/voronoi_diagram.h>
#include <rcsc/timer.h>

// #define DEBUG_PROFILE
// #define DEBUG_PRINT

using namespace rcsc;

namespace {

struct PointDistCompare {

    const Vector2D point_;

    PointDistCompare( const Vector2D & p )
        : point_( p )
      { }

    bool operator()( const Vector2D & lhs,
                     const Vector2D & rhs ) const
      {
          return lhs.dist2( point_ ) < rhs.dist2( point_ );
      }
};

}

/*-------------------------------------------------------------------*/
/*!

 */
void
ActGen_VoronoiPass::generate( std::vector< ActionStatePair > * result,
                              const PredictState & state,
                              const WorldModel & wm,
                              const std::vector< ActionStatePair > & path ) const
{
    static std::vector< Vector2D > cand;
    static GameTime s_update_time;
    static int s_call_counter = 0;
    static int s_action_count = 0;
    static double s_cumulative_msec = 0.0;

    // not generate as first action
    if ( path.empty() )
    {
        return;
    }


#ifdef DEBUG_PROFILE
    Timer timer;
#endif

    if ( s_update_time != wm.time() )
    {
        s_call_counter = 0;
        s_action_count = 0;
        s_cumulative_msec = 0.0;
        s_update_time = wm.time();
        cand.clear();

        //
        // get voronoi diagram
        //
        const VoronoiDiagram & voronoi = FieldAnalyzer::i().passVoronoiDiagram();

        const double MIN_LEN = 3.0;
        const int MAX_DIV = 8;

        voronoi.getPointsOnSegments( MIN_LEN,
                                     MAX_DIV,
                                     &cand );
        std::sort( cand.begin(), cand.end(),
                   //PointDistCompare( state.ball().pos() ) );
                   PointDistCompare( ServerParam::i().theirTeamGoalPos() ) );

#ifdef DEBUG_PRINT
        const VoronoiDiagram::Segment2DCont::const_iterator s_end = voronoi.resultSegments().end();
        for ( VoronoiDiagram::Segment2DCont::const_iterator it = voronoi.resultSegments().begin();
              it != s_end;
              ++it )
        {
            dlog.addLine( Logger::ACTION_CHAIN,
                          (*it).origin(), (*it).terminal(),
                          "#0000ff" );
        }

        const std::vector< Vector2D >::const_iterator p_end = cand.end();
        for ( std::vector< Vector2D >::const_iterator p = cand.begin();
              p != p_end;
              ++p )
        {
            dlog.addRect( Logger::ACTION_CHAIN,
                          p->x - 0.15,  p->y - 0.15,
                          0.3, 0.3,
                          "#ff00ff" );
        }
#endif
#ifdef DEBUG_PROFILE
        dlog.addText( Logger::ACTION_CHAIN,
                      "voronoi_pass: generate candidates, elapsed %f [ms]",
                      timer.elapsedReal() );
#endif
    }

    ++s_call_counter;

    //
    // check same ball holder
    //
    bool old_holder_table[11];
    for ( int i = 0; i < 11; ++i )
    {
        old_holder_table[i] = false;
    }


    for ( int c = static_cast< int >( path.size() ) - 1; c >= 0; --c )
    {
        int ball_holder_unum = Unum_Unknown;

        if ( c == 0 )
        {
            int self_min = wm.interceptTable()->selfReachCycle();
            int teammate_min = wm.interceptTable()->teammateReachCycle();

            if ( teammate_min < self_min )
            {
                const PlayerObject * teammate = wm.interceptTable()->fastestTeammate();
                if ( teammate )
                {
                    ball_holder_unum = teammate->unum();
                }
            }
            else
            {
                ball_holder_unum = wm.self().unum();
            }
        }
        else
        {
            ball_holder_unum = path[c - 1].state().ballHolder()->unum();
        }

        if ( ball_holder_unum == Unum_Unknown )
        {
            continue;
        }

        old_holder_table[ ball_holder_unum - 1 ] = true;
    }

    //
    // check pass to each pass target
    //
#ifdef DEBUG_PRINT
    dlog.addText( Logger::ACTION_CHAIN,
                  "voronoi_pass: ball = (%f, %f)",
                  state.ball().pos().x, state.ball().pos().y );
#endif

    const ServerParam & SP = ServerParam::i();

    const AbstractPlayerObject * holder = state.ballHolder();

    const SimplePassChecker pass_check;

    PredictPlayerPtrCont receiver_candidates;
    receiver_candidates.reserve( state.allTeammates().size() );

    const PredictPlayerPtrCont::const_iterator p_end = state.allTeammates().end();
    for ( PredictPlayerPtrCont::const_iterator p = state.allTeammates().begin();
          p != p_end;
          ++p )
    {
        if ( ! (*p)->isValid()
             || (*p)->posCount() > 10
             || (*p)->isGhost()
             || (*p)->unum() == state.ballHolderUnum()
             || (*p)->unum() == Unum_Unknown
             || (*p)->unumCount() > 10 )
        {
            continue;
        }

        if ( old_holder_table[ (*p)->unum() - 1 ] )
        {
            // ignore ball holder in chain
            continue;
        }

        receiver_candidates.push_back( *p );
    }

    if ( receiver_candidates.empty() )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      __FILE__": no receiver " );
#endif
        return;
    }

    int generated_count = 0;

    const std::vector< Vector2D >::const_iterator end = cand.end();
    for ( std::vector< Vector2D >::const_iterator it = cand.begin();
          it != end;
          ++it )
    {
        const Vector2D & receive_point = *it;

        //#ifdef DEBUG_PRINT
#if 0
        dlog.addText( Logger::ACTION_CHAIN,
                      "voronoi_pass: receive_point = (%f, %f)",
                      receive_point.x, receive_point.y );
#endif
        if ( ( receive_point.x < 12.0
               && receive_point.x < state.ball().pos().x - 10.0 )
             || receive_point.x < state.ball().pos().x - 16.0 )
        {
            continue;
        }

        const double ball_dist = ( state.ball().pos() - receive_point ).r();
        if ( ball_dist < 5.0
            //|| 30.0 < ball_dist )
             || 25.0 < ball_dist )
        {
            continue;
        }

        PredictPlayerObject::ConstPtr receiver;
        int min_receiver_step = 1000;
        for ( PredictPlayerPtrCont::const_iterator p = receiver_candidates.begin();
              p != receiver_candidates.end();
              ++p )
        {
            const Vector2D receiver_pos = (*p)->inertiaFinalPoint();
            const double receiver_move = receiver_pos.dist( receive_point );

            int receiver_step = (*p)->playerTypePtr()->cyclesToReachDistance( receiver_move );
            //receiver_step += 2; // turn margin
            //receiver_step += 2; // buffer

            if ( min_receiver_step > receiver_step )
            {
                min_receiver_step = receiver_step;
                receiver = *p;
            }
        }

        if ( ! receiver )
        {
            continue;
        }

        double ball_first_speed
            = std::min( calc_first_term_geom_series( ball_dist, SP.ballDecay(), min_receiver_step ),
                        SP.ballSpeedMax() );

        if ( ball_first_speed < 1.2 )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          "voronoi_pass: receive_point=(%.2f %.2f) NG ball_first_speed=%.2f < 1.2",
                          receive_point.x, receive_point.y,
                          ball_first_speed );
#endif
            continue;
        }

        int ball_step = min_receiver_step;
        if ( ball_first_speed > SP.ballSpeedMax() )
        {
            ball_first_speed = SP.ballSpeedMax();

            ball_step
                = static_cast< int >( std::ceil( calc_length_geom_series( ball_first_speed,
                                                                          ball_dist,
                                                                          SP.ballDecay() ) ) );
            if ( min_receiver_step > ball_step )
            {
#ifdef DEBUG_PRINT
                dlog.addText( Logger::ACTION_CHAIN,
                              "voronoi_pass: receive_point=(%.2f %.2f) NG min_recv_step=%d > ball_step=%d",
                              receive_point.x, receive_point.y,
                              min_receiver_step, ball_step );
#endif
                continue;
            }
        }

        if ( ! pass_check( state, *holder, *receiver, receive_point, ball_first_speed ) )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          "voronoi_pass: receive_point=(%.2f %.2f) NG pass_check",
                          receive_point.x, receive_point.y );
#endif
            continue;
        }

        int kick_step = ( ball_first_speed > 2.5
                          ? 3
                          : ball_first_speed > 1.5
                          ? 2
                          : 1 );

        PredictState::ConstPtr result_state( new PredictState( state,
                                                               ball_step + kick_step,
                                                               receiver->unum(),
                                                               receive_point ) );
        CooperativeAction::Ptr action( new Pass( holder->unum(),
                                                 receiver->unum(),
                                                 receive_point,
                                                 ball_first_speed,
                                                 kick_step + ball_step,
                                                 kick_step,
                                                 FieldAnalyzer::to_be_final_action( *result_state ),
                                                 "voronoiPass" ) );
        ++s_action_count;
        ++generated_count;
        action->setIndex( s_action_count );
        result->push_back( ActionStatePair( action, result_state ) );
    }

#ifdef DEBUG_PROFILE
    double msec = timer.elapsedReal();
    s_cumulative_msec += msec;
    dlog.addText( Logger::ACTION_CHAIN,
                  __FILE__": PROFILE %d: path=%d, generated=%d/%d, elapsed %f / %f [ms] ",
                  s_call_counter,
                  path.size(),
                  generated_count, s_action_count,
                  msec, s_cumulative_msec );
#endif
}
