// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_center_back_normal_move.h"

#include "bhv_center_back_defensive_move.h"

#include "strategy.h"
#include "mark_analyzer.h"
#include "defense_system.h"

#include "bhv_basic_tackle.h"
#include "bhv_defensive_mark.h"
#include "bhv_find_player.h"
#include "bhv_get_ball.h"
#include "neck_check_ball_owner.h"
#include "neck_default_intercept_neck.h"
#include "neck_offensive_intercept_neck.h"

#include <rcsc/action/bhv_go_to_point_look_ball.h>
#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_intercept.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/neck_turn_to_ball_and_player.h>
#include <rcsc/action/view_synch.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/intercept_table.h>

#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterBackNormalMove::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::TEAM,
                  __FILE__": Bhv_CenterBackNormalMove" );

    //
    // tackle
    //
    if ( Bhv_BasicTackle( 0.87, 70.0 ).execute( agent ) )
    {
        agent->debugClient().addMessage( "CB:Normal:Tackle" );
        return true;
    }

    //
    // intercept
    //
    if ( doIntercept( agent ) )
    {
        return true;
    }

    if ( doDefensiveMove( agent ) )
    {
        return true;
    }

    if ( doOffensiveMarkMove( agent ) )
    {
        return true;
    }

    doBasicMove( agent );
    return true;

}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterBackNormalMove::doIntercept( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    if ( wm.existKickableTeammate() )
    {
        return false;
    }

    // chase ball
    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    bool intercept = false;

    if ( self_min <= 1 && 1 <= opp_min )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) ok (%d)", __LINE__ );
        intercept = true;
    }
    else if ( self_min <= 2 && 3 <= opp_min )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) ok (%d)", __LINE__ );
        intercept = true;
    }
    else if ( self_min <= 3 && 4 <= opp_min )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) ok (%d)", __LINE__ );
        intercept = true;
    }
    else if ( self_min < 20
              && self_min < mate_min
              && ( self_min <= opp_min - 1
                   && opp_min >= 2 ) )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) ok (%d)", __LINE__ );
        intercept = true;
    }
    else if ( opp_min >= 2
              && self_min <= opp_min + 1
              && self_min <= mate_min )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) ok (%d)", __LINE__ );
        intercept = true;
    }

    Vector2D intercept_pos = wm.ball().inertiaPoint( self_min );

    if ( self_min < 30
         && wm.self().pos().x < -20.0
         && intercept_pos.x < wm.self().pos().x + 1.0 )
    {
        intercept = false;

        if ( self_min <= opp_min + 1
             && 2 <= opp_min
             && self_min <= mate_min )
        {
            dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) ok (%d)", __LINE__ );
            intercept = true;
        }
        else if ( self_min <= opp_min - 3
                  && self_min <= mate_min )
        {
            dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) ok (%d)", __LINE__ );
            intercept = true;
        }
        else if ( self_min <= opp_min - 3
                  && self_min <= mate_min )
        {
            dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) ok (%d)", __LINE__ );
            intercept = true;
        }
        else if ( self_min <= 1 )
        {
            dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) ok (%d)", __LINE__ );
            intercept = true;
        }
    }

    if ( intercept )
    {
        agent->debugClient().addMessage( "CB:Normal:Intercept" );

        Body_Intercept().execute( agent );
        const PlayerObject * opp = ( wm.opponentsFromBall().empty()
                                     ? NULL
                                     : wm.opponentsFromBall().front() );
        if ( opp && opp->distFromBall() < 2.0 )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": (doIntercept) turn_neck to ball");
            agent->setNeckAction( new Neck_TurnToBall() );
        }
        else
        {
            if ( opp_min >= self_min + 3 )
            {
                dlog.addText( Logger::ROLE,
                              __FILE__": (doIntercept) offensive neck");
                agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
            }
            else
            {
                dlog.addText( Logger::ROLE,
                              __FILE__": (doIntercept) default neck");
                agent->setNeckAction( new Neck_DefaultInterceptNeck
                                      ( new Neck_TurnToBallOrScan() ) );
            }
        }

        return true;
    }

    dlog.addText( Logger::ROLE,
                  __FILE__": (doIntercept) xxx");

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterBackNormalMove::doDefensiveMove( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( opp_min <= mate_min + 2 )
    {
        Vector2D opp_trap_pos = wm.ball().inertiaPoint( opp_min );
        if ( opp_trap_pos.x < 12.0 )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": (doBasicMove) switch to the defensive move" );
            agent->debugClient().addMessage( "CB:Normal:Def" );
            Bhv_CenterBackDefensiveMove().execute( agent );
            return true;
        }
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterBackNormalMove::doOffensiveMarkMove( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const AbstractPlayerObject * mark_target = MarkAnalyzer::i().getTargetOf( wm.self().unum() );

    if ( ! mark_target )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doOffensiveMarkMove) no mark target" );
        return false;
    }

    if ( mark_target->ghostCount() >= 3 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doOffensiveMarkMove) ghost mark target. ghost_count=%d",
                      mark_target->ghostCount() );
        return false;
    }

    if ( mark_target->posCount() >= 5 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doOffensiveMarkMove) find mark target" );
        agent->debugClient().addMessage( "CB:Normal:Mark:Find" );
        if ( ! Bhv_FindPlayer( mark_target ).execute( agent ) )
        {
            agent->debugClient().addMessage( "FailedFind" );
            return false;
        }
        return true;
    }

    const Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );
    const Vector2D base_pos = mark_target->pos() + ( mark_target->vel() * 2.0 );
    const double dist_from_home = base_pos.dist( home_pos );

    Vector2D target_point = base_pos;

    if ( dist_from_home > 15.0
         || std::fabs( base_pos.y - home_pos.y ) > 15.0 )
    {
        target_point = base_pos + home_pos;
        target_point *= 0.5;

        dlog.addText( Logger::ROLE,
                      __FILE__": (doOffensiveMarkMove) too far. base=(%.2f %.2f) y_diff=%.3f dist=%.3f",
                      base_pos.x, base_pos.y,
                      std::fabs( base_pos.y - home_pos.y ),
                      dist_from_home );
    }
    else if ( wm.ball().pos().x > target_point.x + 35.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doOffensiveMarkMove) mark adjust (1)" );
        target_point.x += 1.6;
    }
    else if ( wm.ball().pos().x > target_point.x + 20.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doOffensiveMarkMove) mark adjust (2)" );
        target_point.x -= 0.8;
    }
    else if ( wm.ball().pos().x > target_point.x + 20.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doOffensiveMarkMove) mark adjust (3)" );
        target_point.x -= 1.6;
    }

    const double dash_power = DefenseSystem::get_defender_dash_power( wm, target_point );

    // double dist_thr = wm.ball().pos().dist( target_point ) * 0.07;
    // if ( dist_thr < 0.5 ) dist_thr = 0.5;
    double dist_thr = 0.5;

    dlog.addText( Logger::ROLE,
                  __FILE__": (doOffensiveMarkMove) target=(%.2f %.2f) dist_thr=%.2f",
                  target_point.x, target_point.y, dist_thr );

    agent->debugClient().addMessage( "CB:Normal:Mark:%.0f", dash_power );
    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    doGoToPoint( agent, target_point, dist_thr, dash_power,
                 15.0 ); // dir_thr

    return true;
}


/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_CenterBackNormalMove::doBasicMove( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const Vector2D target_point = Strategy::i().getPosition( wm.self().unum() );
    const double dash_power = DefenseSystem::get_defender_dash_power( wm, target_point );

    double dist_thr = wm.ball().pos().dist( target_point ) * 0.1;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    agent->debugClient().addMessage( "CB:Normal:Basic%.0f", dash_power );
    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    dlog.addText( Logger::ROLE,
                  __FILE__": (doNormalMove) go to (%.1f %.1f) dist_thr=%.2f power=%.1f",
                  target_point.x, target_point.y,
                  dist_thr,
                  dash_power );

    doGoToPoint( agent, target_point, dist_thr, dash_power,
                 15.0 ); // dir_thr
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_CenterBackNormalMove::doGoToPoint( PlayerAgent * agent,
                                       const Vector2D & target_point,
                                       const double & dist_thr,
                                       const double & dash_power,
                                       const double & dir_thr )
{
    const WorldModel & wm = agent->world();

    if ( wm.ball().pos().x < -35.0 )
    {
        if ( wm.self().stamina() > ServerParam::i().staminaMax() * 0.6
             && wm.self().pos().x < target_point.x - 4.0
             && wm.self().pos().dist( target_point ) > dist_thr )
        {
            Bhv_GoToPointLookBall( target_point,
                                   dist_thr,
                                   dash_power
                                   ).execute( agent );
            return;
        }
    }

    if ( ! Body_GoToPoint( target_point,
                           dist_thr,
                           dash_power,
                           -1.0, // dash speed
                           100, // cycle
                           true, // stamina save
                           dir_thr
                           ).execute( agent ) )
    {
        AngleDeg body_angle;
        if ( wm.ball().pos().x < -30.0 )
        {
            body_angle = wm.ball().angleFromSelf() + 90.0;
            if ( wm.ball().pos().x < -45.0 )
            {
                if (  body_angle.degree() < 0.0 )
                {
                    body_angle += 180.0;
                }
            }
            else if ( body_angle.degree() > 0.0 )
            {
                body_angle += 180.0;
            }
        }
        else // if ( std::fabs( wm.self().pos().y - wm.ball().pos().y ) > 4.0 )
        {
            //body_angle = wm.ball().angleFromSelf() + ( 90.0 + 20.0 );
            body_angle = wm.ball().angleFromSelf() + 90.0;
            if ( wm.ball().pos().x > wm.self().pos().x + 15.0 )
            {
                if ( body_angle.abs() > 90.0 )
                {
                    body_angle += 180.0;
                }
            }
            else
            {
                if ( body_angle.abs() < 90.0 )
                {
                    body_angle += 180.0;
                }
            }
        }

        Body_TurnToAngle( body_angle ).execute( agent );
    }

    if ( wm.interceptTable()->opponentReachCycle() <= 3
         || wm.ball().distFromSelf() < 10.0 )
    {
        agent->setNeckAction( new Neck_TurnToBall() );
    }
    else
    {
        agent->setNeckAction( new Neck_TurnToBallOrScan() );
    }
}
