// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_center_half_cross_block_move.h"

#include "strategy.h"
#include "defense_system.h"
#include "mark_analyzer.h"

#include "bhv_get_ball.h"
#include "bhv_danger_area_tackle.h"
#include "neck_check_ball_owner.h"
#include "neck_default_intercept_neck.h"
#include "neck_offensive_intercept_neck.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_intercept.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/neck_turn_to_player_or_scan.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>

#include <limits>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterHalfCrossBlockMove::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ROLE,
                  __FILE__": Bhv_CenterHalfCrossBlockMove" );
    //
    // tackle
    //
    if ( Bhv_DangerAreaTackle().execute( agent ) )
    {
        agent->debugClient().addMessage( "CH:CrossBlock:Tackle" );
        dlog.addText( Logger::ROLE,
                      __FILE__":  tackle" );
        return true;
    }

    //
    // intercept
    //
    if ( doIntercept( agent ) )
    {
        return true;
    }

    //
    // get ball
    //
    if ( doGetBall( agent ) )
    {
        return true;
    }

    //
    // mark opponent attacker
    //
    if ( doMark( agent ) )
    {
        return true;
    }

    //
    // normal move
    //
    doNormalMove( agent );

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterHalfCrossBlockMove::doIntercept( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( self_min <= opp_min + 1
         && self_min <= mate_min + 1
         && ! wm.existKickableTeammate() )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doIntercept) performed" );
        agent->debugClient().addMessage( "CH:CrossBlock:Intercept" );
        Body_Intercept().execute( agent );
        if ( opp_min >= self_min + 3 )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": (doIntercept) offensive turn_neck" );
            agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        }
        else
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": (doIntercept) default turn_neck" );
            agent->setNeckAction( new Neck_DefaultInterceptNeck
                                  ( new Neck_TurnToBallOrScan() ) );
        }
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterHalfCrossBlockMove::doGetBall( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    int opp_min = wm.interceptTable()->opponentReachCycle();
    const PlayerObject * fastest_opp = wm.interceptTable()->fastestOpponent();
    Vector2D opp_trap_pos = wm.ball().inertiaPoint( opp_min );

    const Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );

    if ( wm.existKickableTeammate()
         || ! fastest_opp
         || opp_trap_pos.x > -30.0
         || opp_trap_pos.dist( home_pos ) > 7.0
         || opp_trap_pos.absY() > 13.0
         )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doGetBall) no get ball situation" );
        return false;
    }

    //
    // search other blocker
    //
    bool exist_blocker = false;

    const double my_dist = wm.self().pos().dist( opp_trap_pos );
    const PlayerPtrCont::const_iterator end = wm.teammatesFromBall().end();
    for ( PlayerPtrCont::const_iterator p = wm.teammatesFromBall().begin();
          p != end;
          ++p )
    {
        if ( (*p)->goalie() ) continue;
        if ( (*p)->isGhost() ) continue;
        if ( (*p)->posCount() >= 3 ) continue;
        if ( (*p)->pos().x > fastest_opp->pos().x ) continue;
        if ( (*p)->pos().x > opp_trap_pos.x ) continue;
        if ( (*p)->pos().dist( opp_trap_pos ) > my_dist ) continue;

        dlog.addText( Logger::ROLE,
                      __FILE__": (doGetBall) exist other blocker %d (%.1f %.1f)",
                      (*p)->unum(),
                      (*p)->pos().x, (*p)->pos().y );
        exist_blocker = true;
        break;
    }

    //
    // try intercept
    //
    if ( exist_blocker )
    {
        int self_min = wm.interceptTable()->selfReachCycle();
        Vector2D self_trap_pos = wm.ball().inertiaPoint( self_min );
        if ( self_min <= 10
             && self_trap_pos.dist( home_pos ) < 10.0 )
        {
            agent->debugClient().addMessage( "CH:CrossBlock:GetBall:Intercept" );
            Body_Intercept().execute( agent );
            if ( opp_min >= self_min + 3 )
            {
                dlog.addText( Logger::ROLE,
                              __FILE__": (doIntercept) offensive turn_neck" );
                agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
            }
            else
            {
                dlog.addText( Logger::ROLE,
                              __FILE__": (doIntercept) default turn_neck" );
                agent->setNeckAction( new Neck_DefaultInterceptNeck
                                      ( new Neck_TurnToBallOrScan() ) );
            }
            return true;
        }

        return false;
    }

    //
    // try get ball
    //
    dlog.addText( Logger::ROLE,
                  __FILE__": (doGetBall) try" );
    double max_x = -34.0;
    Rect2D bounding_rect( Vector2D( -60.0, home_pos.y - 6.0 ),
                          Vector2D( max_x, home_pos.y + 6.0 ) );
    if ( Bhv_GetBall( bounding_rect ).execute( agent ) )
    {
        agent->debugClient().addMessage( "CH:CrossBlock:GetBall" );
        dlog.addText( Logger::ROLE,
                      __FILE__": (doGetBall) performed" );
        return true;
    }

    return false;
}


/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterHalfCrossBlockMove::doMark( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();
    const MarkAnalyzer & mark_analyzer = MarkAnalyzer::i();

    Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );

    const double goal_half_width = ServerParam::i().goalHalfWidth();
    const Vector2D center_pos( -47.0,
                               goal_half_width * sign( wm.ball().pos().y ) );

    const PlayerObject * target_opponent = static_cast< const PlayerObject * >( 0 );
    double min_dist2 = std::numeric_limits< double >::max();

    const PlayerPtrCont::const_iterator end = wm.opponentsFromBall().end();
    for ( PlayerPtrCont::const_iterator p = wm.opponentsFromBall().begin();
          p != end;
          ++p )
    {
        if ( (*p)->posCount() >= 10 ) continue;
        if ( (*p)->pos().x > -36.0 ) continue;
        if ( (*p)->pos().absY() > 17.0 ) continue;
        if ( std::fabs( (*p)->pos().y - wm.ball().pos().y ) > 30.0 ) continue;
        if ( std::fabs( (*p)->pos().y - home_pos.y ) > 5.0 ) continue;
        if ( (*p)->pos().x < home_pos.x - 5.0 ) continue;
        if ( (*p)->pos().x < wm.ourDefenseLineX() ) continue;
        //if ( (*p)->pos().y * wm.ball().pos().y < 0.0 ) continue;
        if ( wm.ball().pos().absY() > goal_half_width
             && (*p)->pos().absY() > wm.ball().pos().absY() )
        {
            continue;
        }

#if 1
        const AbstractPlayerObject * marker = mark_analyzer.getMarkerOf( *p );
        if ( marker
             && marker->unum() != wm.self().unum()
             && marker->pos().dist2( (*p)->pos() ) < std::pow( 3.0, 2 ) )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": (doMark) opponent=%d. marker=%d marked(1)",
                          (*p)->unum(), marker->unum() );
            continue;
        }

        double dist = 65535.0;
        marker = wm.getTeammateNearestTo( (*p)->pos(), 5, &dist );
        if ( marker
             && marker->unum() != wm.self().unum()
             && dist < 2.0
             && (*p)->distFromBall() > marker->distFromBall() )
        {
            const AbstractPlayerObject * mark_target = mark_analyzer.getTargetOf( marker->unum() );
            if ( ! mark_target
                 || mark_target->unum() == (*p)->unum() )
            {
                dlog.addText( Logger::ROLE,
                              __FILE__": (doMark) opponent=%d. marker=%d marked(2)",
                              (*p)->unum(), marker->unum() );
                continue;
            }
        }
#endif

        double d2 = (*p)->pos().dist2( center_pos );
        if ( d2 < min_dist2 )
        {
            min_dist2 = d2;
            target_opponent = *p;
        }
    }

    if ( ! target_opponent )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doMark) no target opponent" );
        return false;
    }

    //
    // TODO:
    // check if target_opponent is already marked or not.
    //


    Vector2D block_point = target_opponent->pos();
    block_point.x -= 0.3;
    if ( block_point.y > wm.ball().pos().y ) block_point.y -= 0.4;
    if ( block_point.y < wm.ball().pos().y ) block_point.y += 0.4;

    dlog.addText( Logger::ROLE,
                  __FILE__": (doMark) target_opponent=%d(%.2f %.1f)"
                  " mark_point=(%.2f %.2f)",
                  target_opponent->unum(),
                  target_opponent->pos().x, target_opponent->pos().y,
                  block_point.x, block_point.y );
    agent->debugClient().addMessage( "CH:CrossBlock:Mark" );

    doGoToPoint( agent,
                 block_point,
                 0.4, // dist thr
                 ServerParam::i().maxDashPower(),
                 15.0 ); // dir thr

#if 0
    agent->setNeckAction( new Neck_CheckBallOwner() );
#else
    if ( wm.ball().posCount() >= 3 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doMark) check ball" );
        agent->debugClient().addMessage( ":NeckBall" );

        agent->setNeckAction( new Neck_TurnToBallOrScan() );
    }
    else
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doTurnNeck) check mark target" );
        agent->debugClient().addMessage( ":NeckMark" );

        agent->setNeckAction( new Neck_TurnToPlayerOrScan( target_opponent ) );
    }
#endif

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_CenterHalfCrossBlockMove::doNormalMove( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    Vector2D target_point = Strategy::i().getPosition( wm.self().unum() );

    double dash_power = DefenseSystem::get_defender_dash_power( wm, target_point );
    double dist_thr = std::fabs( wm.ball().pos().x - wm.self().pos().x ) * 0.1;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    doGoToPoint( agent, target_point, dist_thr, dash_power, 12.0 );

    agent->setNeckAction( new Neck_CheckBallOwner() );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_CenterHalfCrossBlockMove::doGoToPoint( PlayerAgent * agent,
                                           const Vector2D & target_point,
                                           const double & dist_thr,
                                           const double & dash_power,
                                           const double & dir_thr )
{
    const WorldModel & wm = agent->world();

    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    if ( Body_GoToPoint( target_point, dist_thr, dash_power,
                         -1.0, // dash speed
                         1, // 1 step
                         true, // save recovery
                         dir_thr
                         ).execute( agent ) )
    {
        agent->debugClient().addMessage( "CH:CrossBlock:Go%.1f", dash_power );
        dlog.addText( Logger::ROLE,
                      __FILE__": (doGoToPoint) (%.1f %.1f) dash_power=%.1f dist_thr=%.2f",
                      target_point.x, target_point.y,
                      dash_power,
                      dist_thr );
        return;
    }

    // already there

    Vector2D ball_next = wm.ball().pos() + wm.ball().vel();
    Vector2D my_final = wm.self().inertiaFinalPoint();
    AngleDeg ball_angle = ( ball_next - my_final ).th();

    AngleDeg target_angle;
    if ( ball_next.x < -30.0 )
    {
        target_angle = ball_angle + 90.0;
        if ( ball_next.x < -45.0 )
        {
            if ( target_angle.degree() < 0.0 )
            {
                target_angle += 180.0;
            }
        }
        else
        {
            if ( target_angle.degree() > 0.0 )
            {
                target_angle += 180.0;
            }
        }
    }
    else
    {
        target_angle = ball_angle + 90.0;
        if ( ball_next.x > my_final.x + 15.0 )
        {
            if ( target_angle.abs() > 90.0 )
            {
                target_angle += 180.0;
            }
        }
        else
        {
            if ( target_angle.abs() < 90.0 )
            {
                target_angle += 180.0;
            }
        }
    }

    Body_TurnToAngle( target_angle ).execute( agent );

    agent->debugClient().addMessage( "CH:CrossBlock:TurnTo%.0f",
                                     target_angle.degree() );
    dlog.addText( Logger::ROLE,
                  __FILE__": (doGoToPoint) turn to angle=%.1f",
                  target_angle.degree() );
}
