// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_center_half_danger_move.h"

#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/intercept_table.h>

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>

#include <rcsc/action/body_intercept.h>

#include "bhv_basic_tackle.h"
#include "bhv_block_dribble.h"
#include "bhv_get_ball.h"
#include "bhv_danger_area_tackle.h"
#include "neck_check_ball_owner.h"
#include "neck_default_intercept_neck.h"
#include "neck_offensive_intercept_neck.h"

#include "strategy.h"

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterHalfDangerMove::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::ROLE,
                  __FILE__": Bhv_CenterHalfDangerMove" );
    //
    // tackle
    //
    if ( Bhv_BasicTackle( 0.79, 75.0 ).execute( agent ) )
    {
        return true;
    }

    if ( doIntercept( agent ) )
    {
        return true;
    }

    if ( doGetBall( agent ) )
    {
        return true;
    }

#if 0
    if ( doBlockDribble( agent ) )
    {
        return true;
    }
#endif

    doDefaultMove( agent );

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterHalfDangerMove::doIntercept( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    if ( wm.existKickableTeammate() )
    {
        return false;
    }

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( self_min > mate_min + 2 )
    {
        return false;
    }


    if ( self_min <= 3
         || self_min <= opp_min + 3
         || ( wm.ball().vel().r() * std::pow( rcsc::ServerParam::i().ballDecay(), opp_min ) < 0.8
              && opp_min >= 3 && self_min <= opp_min + 5 )
         )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": intercept" );
        Body_Intercept().execute( agent );

        if ( opp_min >= self_min + 3 )
        {
            agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        }
        else
        {
            agent->setNeckAction( new Neck_DefaultInterceptNeck
                                  ( new Neck_TurnToBallOrScan() ) );
        }

        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterHalfDangerMove::doGetBall( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );
    const PositionType position_type = Strategy::i().getPositionType( wm.self().unum() );

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( wm.existKickableTeammate()
         || mate_min < opp_min
         || self_min <= opp_min )
    {
        return false;
    }


#if 1
    // 2009-07-04

    //
    // check other blocker
    //
    {
        const Vector2D opp_trap_pos = wm.ball().inertiaPoint( opp_min );
        const Vector2D mid_point = ( opp_trap_pos + Vector2D( -50.0, 0.0 ) ) * 0.5;
        const PlayerObject * other_blocker = static_cast< PlayerObject * >( 0 );
        double my_dist = wm.self().pos().dist( mid_point );
        double min_dist = my_dist;

        const PlayerPtrCont::const_iterator t_end = wm.teammatesFromSelf().end();
        for ( PlayerPtrCont::const_iterator t = wm.teammatesFromSelf().begin();
              t != t_end;
              ++t )
        {
            if ( (*t)->posCount() >= 10 ) continue;
            if ( (*t)->pos().x > opp_trap_pos.x - 1.0 ) continue;

            double d = (*t)->pos().dist( mid_point );
            if ( d < min_dist )
            {
                min_dist = d;
                other_blocker = *t;
            }
        }

        if ( my_dist > min_dist * 1.2 )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": (doGetBall) exist other blocker=%d (%.1f %.1f)",
                          other_blocker->unum(),
                          other_blocker->pos().x, other_blocker->pos().y );
            return false;
        }
    }
#endif

    //
    // get ball
    //
    Rect2D bounding_rect;
    if ( position_type == Position_Left )
    {
        bounding_rect
            = Rect2D( Vector2D( home_pos.x - 10.0, home_pos.y - 15.0 ),
                      Vector2D( home_pos.x + 6.0, home_pos.y + 9.5 ) );
    }
    else if ( position_type == Position_Right )
    {
        bounding_rect
            = Rect2D( Vector2D( home_pos.x - 10.0, home_pos.y - 9.5 ),
                      Vector2D( home_pos.x + 6.0, home_pos.y + 15.0 ) );
    }
    else
    {
        bounding_rect
            = Rect2D( Vector2D( home_pos.x - 10.0, home_pos.y - 15.0 ),
                      Vector2D( home_pos.x + 6.0, home_pos.y + 15.0 ) );
    }

    dlog.addText( Logger::ROLE,
                  __FILE__": try GetBall. rect=(%.1f %.1f)(%.1f %.1f)",
                  bounding_rect.left(), bounding_rect.top(),
                  bounding_rect.right(), bounding_rect.bottom() );
    if ( Bhv_GetBall( bounding_rect ).execute( agent ) )
    {
        agent->debugClient().addMessage( "CH:GetBall" );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterHalfDangerMove::doBlockDribble( rcsc::PlayerAgent * agent )
{
    //
    // block dribble or intercept
    //

    const WorldModel & wm = agent->world();

    const Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    const Vector2D self_reach_point = wm.ball().inertiaPoint( self_min );
    const Vector2D opp_reach_point = wm.ball().inertiaPoint( opp_min );

    if ( ! wm.existKickableTeammate()
         && self_reach_point.dist( home_pos ) < 13.0
         && ( self_min < mate_min
              || ( self_min <= 3 && wm.ball().pos().dist2( home_pos ) < 10.0*10.0 )
              || ( self_min <= 5 && wm.ball().pos().dist2( home_pos ) < 8.0*8.0 ) )
         )
    {
        if ( opp_min < mate_min - 1
             && opp_min < self_min - 2 )
        {
            if ( Bhv_BlockDribble().execute( agent ) )
            {
                dlog.addText( Logger::ROLE,
                              __FILE__": BlockDribble" );
                agent->debugClient().addMessage( "CH:BlockDrib" );
                return true;
            }
        }

        dlog.addText( Logger::ROLE,
                      __FILE__": intercept line=%d", __LINE__ );
        Body_Intercept().execute( agent );
        agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_CenterHalfDangerMove::doDefaultMove( rcsc::PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );

    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    Vector2D trap_pos = wm.ball().inertiaPoint( std::min( mate_min, opp_min ) );
    double dash_power = ServerParam::i().maxDashPower();

    //
    // decide dash_power
    //
    if ( wm.self().staminaModel().capacityIsEmpty() )
    {
        dash_power = std::min( ServerParam::i().maxDashPower(),
                               wm.self().stamina() + wm.self().playerType().extraStamina() );
    }
    else if ( wm.existKickableTeammate() )
    {
        if ( wm.self().pos().dist( trap_pos ) > 10.0 )
        {
            if ( wm.self().stamina() < ServerParam::i().staminaMax() * 0.8 )
            {
                dash_power = ( wm.self().playerType().staminaIncMax()
                               * wm.self().recovery() );
                dlog.addText( Logger::ROLE,
                              __FILE__": dash_power, teammate kickable, stamina save" );
            }
        }
    }
    else if ( trap_pos.x < wm.self().pos().x ) // ball is behind
    {
        dash_power *= 0.9;
        dlog.addText( Logger::ROLE,
                      __FILE__": dash_power, trap_pos is behind. trap_pos=(%.1f %.1f)",
                      trap_pos.x, trap_pos.y );
    }
    else if ( wm.self().stamina() > ServerParam::i().staminaMax() * 0.8 )
    {
        dash_power *= 0.8;
        dlog.addText( Logger::ROLE,
                      __FILE__": dash_power, enough stamina" );
    }
    else
    {
        dash_power = ( wm.self().playerType().staminaIncMax()
                       * wm.self().recovery()
                       * 0.9 );
        dlog.addText( Logger::ROLE,
                      __FILE__": dash_power, default" );
    }

    // save recovery
    dash_power = wm.self().getSafetyDashPower( dash_power );

    //
    // register action
    //

    double dist_thr = wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    dlog.addText( Logger::ROLE,
                  __FILE__": go to home (%.1f %.1f) dist_thr=%.3f. dash_power=%.1f",
                  home_pos.x, home_pos.y,
                  dist_thr,
                  dash_power );

    agent->debugClient().setTarget( home_pos );
    agent->debugClient().addCircle( home_pos, dist_thr );

    if ( Body_GoToPoint( home_pos, dist_thr, dash_power,
                         -1.0, // dash speed
                         1 // 1 step
                         ).execute( agent ) )
    {
        agent->debugClient().addMessage( "CH:DefMove:Go%.0f", dash_power );
    }
    else
    {
        AngleDeg body_angle = 0.0;
        if ( wm.ball().angleFromSelf().abs() > 80.0 )
        {
            body_angle = ( wm.ball().pos().y > wm.self().pos().y
                           ? 90.0
                           : -90.0 );
        }
        Body_TurnToAngle( body_angle ).execute( agent );
        agent->debugClient().addMessage( "CH:DefMove:Turn%.0f",
                                         body_angle.degree() );
    }

    agent->setNeckAction( new Neck_TurnToBallOrScan() );

    return true;
}
