// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_danger_area_tackle.h"

#include "tackle_generator.h"

#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/neck_turn_to_point.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/line_2d.h>
#include <rcsc/geom/ray_2d.h>

// #define DEBUG_PRINT

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_DangerAreaTackle::execute( rcsc::PlayerAgent * agent )
{
    dlog.addText( Logger::TEAM,
                  __FILE__": Bhv_DangerAreaTackle" );

    if ( clearGoal( agent ) )
    {
        return true;
    }

    const ServerParam & SP = ServerParam::i();
    const WorldModel & wm = agent->world();

    bool use_foul = false;
    double tackle_prob = wm.self().tackleProbability();

    dlog.addText( Logger::TEAM,
                  __FILE__": %s tackleProb=%f foulProb=%f",
                  wm.self().card() == NO_CARD ? "NoCard" : "YellowCard?",
                  wm.self().tackleProbability(),
                  wm.self().foulProbability() );

    if ( wm.self().card() == NO_CARD
         && ( wm.ball().pos().x > SP.ourPenaltyAreaLineX() + 0.5
              || wm.ball().pos().absY() > SP.penaltyAreaHalfWidth() + 0.5 )
         && tackle_prob < wm.self().foulProbability() )
    {
        tackle_prob = wm.self().foulProbability();
        use_foul = true;
    }

    if ( tackle_prob < M_min_probability )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": failed. low tackle_prob=%.2f < %.2f",
                      tackle_prob,
                      M_min_probability );
        return false;
    }

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    const Vector2D self_reach_point = wm.ball().inertiaPoint( self_min );


    //
    // check where the ball shall be gone without tackle
    //

    bool ball_will_be_in_our_goal = false;

    if ( self_reach_point.x < -SP.pitchHalfLength() )
    {
        const Ray2D ball_ray( wm.ball().pos(), wm.ball().vel().th() );
        const Line2D goal_line( Vector2D( -SP.pitchHalfLength(), 10.0 ),
                                Vector2D( -SP.pitchHalfLength(), -10.0 ) );

        const Vector2D intersect = ball_ray.intersection( goal_line );
        if ( intersect.isValid()
             && intersect.absY() < SP.goalHalfWidth() + 1.0 )
        {
            ball_will_be_in_our_goal = true;

            dlog.addText( Logger::TEAM,
                          __FILE__": ball will be in our goal. intersect=(%.2f %.2f)",
                          intersect.x, intersect.y );
        }
    }

    if ( wm.existKickableOpponent()
         || ball_will_be_in_our_goal
         || ( opp_min < self_min
              && opp_min < mate_min ) )
    {
        // try tackle
    }
    else
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": failed. not necessary." );
        return false;
    }

    if ( agent->config().version() < 12.0 )
    {
        // v11 or older
        return executeOld( agent );
    }
    else if( agent->config().version() < 14.0 )
    {
        // v12-13
        return executeV12( agent );
    }

    // v14+
    return executeV14( agent, use_foul );
}


/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_DangerAreaTackle::clearGoal( PlayerAgent * agent )
{
    const ServerParam & SP = ServerParam::i();
    const WorldModel & wm = agent->world();

    bool use_foul = false;
    double tackle_prob = wm.self().tackleProbability();

    if ( agent->config().version() >= 14.0
         && wm.self().card() == NO_CARD
         && ( wm.ball().pos().x > SP.ourPenaltyAreaLineX() + 0.5
              || wm.ball().pos().absY() > SP.penaltyAreaHalfWidth() + 0.5 )
         && tackle_prob < wm.self().foulProbability() )
    {
        tackle_prob = wm.self().foulProbability();
        use_foul = true;
    }

    if ( tackle_prob <= 1.0e-3 )
    {
        return false;
    }

    const int self_min = wm.interceptTable()->selfReachCycle();

    const Vector2D self_trap_pos = wm.ball().inertiaPoint( self_min );
    if ( self_trap_pos.x > - SP.pitchHalfLength() + 0.5 )
    {
        return false;
    }

    //
    // cannot intercept the ball in the field
    //

    dlog.addText( Logger::TEAM,
                  __FILE__": my trap pos(%.1f %.1f) < pitch.x",
                  self_trap_pos.x, self_trap_pos.y );

    const Ray2D ball_ray( wm.ball().pos(), wm.ball().vel().th() );
    const Line2D goal_line( Vector2D( - SP.pitchHalfLength(), 10.0 ),
                            Vector2D( - SP.pitchHalfLength(), -10.0 ) );
    const Vector2D intersect =  ball_ray.intersection( goal_line );
    if ( ! intersect.isValid()
         || intersect.absY() > SP.goalHalfWidth() + 0.5 )
    {
        return false;
    }

    //
    // ball is moving to our goal
    //

    dlog.addText( Logger::TEAM,
                  __FILE__": ball is moving to our goal" );


    if ( agent->config().version() < 12.0 )
    {
        double tackle_power = ( wm.self().body().abs() > 90.0
                                ? SP.maxTacklePower()
                                : - SP.maxBackTacklePower() );

        dlog.addText( Logger::TEAM,
                      __FILE__": clear goal old (%s)",
                      use_foul ? "foul" : "tackle" );
        agent->debugClient().addMessage( "%sClearOld%.0f",
                                         use_foul ? "foul" : "tackle",
                                         tackle_power );
        agent->doTackle( tackle_power, use_foul );
        agent->setNeckAction( new Neck_TurnToBallOrScan() );
        return true;
    }

    //
    // search best angle
    //
#if 1
    TackleGenerator::instance().generate( wm );

    const TackleGenerator::TackleResult & result
        = TackleGenerator::instance().bestResult( wm );

    Vector2D ball_next = wm.ball().pos() + result.ball_vel_;

    dlog.addText( Logger::TEAM,
                  __FILE__": (clearGoal) %s angle=%.0f resultVel=(%.2f %.2f) ballNext=(%.2f %.2f)",
                  use_foul ? "foul" : "tackle",
                  result.tackle_angle_.degree(),
                  result.ball_vel_.x, result.ball_vel_.y,
                  ball_next.x, ball_next.y );
    agent->debugClient().addMessage( "clear%s%.0f",
                                     use_foul ? "Foul" : "Tackle",
                                     result.tackle_angle_.degree() );

    double tackle_dir = ( result.tackle_angle_ - wm.self().body() ).degree();

    agent->doTackle( tackle_dir, use_foul );
    agent->setNeckAction( new Neck_TurnToPoint( ball_next ) );

    return true;

#else
    const double goal_half_width = ServerParam::i().goalHalfWidth();

    const Vector2D goal_center = ServerParam::i().ourTeamGoalPos();
    const Vector2D goal_left_post( goal_center.x, +goal_half_width );
    const Vector2D goal_right_post( goal_center.x, -goal_half_width );

    const Line2D line_c( Vector2D( -SP.pitchHalfLength(), 0.0 ),
                         Vector2D( 0.0, 0.0 ) );
    const Line2D line_l( Vector2D( -SP.pitchHalfLength(), -SP.goalHalfWidth() ),
                         Vector2D( 0.0, -SP.goalHalfWidth() ) );
    const Line2D line_r( Vector2D( -SP.pitchHalfLength(), -SP.goalHalfWidth() ),
                         Vector2D( 0.0, -SP.goalHalfWidth() ) );

    const AngleDeg ball_rel_angle = wm.ball().angleFromSelf() - wm.self().body();
    const double tackle_rate = ( SP.tacklePowerRate()
                                 * ( 1.0 - 0.5 * ( ball_rel_angle.abs() / 180.0 ) ) );

    AngleDeg best_angle = 0.0;
    double max_speed = -1.0;
    AngleDeg danger_clear_angle = 0.0;
    double shooting_min_speed = 1000.0;

    for ( double a = -180.0; a < 180.0; a += 10.0 )
    {
        AngleDeg target_rel_angle = a - wm.self().body().degree();

        double eff_power = SP.maxBackTacklePower()
            + ( ( SP.maxTacklePower() - SP.maxBackTacklePower() )
                * ( 1.0 - target_rel_angle.abs() / 180.0 ) );
        eff_power *= tackle_rate;

        Vector2D vel = wm.ball().vel()
            + Vector2D::polar2vector( eff_power, AngleDeg( a ) );
        double speed = vel.r();
        if ( speed > ServerParam::i().ballSpeedMax() )
        {
            vel *= ( ServerParam::i().ballSpeedMax() / speed );
            speed = ServerParam::i().ballSpeedMax();
        }

        AngleDeg vel_angle = vel.th();

        const bool is_shoot_ball = ( ( ( goal_left_post - wm.ball().pos() ).th()
                                       - vel_angle ).degree() < 0
                                     && ( ( goal_right_post - wm.ball().pos() ).th()
                                          - vel_angle ).degree() > 0 );
        if ( is_shoot_ball )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          __FILE__": clearGoal() speed=%.3f angle=%.1f. vel_angle=%.1f is dangerouns",
                          speed, a,
                          vel_angle.degree() );
#endif
            if ( shooting_min_speed > speed )
            {
#ifdef DEBUG_PRINT
                dlog.addText( Logger::TEAM,
                              __FILE__": __ updated shooting_min_speed=%.3f -> %.3f",
                              shooting_min_speed, speed );
#endif
                shooting_min_speed = speed;
                danger_clear_angle = target_rel_angle + wm.self().body();
            }
            continue;
        }


        int n_intersects = 0;
        if ( ball_ray.intersection( line_c ).isValid() ) ++n_intersects;
        if ( ball_ray.intersection( line_l ).isValid() ) ++n_intersects;
        if ( ball_ray.intersection( line_r ).isValid() ) ++n_intersects;

        if ( n_intersects == 3 )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          "__ angle=%.1f vel=(%.1f %.1f)"
                          " 3 intersects with v_lines. angle is dangerous.",
                          a, vel.x, vel.y );
#endif
            speed -= 2.0;
        }
        else if ( n_intersects == 2
                  && wm.ball().pos().absY() > 3.0 )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          "__ executeV12() angle=%.1f vel=(%.1f %.1f)"
                          " 2 intersects with v_lines. angle is dangerous.",
                          a, vel.x, vel.y );
#endif
            speed -= 2.0;
        }

#ifdef DEBUG_PRINT
        Vector2D ball_end_point = inertia_final_point( wm.ball().pos(),
                                                       vel,
                                                       SP.ballDecay() );
        dlog.addLine( Logger::TEAM,
                      wm.ball().pos(),
                      ball_end_point,
                      "#0000ff" );
        char buf[16];
        snprintf( buf, 16, "%.3f", speed );
        dlog.addMessage( Logger::CLEAR,
                         ball_end_point, buf, "#ffffff" );
#endif

        if ( speed > max_speed )
        {
            max_speed = speed;
            best_angle = target_rel_angle + wm.self().body();
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          __FILE__": clearGoal() update. angle=%.1f vel_angle=%.1f speed=%.2f",
                          a,
                          vel_angle.degree(),
                          speed );
#endif
        }
        else
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          __FILE__": clearGoal() no_update. angle=%.1f vel_angle=%.1f speed=%.2f",
                          a,
                          vel_angle.degree(),
                          speed );
#endif
        }
    }

    //
    // never accelerate the ball
    //

    if ( max_speed > 0.0 )
    {
        double tackle_dir = ( best_angle - wm.self().body() ).degree();
        dlog.addText( Logger::TEAM,
                      __FILE__": clear goal. (%s) best_angle=%.1f max_speed=%.3f",
                      use_foul ? "foul" : "tackle",
                      best_angle.degree(),
                      max_speed );
        agent->debugClient().addMessage( "%sClear%.0f",
                                         use_foul ? "foul" : "tackle",
                                         best_angle.degree() );
        agent->doTackle( tackle_dir, use_foul );
        agent->setNeckAction( new Neck_TurnToBallOrScan() );
        return true;
    }

    if ( shooting_min_speed < 0.5 )
    {
        double tackle_dir = ( danger_clear_angle - wm.self().body() ).degree();
        dlog.addText( Logger::TEAM,
                      __FILE__": clear goal. (%s) danger_clear_angle=%.1f shooting_min_speed=%.3f",
                      use_foul ? "foul" : "tackle",
                      danger_clear_angle.degree(),
                      shooting_min_speed );
        agent->debugClient().addMessage( "%sClearDanger%.0f",
                                         use_foul ? "foul" : "tackle",
                                         danger_clear_angle.degree() );
        agent->doTackle( tackle_dir, use_foul );
        agent->setNeckAction( new Neck_TurnToBallOrScan() );
        return true;
    }

    dlog.addText( Logger::TEAM,
                  __FILE__": failed clearGoal. max_speed=%.3f  shooting_min_speed=%.3f",
                  max_speed, shooting_min_speed );
    return false;
#endif
}


/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_DangerAreaTackle::executeOld( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    if ( wm.self().pos().absY() > ServerParam::i().goalHalfWidth() + 5.0 )
    {
        // out of goal
        double tackle_power = ServerParam::i().maxTacklePower();
        if ( wm.self().body().abs() < 10.0 )
        {

        }
        else if ( wm.self().body().abs() > 170.0 )
        {
            tackle_power = - ServerParam::i().maxBackTacklePower();
            if ( tackle_power >= -1.0 )
            {
                return false;
            }
        }
        else if ( wm.self().body().degree() * wm.self().pos().y < 0.0 )
        {
            tackle_power = - ServerParam::i().maxBackTacklePower();
            if ( tackle_power >= -1.0 )
            {
                return false;
            }
        }

        if ( std::fabs( tackle_power ) < 1.0 )
        {
            return false;
        }


        dlog.addText( Logger::TEAM,
                      __FILE__": out of goal width. " );
        agent->debugClient().addMessage( "Tackle(1)" );
        agent->doTackle( tackle_power );
        agent->setNeckAction( new Neck_TurnToBallOrScan() );
        return true;
    }
    else
    {
        // within goal width
        int power_sign = 0;
        double abs_body = wm.self().body().abs();

        if ( abs_body < 70.0 ) power_sign = 1;
        if ( abs_body > 110.0 ) power_sign = -1;
        if ( power_sign == 0 )
        {
            power_sign = ( wm.self().body().degree() > 0.0
                           ? 1
                           : -1 );
            if ( wm.ball().pos().y < 0.0 )
            {
                power_sign *= -1;
            }
        }

        double tackle_power = ( power_sign >= 0
                                ? ServerParam::i().maxTacklePower()
                                : - ServerParam::i().maxBackTacklePower() );
        if ( std::fabs( tackle_power ) < 1.0 )
        {
            return false;
        }

        if ( power_sign != 0 )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": power_sign=%d",
                          power_sign );
            agent->debugClient().addMessage( "tackle(%d) ",
                                             power_sign );
            agent->doTackle( tackle_power );
            agent->setNeckAction( new Neck_TurnToBallOrScan() );
            return true;
        }
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_DangerAreaTackle::executeV12( PlayerAgent * agent )
{
    const ServerParam & SP = ServerParam::i();
    const WorldModel & wm = agent->world();

    const Vector2D goal( - SP.pitchHalfLength(), 0.0 );
    const Vector2D virtual_accel = ( wm.existKickableOpponent()
                                     ? ( goal - wm.ball().pos() ).setLengthVector( 1.0 )
                                     : Vector2D( 0.0, 0.0 ) );
    const Line2D goal_line( Vector2D( - SP.pitchHalfLength(), 10.0 ),
                            Vector2D( - SP.pitchHalfLength(), -10.0 ) );
    const Line2D line_c( Vector2D( -SP.pitchHalfLength(), 0.0 ),
                         Vector2D( 0.0, 0.0 ) );
    const Line2D line_l( Vector2D( -SP.pitchHalfLength(), -SP.goalHalfWidth() ),
                         Vector2D( 0.0, -SP.goalHalfWidth() ) );
    const Line2D line_r( Vector2D( -SP.pitchHalfLength(), -SP.goalHalfWidth() ),
                         Vector2D( 0.0, -SP.goalHalfWidth() ) );

    const AngleDeg ball_rel_angle
        = wm.ball().angleFromSelf() - wm.self().body();
    const double tackle_rate
        = ( SP.tacklePowerRate()
            * ( 1.0 - 0.5 * ( ball_rel_angle.abs() / 180.0 ) ) );

    AngleDeg best_angle = 0.0;
    double max_speed = -1.0;

    for ( double a = -180.0; a < 180.0; a += 10.0 )
    {
        AngleDeg rel_angle = a - wm.self().body().degree();

        double eff_power = ( SP.maxBackTacklePower()
                             + ( ( SP.maxTacklePower() - SP.maxBackTacklePower() )
                                 * ( 1.0 - rel_angle.abs() / 180.0 ) ) );
        eff_power *= tackle_rate;

        Vector2D vel = ( wm.ball().vel()
                         + Vector2D::polar2vector( eff_power, AngleDeg( a ) ) );
        vel += virtual_accel;

        double speed = vel.r();
        if ( speed > SP.ballSpeedMax() )
        {
            vel *= ( SP.ballSpeedMax() / speed );
            speed = SP.ballSpeedMax();
        }

        const Ray2D ball_ray( wm.ball().pos(), vel.th() );
        const Vector2D intersect =  ball_ray.intersection( goal_line );
        if ( intersect.isValid()
             && intersect.absY() < SP.goalHalfWidth() + 5.0 )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          __FILE__": executeV12() angle=%.1f vel=(%.1f %.1f)"
                          " intersect is dangerous.",
                          a, vel.x, vel.y );
#endif
            continue;
        }

        const Vector2D ball_next = wm.ball().pos() + vel;

        bool maybe_opponent_get_ball = false;

        const PlayerPtrCont::const_iterator o_end = wm.opponentsFromBall().end();
        for ( PlayerPtrCont::const_iterator o = wm.opponentsFromBall().begin();
              o != o_end;
              ++o )
        {
            if ( (*o)->posCount() > 10 ) continue;
            if ( (*o)->isGhost() ) continue;
            if ( (*o)->isTackling() ) continue;
            if ( (*o)->distFromBall() > 6.0 ) break;

            Vector2D opp_pos = (*o)->pos() + (*o)->vel();
            if ( opp_pos.dist( ball_next ) < 1.0 )
            {
                maybe_opponent_get_ball = true;
                break;
            }
        }

        if ( maybe_opponent_get_ball )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          __FILE__": executeV12() angle=%.1f vel=(%.1f %.1f)"
                          " maybe opponent get ball.",
                          a, vel.x, vel.y );
#endif
            continue;
        }


        int n_intersects = 0;
        if ( ball_ray.intersection( line_c ).isValid() ) ++n_intersects;
        if ( ball_ray.intersection( line_l ).isValid() ) ++n_intersects;
        if ( ball_ray.intersection( line_r ).isValid() ) ++n_intersects;

        if ( n_intersects == 3 )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          "__ angle=%.1f vel=(%.1f %.1f)"
                          " 3 intersects with v_lines. angle is dangerous.",
                          a, vel.x, vel.y );
#endif
            speed -=2.0;
        }
        else if ( n_intersects == 2
                  && wm.ball().pos().absY() > 3.0 )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          "__ angle=%.1f vel=(%.1f %.1f)"
                          " 2 intersects with v_lines. angle is dangerous.",
                          a, vel.x, vel.y );
#endif
            speed -= 2.0;
        }

        if ( speed > max_speed )
        {
            max_speed = speed;
            best_angle = a;
#ifdef DEBUG_PRINT
            dlog.addText( Logger::TEAM,
                          __FILE__": executeV12() angle=%.1f vel=(%.1f %.1f)%.2f"
                          " update.",
                          a, vel.x, vel.y, speed );
#endif
        }
#ifdef DEBUG_PRINT
        else
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": executeV12() angle=%.1f vel=(%.1f %.1f)%.2f"
                          " no update.",
                          a, vel.x, vel.y, speed );
        }
#endif
    }

    //
    // never accelerate the ball
    //

    if ( max_speed < 1.0 )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::TEAM,
                      __FILE__": failed executeV12 max_speed=%.3f",
                      max_speed );
#endif
        return false;
    }

    //
    // execute tackle
    //

    double tackle_dir = ( best_angle - wm.self().body() ).degree();

    dlog.addText( Logger::TEAM,
                  __FILE__": danger area angle=%.0f",
                  best_angle.degree() );
    agent->debugClient().addMessage( "DanAreaTackle%.0f",
                                     best_angle.degree() );

    agent->doTackle( tackle_dir );
    agent->setNeckAction( new Neck_TurnToBallOrScan() );

    return true;

}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_DangerAreaTackle::executeV14( PlayerAgent * agent,
                                  const bool use_foul )
{
    const WorldModel & wm = agent->world();

    const TackleGenerator::TackleResult & result
        = TackleGenerator::instance().bestResult( wm );

    dlog.addText( Logger::TEAM,
                  __FILE__": (executeV14) %s angle=%.0f resultVel=(%.2f %.2f) r=%.3f th=%.1f",
                  use_foul ? "foul" : "tackle",
                  result.tackle_angle_.degree(),
                  result.ball_vel_.x, result.ball_vel_.y,
                  result.ball_speed_, result.ball_move_angle_.degree() );
    agent->debugClient().addMessage( "Basic%s%.0f",
                                     use_foul ? "Foul" : "Tackle",
                                     result.tackle_angle_.degree() );

    double tackle_dir = ( result.tackle_angle_ - wm.self().body() ).degree();
    agent->doTackle( tackle_dir, use_foul );
    agent->setNeckAction( new Neck_TurnToBallOrScan() );

    return true;
}
