// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_defensive_mark.h"

#include "strategy.h"
#include "mark_analyzer.h"
#include "defense_system.h"

#include "bhv_find_player.h"
#include "neck_check_ball_owner.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/neck_turn_to_player_or_scan.h>
#include <rcsc/action/view_synch.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

*/
bool
Bhv_DefensiveMark::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::TEAM | Logger::MARK,
                  __FILE__": Bhv_DefensiveMark" );

    const WorldModel & wm = agent->world();

    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( mate_min <= opp_min - 2 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) no mark situation." );
        return false;
    }

    const AbstractPlayerObject * mark_target = MarkAnalyzer::i().getTargetOf( wm.self().unum() );

    if ( ! mark_target )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) no mark target." );
        return false;
    }

    dlog.addText( Logger::ROLE,
                  __FILE__": (execute) mark target ghostCount()=%d",
                  mark_target->ghostCount() );

    if ( mark_target->ghostCount() >= 5 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) mark target is ghost." );
        return false;
    }

    if ( mark_target->ghostCount() >= 2 )
    {
        Bhv_FindPlayer( mark_target, 0 ).execute( agent );
        return true;
    }


    const Vector2D mark_target_pos = mark_target->pos() + ( mark_target->vel() * 2.0 );
    const Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );

    if ( std::fabs( mark_target_pos.y - home_pos.y ) > 15.0
         || mark_target_pos.dist2( home_pos ) > std::pow( 15.0, 2 ) )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) too far mark target (%.2f %.2f) y_diff=%.3f dist=%.3f",
                      mark_target_pos.x, mark_target_pos.y,
                      std::fabs( mark_target_pos.y - home_pos.y ),
                      mark_target_pos.dist( home_pos ) );
        return false;
    }

    const Vector2D opp_trap_pos = wm.ball().inertiaPoint( opp_min );

//     if ( std::fabs( mark_target_pos.y - opp_trap_pos.y ) > 30.0 )
//     {
//         dlog.addText( Logger::ROLE,
//                       __FILE__": (execute) big y difference=%.3f."
//                       " mark_target_pos=(%.2f %.2f) opp_trap_pos=(%.2f %.2f)",
//                       std::fabs( mark_target_pos.y - opp_trap_pos.y ),
//                       mark_target_pos.x, mark_target_pos.y,
//                       opp_trap_pos.x, opp_trap_pos.y );
//         return false;
//     }

    double y_diff = opp_trap_pos.y - mark_target_pos.y;

    Vector2D target_point = mark_target_pos;
    //target_point.x -= 3.5;
    target_point.x -= 4.0; // 2009-12-14

    if ( std::fabs( y_diff ) < 3.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) y adjust(1) no adjust" );
    }
    else if ( std::fabs( y_diff ) < 7.0 )
    {
        target_point.y += 0.45 * sign( y_diff );
        //target_point.y += 0.75 * sign( y_diff );
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) y adjust(2) %f", 0.45 * sign( y_diff ) );
    }
    else
    {
        target_point.y += 0.9 * sign( y_diff );
        //target_point.y += 1.25 * sign( y_diff );
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) y adjust(3) %f", 0.9 * sign( y_diff ) );
    }

    if ( target_point.x < wm.ourDefenseLineX() - 3.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) adjust target_point_x=(%.2f %.2f) to defense_line=%.2f",
                      target_point.x, target_point.y,
                      wm.ourDefenseLineX() );
        target_point.x = wm.ourDefenseLineX() - 3.0;
    }

    if ( mark_target->velCount() <= 2
         && mark_target->vel().x < -0.2 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) adjust target_point_x again for opponent attacker dash" );
        target_point.x -= 1.0;
    }

    const double min_x = ( wm.ball().pos().x > 0.0
                           ? -15.0
                           : wm.ball().pos().x > -12.5
                           ? -23.5 //? -25.0 //? -22.0
                           : wm.ball().pos().x > -21.5
                           ? -30.0
                           : -36.5 );

    if ( target_point.x < min_x )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) adjust target_point_x(2) min_x=%.1f",
                      min_x);
        target_point.x = min_x;

        if ( target_point.x > mark_target_pos.x + 0.6 )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": (execute) adjust target_point_x. ajuast to opponent x."
                          " min_x=%.2f target_x=%.2f mark_target_x=%.2f",
                          min_x, target_point.x, mark_target_pos.x );
            target_point.x = mark_target_pos.x + 0.6;
        }
    }

    if ( target_point.x > home_pos.x )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) adjust target_point_x=(%.2f %.2f) to home_x=(%.2f %.2f)",
                      target_point.x, target_point.y,
                      home_pos.x, home_pos.y );
        target_point.x = home_pos.x;
    }

#if 1
    // 2009-07-03
    Vector2D my_inertia = wm.self().inertiaFinalPoint();
    if ( target_point.x < my_inertia.x - 1.0
         && std::fabs( my_inertia.y - mark_target_pos.y ) < std::fabs( target_point.y - mark_target_pos.y ) )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) adjust target_point_y=(%.2f %.2f) to self_y=%.2f",
                      target_point.x, target_point.y,
                      my_inertia.y );
        target_point.y = my_inertia.y;
    }
#else
    // 2009-07-03
    Vector2D my_inertia = wm.self().inertiaFinalPoint();
    if ( target_point.x < my_inertia.x - 1.0
         && std::fabs( my_inertia.y - opp_trap_pos.y ) < std::fabs( target_point.y - opp_trap_pos.y ) )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) adjust target_point_y=(%.2f %.2f) to self_y=%.2f",
                      target_point.x, target_point.y,
                      my_inertia.y );
        target_point.y = my_inertia.y;
    }
#endif

    double dash_power = ServerParam::i().maxPower();
    if ( wm.self().pos().dist( target_point ) < 2.0 )
    {
        dash_power = DefenseSystem::get_defender_dash_power( wm, target_point );
    }

    double dist_thr = 0.5;

    if ( //wm.self().pos().x < target_point.x
        wm.ball().pos().dist( target_point ) > 30.0 )
    {
        dist_thr = 2.0;
    }

    agent->debugClient().addMessage( "Def:Mark" );
    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    dlog.addText( Logger::ROLE,
                  __FILE__": (execute) pos=(%.2f %.2f) dist_thr=%.3f dash_power=%.1f",
                  target_point.x, target_point.y,
                  dist_thr, dash_power );

    if ( doGoToPoint( agent, mark_target,
                      target_point, dist_thr, dash_power, 12.0 ) )
    {
        return true;
    }

    doTurn( agent, mark_target );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_DefensiveMark::doGoToPoint( PlayerAgent * agent,
                                const rcsc::AbstractPlayerObject * mark_target,
                                const Vector2D & target_point,
                                const double & dist_thr,
                                const double & dash_power,
                                const double & dir_thr )
{
    if ( Body_GoToPoint( target_point, dist_thr, dash_power,
                         -1.0, // dash speed
                         1, true, dir_thr
                         ).execute( agent ) )
    {
        agent->debugClient().addMessage( "Go%.1f", dash_power );
        dlog.addText( Logger::ROLE,
                      __FILE__": GoToPoint (%.1f %.1f) dash_power=%.1f dist_thr=%.2f",
                      target_point.x, target_point.y,
                      dash_power,
                      dist_thr );

        doTurnNeck( agent, mark_target );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_DefensiveMark::doTurn( PlayerAgent * agent,
                           const rcsc::AbstractPlayerObject * mark_target )
{
    const WorldModel & wm = agent->world();

    const int min_step
        = std::min( wm.interceptTable()->teammateReachCycle(),
                    wm.interceptTable()->opponentReachCycle() );

    Vector2D ball_pos = wm.ball().inertiaPoint( min_step );
    Vector2D my_pos = wm.self().inertiaPoint( min_step );

    Vector2D mark_vel = mark_target->vel() / mark_target->playerTypePtr()->playerDecay();
    Vector2D mark_pos = mark_target->pos() + ( mark_vel * min_step );

    AngleDeg ball_angle = ( ball_pos - my_pos ).th();
    AngleDeg mark_angle = ( mark_pos - my_pos ).th();

    AngleDeg target_angle = AngleDeg::bisect( ball_angle, mark_angle );

    if ( ( target_angle - ball_angle ).abs() > 90.0 )
    {
        target_angle += 180.0;
    }

    Body_TurnToAngle( target_angle ).execute( agent );
    agent->debugClient().addMessage( "Mark:TurnTo%.0f",
                                     target_angle.degree() );
    dlog.addText( Logger::ROLE,
                  __FILE__": TurnToAngle %.1f, ball_angle=%.1f mark_angle=%.1f mark_pos=(%.1f %.1f)",
                  target_angle.degree(),
                  ball_angle.degree(),
                  mark_angle.degree(), mark_pos.x, mark_pos.y );


    doTurnNeck( agent, mark_target );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_DefensiveMark::doTurnNeck( PlayerAgent * agent,
                               const rcsc::AbstractPlayerObject * mark_target )
{
    const WorldModel & wm = agent->world();

    agent->setViewAction( new View_Synch() );

    if ( wm.ball().seenPosCount() >= 3
         || ( wm.existKickableOpponent()
              && wm.ball().distFromSelf() < 10.0 ) )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doTurnNeck) check ball" );
        agent->debugClient().addMessage( "Mark:NeckBall" );

        agent->setNeckAction( new Neck_TurnToBallOrScan() );
        return;
    }

    if ( mark_target->ghostCount() >= 2 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": (doTurnNeck) check mark target, ghost" );
        agent->debugClient().addMessage( "Mark:NeckGhost" );

        agent->setNeckAction( new Neck_ScanField() );
        return;
    }

    dlog.addText( Logger::ROLE,
                  __FILE__": (doTurnNeck) check mark target, normal" );
    agent->debugClient().addMessage( "Mark:NeckMark" );

    agent->setNeckAction( new Neck_TurnToPlayerOrScan( mark_target, 0 ) );
}
