// -*-c++-*-

/*!
  \file bhv_forward_shoot_area_kick.cpp
  \brief forward role behavior
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_forward_shoot_area_kick.h"

#include "strategy.h"
#include "bhv_cross.h"
#include "bhv_self_pass.h"
#include "body_kick_to_corner.h"

#include <rcsc/action/body_dribble.h>
#include <rcsc/action/body_pass.h>
#include <rcsc/action/body_hold_ball.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/body_smart_kick.h>

#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_goalie_or_scan.h>
#include <rcsc/action/neck_turn_to_point.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/common/player_type.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <rcsc/geom/sector_2d.h>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_ForwardShootAreaKick::execute( PlayerAgent * agent )
{
    static int s_count_hold = 0;

    dlog.addText( Logger::ROLE,
                  __FILE__": (execute)" );
    agent->debugClient().addMessage( "ForwardShootAreaKick" );

    const WorldModel & wm = agent->world();

    Vector2D receive_point = Vector2D::INVALIDATED;
    const bool exist_pass = Body_Pass::get_best_pass( wm, &receive_point, NULL, NULL );


    if ( exist_pass
         && wm.self().pos().absY() > 10.0
         && receive_point.x > 38.0
         && receive_point.absY() < 7.0
         && wm.countOpponentsIn( Circle2D( receive_point, 5.0 ), 10, false ) == 0 )
    {
        agent->debugClient().addMessage( "ShootAreaFreePass" );
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) first receive point (%.1f %.1f) final=(%.1f %.1f)",
                      receive_point.x,
                      receive_point.y,
                      receive_point.x,
                      receive_point.y );
        Body_Pass().execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    if ( Bhv_SelfPass().execute( agent ) )
    {
        return true;
    }

    //------------------------------------------------------------//
    // get the nearest opponent
    const PlayerObject * nearest_opp = wm.getOpponentNearestToSelf( 10 );
    const double opp_dist = ( nearest_opp
                              ? nearest_opp->distFromSelf()
                              : 1000.0 );
    const Vector2D opp_pos = ( nearest_opp
                               ? nearest_opp->pos()
                               : Vector2D( -1000.0, 0.0 ) );

    //------------------------------------------------------------//
    // get goalie parameters
    const PlayerObject * opp_goalie = wm.getOpponentGoalie();
    Vector2D goalie_pos( -1000.0, 0.0 );
    double goalie_dist = 15.0;
    if ( opp_goalie )
    {
        goalie_pos = opp_goalie->pos();
        goalie_dist = opp_goalie->distFromSelf();
    }

    //------------------------------------------------------------//
    // count opponent at the front of goal
    const int num_opp_at_goal_front = wm.countOpponentsIn( Rect2D( Vector2D( 45.0, -6.0 ),
                                                                   Size2D( 7.0, 12.0 ) ),
                                                           10,
                                                           false ); // without goalie
    agent->debugClient().addRectangle( Rect2D( Vector2D( 45.0, -6.0 ),
                                               Size2D( 7.0, 12.0 ) ) );
    dlog.addText( Logger::ROLE,
                  __FILE__":  (execute) num_opp_at_goal_front = %d",
                  num_opp_at_goal_front );

    //------------------------------------------------------------//
    // check dribble chance
    if ( num_opp_at_goal_front <= 2
         && wm.self().pos().x < 42.0
         && wm.self().pos().absY() < 8.0
         && wm.self().stamina() > ServerParam::i().staminaMax() * 0.5 )
    {
        const Rect2D target_rect
            = Rect2D::from_center( wm.ball().pos().x + 5.5,
                                   wm.ball().pos().y,
                                   10.0, 12.0 );
        if ( ! nearest_opp
             || ( wm.self().body().abs() < 30.0
                  && opp_pos.x < wm.ball().pos().x
                  && nearest_opp->distFromBall() > 2.0
                  && ! wm.existOpponentIn( target_rect, 10, false ) )
             )
        {
            if ( ! opp_goalie
                 || opp_goalie->distFromBall() > 5.0 )
            {
                Vector2D drib_target
                    = wm.self().pos()
                    + Vector2D::polar2vector( 10.0, wm.self().body() );

                agent->debugClient().addMessage( "ShootDribSt" );
                agent->debugClient().addRectangle( target_rect );
                dlog.addText( Logger::ROLE,
                              __FILE__":  (execute) dribble straight to (%.1f %.1f)",
                              drib_target.x, drib_target.y );
                Body_Dribble( drib_target,
                              1.0,
                              ServerParam::i().maxDashPower(),
                              5, // dash count
                              false // no dodge
                              ).execute( agent );
                if ( opp_goalie )
                {
                    agent->setNeckAction( new Neck_TurnToPoint( opp_goalie->pos() ) );
                }
                else
                {
                    agent->setNeckAction( new Neck_ScanField() );
                }
                return true;
            }
        }

    }

    // check pass parameters

    double pass_point_opp_dist = 1000.0;
    if ( exist_pass )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__":  (execute) PassPoint(%.1f %.1f)",
                      receive_point.x,
                      receive_point.y );
        if ( ! wm.getOpponentNearestTo( receive_point,
                                        10,
                                        &pass_point_opp_dist ) )
        {
            pass_point_opp_dist = 1000.0;
        }
    }

    //------------------------------------------------------------//
    // check very good pass

    if ( exist_pass
         && pass_point_opp_dist > 5.0
         && ( receive_point.x > 42.0
              || receive_point.x > wm.self().pos().x - 1.0 )
         && receive_point.absY() < 10.0 )
    {
        agent->debugClient().addMessage( "ShootAreaPass(1)" );
        dlog.addText( Logger::ROLE,
                      __FILE__": (execute) Pass1-1 To(%.1f %.1f)",
                      receive_point.x,
                      receive_point.y );
        Body_Pass().execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    if ( exist_pass
         && pass_point_opp_dist > 3.0
         && ( receive_point.x > 42.0
              || receive_point.x > wm.self().pos().x - 1.0 )
         && receive_point.absY() < 5.0 )
    {
        agent->debugClient().addMessage( "ShotAreaPass(1-2)" );
        dlog.addText( Logger::ROLE,
                      __FILE__": doShootAreaKick. Pass1-2 To(%.1f %.1f)",
                      receive_point.x,
                      receive_point.y );
        Body_Pass().execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    //------------------------------------------------------------//
    // check cross paramters
    Vector2D cross_point( Vector2D::INVALIDATED );
    bool can_cross = Bhv_Cross::get_best_point( agent, &cross_point );
    double cross_point_opp_dist = 20.0;
    if ( can_cross )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": doShootAreaKick. CrossPoint(%.1f %.1f)",
                      cross_point.x, cross_point.y );

        if ( cross_point.absY() > 10.0
             && cross_point.y * wm.self().pos().y > 0.0 // same side
             && cross_point.absY() > wm.self().pos().absY() )
        {
            can_cross = false;
            agent->debugClient().addMessage( "InvalidCross" );
            dlog.addText( Logger::ROLE,
                          __FILE__": doShootAreaKick. cross point is same side &outer. cancel",
                          cross_point.x, cross_point.y );
        }

        if ( can_cross )
        {
            if ( ! wm.getOpponentNearestTo( cross_point,
                                            10,
                                            &cross_point_opp_dist ) )
            {
                cross_point_opp_dist = 20.0;
            }
        }
    }

    dlog.addText( Logger::ROLE,
                  __FILE__": (doShootAreaKick) cross_point_opp_dist=%.3f",
                  cross_point_opp_dist );

    //------------------------------------------------------------//
    // check good cross chance
    if ( can_cross
         //&& cross_point.x > 42.0
         //&& cross_point_opp_dist > 3.0 )
         && cross_point.x > 37.0
         && cross_point_opp_dist > bound( 3.0, 42.0 - cross_point.x, 7.0 ) )
    {
        agent->debugClient().addMessage( "ShotAreaCross(1)" );
        dlog.addText( Logger::ROLE,
                      __FILE__": doShootAreaKick. found good cross(%.1f %.1f)",
                      cross_point.x, cross_point.y );
        Bhv_Cross().execute( agent );
        return true;
    }

#if 1
    // 2009-07-03
    if ( can_cross
         && cross_point.x > 41.0
         && cross_point_opp_dist > 2.25
         && cross_point.absY() < 7.0 )
    {
        agent->debugClient().addMessage( "ShotAreaCross(1-3)" );
        dlog.addText( Logger::ROLE,
                      __FILE__": (doShootAreaKick) found good cross(3) pos=(%.1f %.1f)",
                      cross_point.x, cross_point.y );
        Bhv_Cross().execute( agent );
        return true;
    }
#endif

    if ( can_cross
         && cross_point.x > 44.0
         && cross_point_opp_dist > 2.0
         && cross_point.absY() < 7.0 )
    {
        agent->debugClient().addMessage( "ShotAreaCross(1-2)" );
        dlog.addText( Logger::ROLE,
                      __FILE__": doShootAreaKick. found good cross(2) pos=(%.1f %.1f)",
                      cross_point.x, cross_point.y );
        Bhv_Cross().execute( agent );
        return true;
    }

    //------------------------------------------------------------//
    // check dribble chance
    if ( num_opp_at_goal_front <= 2
         && wm.self().pos().x < 46.5
         && ( opp_dist > 3.0 || opp_pos.x < wm.self().pos().x + 0.5 )
         && ( goalie_dist > 5.0
              || wm.self().pos().x < goalie_pos.x - 3.0 )
         )
    {
        bool goalie_front = false;
        if ( opp_goalie
             && opp_goalie->pos().x < 46.0
             && opp_goalie->pos().x < wm.self().pos().x + 7.0
             && std::fabs( opp_goalie->pos().y - wm.self().pos().y ) < 2.0 )
        {
            goalie_front = true;
        }

        if ( ! goalie_front )
        {
            const Rect2D target_rect( Vector2D( wm.ball().pos().x + 0.5,
                                                wm.ball().pos().y - 1.4 ),
                                      Size2D( 2.5, 2.8 ) );
            const Rect2D target_rect2( Vector2D( wm.ball().pos().x + 1.0,
                                                 wm.ball().pos().y - 0.7 ),
                                       Size2D( 7.0, 1.4 ) );
            if ( ! wm.existOpponentIn( target_rect, 10, true ) // with goalie
                 //&& ! wm.existOpponentIn( target_rect2, 10, false ) // without goalie
                 )
            {
                Vector2D drib_target( 47.0, wm.self().pos().y );
#if 1
                if ( wm.self().body().abs() < 20.0 )
                {
                    Line2D my_line( wm.self().pos(), wm.self().body() );
                    drib_target.y = my_line.getY( drib_target.x );
                    if ( drib_target.y == Line2D::ERROR_VALUE )
                    {
                        drib_target.y = wm.self().pos().y;
                    }
                }
#endif
                if ( drib_target.absY() > 12.0 )
                {
                    drib_target.y = ( drib_target.y > 0.0 ? 12.0 : -12.0 );
                }
#if 1
                if ( wm.ball().pos().x > 42.0
                     && drib_target.absY() > 7.0 )
                {
                    drib_target.y = ( drib_target.y > 0.0 ? 7.0 : -7.0 );
                }
#endif
                if ( opp_goalie
                     && opp_goalie->pos().x > wm.self().pos().x
                     //&& opp_goalie->pos().y * wm.self().pos().y < 0.0
                     && std::fabs( opp_goalie->pos().y - wm.self().pos().y ) < 3.0
                     && opp_goalie->pos().absY() < wm.self().pos().absY() )
                {
                    drib_target.x = opp_goalie->pos().x - 3.0;
                }
                if ( wm.self().pos().x > drib_target.x - 0.8
                     && wm.self().pos().absY() > 9.0 )
                {
                    drib_target.y = 7.0 * ( drib_target.y < 0.0 ? -1.0 : 1.0 );
                }

                if ( wm.self().pos().dist( drib_target ) > 1.0 )
                {
                    agent->debugClient().addMessage( "ShotAreaDrib1" );
                    agent->debugClient().addRectangle( target_rect );
                    agent->debugClient().addRectangle( target_rect2 );
                    dlog.addText( Logger::ROLE,
                                  __FILE__": doShootAreaKick. dribble to (%.1f %.1f)",
                                  drib_target.x, drib_target.y );
                    Body_Dribble( drib_target,
                                  1.0,
                                  ServerParam::i().maxDashPower(),
                                  1, // one dash
                                  false // no dodge
                                  ).execute( agent );
                    if ( ! doCheckCrossPoint( agent ) )
                    {
                        agent->setNeckAction( new Neck_TurnToGoalieOrScan( 0 ) );
                    }
                    return true;
                }
            }
        }
    }

#if 1
    //------------------------------------------------------------//
    // check dribble chance (2)
    if ( wm.self().pos().x > 44.0
         && goalie_dist > 4.0
         )
    {
        Vector2D drib_target( 45.0, 0.0 );
        const Rect2D target_rect( Vector2D( wm.self().pos().x - 2.0,
                                            ( wm.self().pos().y > 0.0
                                              ? wm.self().pos().y - 5.0
                                              : wm.self().pos().y ) ),
                                  Size2D( 4.0, 5.0 ) );
        bool try_dribble = false;
        // opponent check with goalie
        if ( ! wm.existOpponentIn( target_rect, 10, true ) )
        {
            try_dribble = true;

            if ( opp_goalie
                 && opp_goalie->pos().x > wm.self().pos().x
                 && std::fabs( opp_goalie->pos().y - wm.self().pos().y ) < 3.0
                 && opp_goalie->pos().absY() < wm.self().pos().absY() )
            {
                drib_target.x = opp_goalie->pos().x - 3.0;
                drib_target.y += ( opp_goalie->pos().y > wm.self().pos().y
                                   ? - 5.0
                                   : + 5.0 );
            }

            if ( nearest_opp
                 && nearest_opp->distFromSelf() < 0.8 )
            {
                Vector2D my_next = wm.self().pos() - wm.self().vel();
                AngleDeg target_angle = ( drib_target - my_next ).th();
                if ( ( target_angle - wm.self().body() ).abs() > 20.0 )
                {
                    try_dribble = false;
                }
            }
        }

        if ( try_dribble )
        {
            agent->debugClient().addMessage( "ShotAreaDrib2" );
            agent->debugClient().addRectangle( target_rect );
            dlog.addText( Logger::ROLE,
                          __FILE__": doShootAreaKick. dribble2 to (%.1f %.1f)",
                          drib_target.x, drib_target.y );
            Body_Dribble( drib_target,
                          1.0,
                          ServerParam::i().maxDashPower() * 0.7,
                          1, // one dash
                          false // no dodge
                          ).execute( agent );
            if ( ! doCheckCrossPoint( agent ) )
            {
                agent->setNeckAction( new Neck_TurnToGoalieOrScan( 0 ) );
            }

            return true;
        }
    }
#endif

    //------------------------------------------------------------//
    // check opponent
    // if opponent is close
    if ( //opp_dist < 2.0
        opp_dist < 3.5 // 2009-12-14
         || goalie_dist < 4.0 )
    {
        if ( can_cross )
        {
            agent->debugClient().addMessage( "ShotAreaCross(2)" );
            dlog.addText( Logger::ROLE,
                          __FILE__": doShootAreaKick. cross(2)  opp_dist=%.2f goalie_dist=%.2f",
                          opp_dist, goalie_dist );
            Bhv_Cross().execute( agent );
            return true;
        }

        if ( exist_pass )
        {
            agent->debugClient().addMessage( "ShotAreaPass(2)" );
            dlog.addText( Logger::ROLE,
                          __FILE__": doShootAreaKick. pass  opp_dist=%.2f goalie_dist=%.2f",
                          opp_dist, goalie_dist );
            Body_Pass().execute( agent );
            agent->setNeckAction( new Neck_ScanField() );
            return true;
        }

        Rect2D goal_area( Vector2D( ServerParam::i().pitchHalfLength() - 10.0,
                                    -10.0 ),
                          Size2D( 13.0, 20.0 ) );
        if ( wm.existTeammateIn( goal_area, 10, true ) )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": doShootAreaKick. enforce cross" );

            Vector2D kick_target( 45.0, 20.0 );

            kick_target.x = std::max( 45.0, wm.self().pos().x - 1.0 );
            if ( wm.self().pos().y > 0.0 ) kick_target.y *= -1.0;

            agent->debugClient().setTarget( kick_target );

            if ( goalie_dist < 3.0 )
            {
                agent->debugClient().addMessage( "ForceCross1K" );
                Body_KickOneStep( kick_target,
                                  ServerParam::i().ballSpeedMax()
                                  ).execute( agent );
            }
            else
            {
                agent->debugClient().addMessage( "ForceCross2K" );
                Body_SmartKick( kick_target,
                                ServerParam::i().ballSpeedMax(),
                                ServerParam::i().ballSpeedMax() * 0.8,
                                2 ).execute( agent );
            }

            if ( ! doCheckCrossPoint( agent ) )
            {
                agent->setNeckAction( new Neck_TurnToGoalieOrScan() );
            }
            return true;
        }
    }

    ++s_count_hold;

    if ( s_count_hold >= 30 )
    {
        Vector2D drib_target( wm.self().pos().x + 5.0, wm.self().pos().y );
        agent->debugClient().addMessage( "ShotAreaDribFinal" );
        dlog.addText( Logger::ROLE,
                      __FILE__": doShootAreaKick. dribble final to (%.1f %.1f)",
                      drib_target.x, drib_target.y );
        Body_Dribble( drib_target,
                      1.0,
                      ServerParam::i().maxDashPower() * 0.7,
                      1, // one dash
                      false // no dodge
                      ).execute( agent );
        if ( ! doCheckCrossPoint( agent ) )
        {
            agent->setNeckAction( new Neck_TurnToGoalieOrScan( 0 ) );
        }

        s_count_hold = 0;
        return true;
    }

    agent->debugClient().addMessage( "ShotAreaHold" );
    Vector2D face_point( 52.5, 0.0 );
    Body_HoldBall( true, face_point ).execute( agent );

    if ( ! doCheckCrossPoint( agent ) )
    {
        agent->setNeckAction( new Neck_TurnToGoalieOrScan( 0 ) );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_ForwardShootAreaKick::doCheckCrossPoint( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    if ( wm.self().pos().x < 35.0 )
    {
        return false;
    }

    const PlayerObject * opp_goalie = wm.getOpponentGoalie();
    if ( opp_goalie && opp_goalie->posCount() > 2 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": doCheckCrossTarget  goalie should be checked" );
        return false;
    }

    Vector2D opposite_pole( 46.0, 7.0 );
    if ( wm.self().pos().y > 0.0 ) opposite_pole.y *= -1.0;

    AngleDeg opposite_pole_angle = ( opposite_pole - wm.self().pos() ).th();


    if ( wm.dirCount( opposite_pole_angle ) <= 1 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": doCheckCrossTarget enough accuracy to angle %.1f",
                      opposite_pole_angle.degree() );
        return false;
    }

    AngleDeg angle_diff
        = agent->effector().queuedNextAngleFromBody( opposite_pole );
    if ( angle_diff.abs() > 100.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": doCheckCrossPoint. want to face opposite pole,"
                      " but over view range. angle_diff=%.1f",
                      angle_diff.degree() );
        return false;
    }


    agent->setNeckAction( new Neck_TurnToPoint( opposite_pole ) );
    agent->debugClient().addMessage( "NeckToOpposite" );
    dlog.addText( Logger::ROLE,
                  __FILE__": doCheckCrossPoint Neck to oppsite pole" );
    return true;
}
