// -*-c++-*-

/*!
  \file block_generator.cpp
  \brief opponent block action generator class Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 3 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "block_generator.h"

#include "strategy.h"
#include "field_analyzer.h"

#include <rcsc/player/world_model.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/game_time.h>
#include <rcsc/math_util.h>
#include <rcsc/timer.h>
#include <rcsc/types.h>

#include <limits>

#define DEBUG_PROFILE
#define DEBUG_PRINT

// #define DEBUG_UPDATE_BLOCKER

// #define DEBUG_PREDICT_OPPONENT
// #define DEBUG_PREDICT_OPPONENT_LEVEL2
#define DEBUG_PREDICT_BLOCKER
// #define DEBUG_PREDICT_BLOCKER_LEVEL2

#define DEBUG_PRINT_SUCCESS_POINT
#define DEBUG_PRINT_FAILED_POINT

// #define DEBUG_PRINT_EVALUATE


using namespace rcsc;

namespace {
const int MAX_DIVS = 32;
}

/*-------------------------------------------------------------------*/
/*!

 */
BlockGenerator::BlockResult::BlockResult()
    : target_point_( Vector2D::INVALIDATED ),
      opponent_unum_( Unum_Unknown ),
      opponent_kick_step_( 0 ),
      opponent_turn_step_( 0 ),
      opponent_dash_step_( 0 ),
      blocker_turn_step_( 0 ),
      blocker_dash_step_( 0 ),
      blocker_stamina_( 0.0 ),
      score_( -65535.0 )
{

}

/*-------------------------------------------------------------------*/
/*!

 */
BlockGenerator::BlockResult::BlockResult( const Vector2D & target_point,
                                          const int opponent_unum,
                                          const int opponent_kick,
                                          const int opponent_turn,
                                          const int opponent_dash,
                                          const int blocker_turn,
                                          const int blocker_dash,
                                          const double & blocker_stamina )
    : target_point_( target_point ),
      opponent_unum_( opponent_unum ),
      opponent_kick_step_( opponent_kick ),
      opponent_turn_step_( opponent_turn ),
      opponent_dash_step_( opponent_dash ),
      blocker_turn_step_( blocker_turn ),
      blocker_dash_step_( blocker_dash ),
      blocker_stamina_( blocker_stamina ),
      score_( 0.0 )
{

}

/*-------------------------------------------------------------------*/
/*!

 */
BlockGenerator::BlockGenerator()
{
    M_candidates.reserve( MAX_DIVS );
    clear();
}

/*-------------------------------------------------------------------*/
/*!

 */
BlockGenerator &
BlockGenerator::instance()
{
    static BlockGenerator s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
BlockGenerator::clear()
{
    M_total_count = 0;
    M_target_opponent = static_cast< const AbstractPlayerObject * >( 0 );
    M_opponent_ball_reach_step  = 1000;
    M_blocker = static_cast< const AbstractPlayerObject * >( 0 );
    M_best_result = BlockResult();
    M_candidates.clear();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
BlockGenerator::generate( const WorldModel & wm )
{
    static GameTime s_update_time( 0, 0 );

    if ( s_update_time == wm.time() )
    {
        return;
    }
    s_update_time = wm.time();

    clear();

    if ( wm.self().isKickable() )
    {
        return;
    }

    if ( wm.time().stopped() > 0 )
    {
        return;
    }

    if ( wm.gameMode().type() != GameMode::PlayOn
         && ! wm.gameMode().isPenaltyKickMode() )
    {
        return;
    }


#ifdef DEBUG_PROFILE
    MSecTimer timer;
#endif

    updateTargetOpponent( wm );

    if ( ! M_target_opponent )
    {
        dlog.addText( Logger::BLOCK,
                      __FILE__": (generate) no target opponent." );
        return;
    }

    updateBlocker( wm );

    if ( ! M_blocker )
    {
        dlog.addText( Logger::BLOCK,
                      __FILE__": (generate) no blocker player." );
        return;
    }

    if ( M_blocker->unum() != wm.self().unum() )
    {
        dlog.addText( Logger::BLOCK,
                      __FILE__": (generate) other blocker." );
        return;
    }

    createCandidates( wm );

#ifdef DEBUG_PROFILE
    dlog.addText( Logger::BLOCK,
                  __FILE__": PROFILE. %d/%d elapsed=%.3f [ms]",
                  (int)M_candidates.size(), M_total_count,
                  timer.elapsedReal() );
#endif

}

/*-------------------------------------------------------------------*/
/*!

 */
void
BlockGenerator::updateTargetOpponent( const WorldModel & wm )
{
    const AbstractPlayerObject * opponent = wm.interceptTable()->fastestOpponent();

    if ( ! opponent )
    {
        dlog.addText( Logger::BLOCK,
                      __FILE__": (updateTargetOpponent) no opponent" );
        return;
    }

    const int s_min = wm.interceptTable()->selfReachCycle();
    const int t_min = wm.interceptTable()->teammateReachCycle();
    int o_min = wm.interceptTable()->opponentReachCycle();

    const AbstractPlayerObject * nearest = wm.getOpponentNearestToBall( 5 );

    if ( nearest
         && o_min >= 5 )
    {
        const PlayerType * ptype = nearest->playerTypePtr();
        const double pos_count = std::min( 3, nearest->posCount() );

        if ( nearest->distFromBall() < ( ptype->kickableArea()
                                         + ptype->realSpeedMax() * pos_count
                                         + nearest->distFromSelf() * 0.05 ) )
        {
            o_min = 0;
            opponent = nearest;

            dlog.addText( Logger::BLOCK,
                          __FILE__": (updateTargetOpponent) ball nearest opponent may be kickable soon" );
        }
    }

    if ( o_min >= s_min + 3
         || o_min >= t_min + 3 )
    {
        dlog.addText( Logger::BLOCK,
                      __FILE__": (updateTargetOpponent) our ball" );
        return;
    }

    dlog.addText( Logger::BLOCK,
                  __FILE__": (updateTargetOpponent) target=%d"
                  " (%.2f %.2f) posCount=%d body=%.1f bodyCount=%d",
                  opponent->unum(), opponent->pos().x, opponent->pos().y,
                  opponent->posCount(),
                  opponent->body().degree(), opponent->bodyCount() );

    M_target_opponent = opponent;
    M_opponent_ball_reach_step = o_min;
    M_first_ball_point = wm.ball().inertiaPoint( o_min );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
BlockGenerator::updateBlocker( const WorldModel & wm )
{
    if ( ! M_target_opponent )
    {
        return;
    }

#if 1

    M_blocker = &wm.self();

#else

    const ServerParam & SP = ServerParam::i();

    const Vector2D goal_pos = SP.ourTeamGoalPos();
    //const Segment2D block_line( M_first_ball_point, goal_pos );

    const Vector2D mid_point = M_first_ball_point * 0.8 + goal_pos * 0.2;

    // dlog.addText( Logger::BLOCK,
    //               __FILE__": (updateBlocker) mid_point=(%.2f %.2f)",
    //               mid_point.x, mid_point.y );

    int min_reach_cycle = 1000;
    const AbstractPlayerObject * candidate = static_cast< const AbstractPlayerObject * >( 0 );

    const AbstractPlayerCont::const_iterator t_end = wm.allTeammates().end();
    for ( AbstractPlayerCont::const_iterator t = wm.allTeammates().begin();
          t != t_end;
          ++t )
    {
        if ( (*t)->goalie() ) continue;
        if ( (*t)->isGhost() ) continue;

        int reach_cycle = FieldAnalyzer::predict_player_reach_cycle( *t,
                                                                     mid_point,
                                                                     0.5, // dist thr
                                                                     0.0, // no penalty
                                                                     2, // body count thr
                                                                     1,
                                                                     0, // no wait
                                                                     false ); // no back dash
        reach_cycle += (*t)->posCount();

        if ( (*t)->unum() != wm.self().unum() )
        {
            reach_cycle += 3;
        }

#ifdef DEBUG_UPDATE_BLOCKER
        dlog.addText( Logger::BLOCK,
                      __FILE__": (updateBlocker) blocker candidate=%d reach_cycle=%d",
                      (*t)->unum(), reach_cycle );
#endif

        if ( reach_cycle < min_reach_cycle )
        {
            min_reach_cycle = reach_cycle;
            candidate = *t;
        }
    }

    M_blocker = candidate;

    if ( M_blocker )
    {
        dlog.addText( Logger::BLOCK,
                      __FILE__": (updateBlocker) blocker=%d mid_point=(%.2f %.2f) reach_cycle=%d",
                      M_blocker->unum(),
                      mid_point.x, mid_point.y,
                      min_reach_cycle );
    }
#endif
}

/*-------------------------------------------------------------------*/
/*!

 */
void
BlockGenerator::createCandidates( const WorldModel & wm )
{
    static const double MAX_DIST_STEP = 3.0;

    const ServerParam & SP = ServerParam::i();

    const Vector2D first_pos = ( M_opponent_ball_reach_step == 0
                                 ? M_target_opponent->inertiaFinalPoint()
                                 : M_first_ball_point );

    const Vector2D block_pos = Strategy::i().getBlockPosition( first_pos );

    dlog.addText( Logger::BLOCK,
                  __FILE__": first_pos=(%.2f %.2f) block_position=(%.2f %.2f)",
                  first_pos.x, first_pos.y,
                  block_pos.x, block_pos.y );

    const AngleDeg angle = ( block_pos - first_pos ).th();
    const double max_dist = ( block_pos - first_pos ).r() + MAX_DIST_STEP;
    const Vector2D unit_vec = Vector2D::from_polar( 1.0, angle );

    double dist_step = 0.3;
    for ( double d = 0.3; d < max_dist; d += dist_step )
    {
        ++M_total_count;
        dist_step = std::min( MAX_DIST_STEP, dist_step + dist_step*0.1 );

        const Vector2D target_point = first_pos + ( unit_vec * d );

        if ( target_point.absX() > SP.pitchHalfLength() - 0.5
             || target_point.absY() > SP.pitchHalfWidth() - 0.5 )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::BLOCK,
                          "%d: xxx out of pitch (%.2f %.2f) angle=%.1f dist=%.2f dist_step=%.3f",
                          M_total_count,
                          target_point.x, target_point.y,
                          angle.degree(), d, dist_step );
#endif
            break;
        }

#ifdef DEBUG_PRINT
        dlog.addText( Logger::BLOCK,
                      "%d: (%.2f %.2f) angle=%.1f dist=%.2f dist_step=%.3f",
                      M_total_count,
                      target_point.x, target_point.y,
                      angle.degree(), d, dist_step );
#endif
        //
        // predict opponent
        //
        int opponent_turn_step = 0;
        int opponent_dash_step = 0;
        int opponent_reach_step = predictOpponentReachStep( M_target_opponent,
                                                            M_opponent_ball_reach_step,
                                                            target_point,
                                                            &opponent_turn_step,
                                                            &opponent_dash_step );

        //
        // predict blocker (self)
        //
        int blocker_turn_step = 1000;
        int blocker_dash_step = 1000;
        double blocker_stamina = 0.0;
        int blocker_reach_step = predictSelfReachStep( wm,
                                                       target_point,
                                                       opponent_reach_step, // as a max step
                                                       &blocker_turn_step,
                                                       &blocker_dash_step,
                                                       &blocker_stamina );

        if ( blocker_reach_step < opponent_reach_step )
        {
            M_candidates.push_back( BlockResult( target_point,
                                                 M_target_opponent->unum(),
                                                 M_opponent_ball_reach_step + 1, // kick step
                                                 opponent_turn_step,
                                                 opponent_dash_step,
                                                 blocker_turn_step,
                                                 blocker_dash_step,
                                                 blocker_stamina ) );
#ifdef DEBUG_PRINT_SUCCESS_POINT
            dlog.addText( Logger::BLOCK,
                          "%d: ok (%.2f %.2f) opp_step=%d(k:%d, t:%d, d:%d) self_step=%d(t:%d, d:%d)",
                          M_total_count,
                          opponent_reach_step,
                          M_opponent_ball_reach_step + 1, opponent_turn_step, opponent_dash_step,
                          blocker_reach_step, blocker_turn_step, blocker_dash_step );
            char num[8];
            snprintf( num, 8, "%d", M_total_count );
            dlog.addMessage( Logger::BLOCK,
                             target_point, num );
            dlog.addRect( Logger::BLOCK,
                          target_point.x - 0.1, target_point.y - 0.1,
                          0.2, 0.2,
                          "#00ff00" );
#endif
        }
#ifdef DEBUG_PRINT_FAILED_POINT
        else
        {
            dlog.addText( Logger::BLOCK,
                          "%d: xxx (%.2f %.2f) opp_step=%d(k:%d, t:%d, d:%d) self_step=%d(t:%d, d:%d)",
                          M_total_count,
                          opponent_reach_step,
                          M_opponent_ball_reach_step + 1, opponent_turn_step, opponent_dash_step,
                          blocker_reach_step, blocker_turn_step, blocker_dash_step );
            char num[8];
            snprintf( num, 8, "%d", M_total_count );
            dlog.addMessage( Logger::BLOCK,
                             target_point, num );
            dlog.addRect( Logger::BLOCK,
                          target_point.x - 0.1, target_point.y - 0.1,
                          0.2, 0.2,
                          "#ff0000" );
        }
#endif
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
int
BlockGenerator::predictOpponentReachStep( const AbstractPlayerObject * opponent,
                                          const int opponent_ball_reach_step,
                                          const rcsc::Vector2D & target_point,
                                          int * result_turn_step,
                                          int * result_dash_step )
{
    const PlayerType * ptype = opponent->playerTypePtr();

    const double opponent_first_speed
        = ( opponent_ball_reach_step == 0
            ? opponent->vel().r() * ptype->playerDecay() // assume one kick
            : ptype->realSpeedMax() * std::pow( ptype->playerDecay(), 2 ) );

    // TODO: more strict estimation
    const Vector2D opponent_trap_pos
        = M_first_ball_point
        + ( opponent->pos() - M_first_ball_point ).setLengthVector( ptype->kickableArea() * 0.5 );

    for ( int cycle = 2; cycle <= 6; ++cycle ) // Magic Number
    {
        const Vector2D inertia_pos = ( opponent_ball_reach_step == 0
                                       ? opponent->inertiaPoint( cycle )
                                       : opponent_trap_pos );
        const double target_dist = inertia_pos.dist( target_point );

        int n_dash = ptype->cyclesToReachDistance( target_dist );


        int n_turn = ( opponent->bodyCount() > 1
                       ? 0
                       : FieldAnalyzer::predict_player_turn_cycle( ptype,
                                                                   opponent->body(),
                                                                   opponent_first_speed,
                                                                   target_dist,
                                                                   ( target_point - inertia_pos ).th(),
                                                                   ptype->kickableArea(),
                                                                   false )  // without backdash
                       );

#ifdef DEBUG_PREDICT_OPPONENT_LEVEL2
        dlog.addText( Logger::BLOCK,
                      "%d: (opponent) cycle=%d inertia=(%.2f %.2f) dist=%.2f turn=%d dash=%d",
                      M_total_count,
                      cycle,
                      inertia_pos.x, inertia_pos.y, target_dist,
                      n_turn, n_dash );
#endif
        if ( 1 + n_turn + n_dash <= cycle )
        {
#ifdef DEBUG_PREDICT_OPPONENT
            dlog.addText( Logger::BLOCK,
                          "%d: (opponent) cycle=%d ok",
                          M_total_count,
                          cycle );
#endif
            *result_turn_step = n_turn;
            *result_dash_step = n_dash;
            return opponent_ball_reach_step + 1 + n_turn + n_dash;
        }
    }

    //
    // calculate final reach point
    //
    {
        const Vector2D inertia_pos = opponent->inertiaFinalPoint();
        const double target_dist = inertia_pos.dist( target_point );

        int n_dash = ptype->cyclesToReachDistance( target_dist );


        int n_turn = ( opponent->bodyCount() > 1
                       ? 0
                       : FieldAnalyzer::predict_player_turn_cycle( ptype,
                                                                   opponent->body(),
                                                                   opponent_first_speed,
                                                                   target_dist,
                                                                   ( target_point - inertia_pos ).th(),
                                                                   ptype->kickableArea(),
                                                                   false )  // without backdash
                       );
#ifdef DEBUG_PREDICT_OPPONENT
        dlog.addText( Logger::BLOCK,
                      "%d: (opponent) final. step=%d turn=%d dash=%d",
                      M_total_count,
                      1 + n_turn + n_dash,
                      n_turn, n_dash );
#endif
        *result_turn_step = n_turn;
        *result_dash_step = n_dash;
        return opponent_ball_reach_step + 1 + n_turn + n_dash;
    }

    *result_turn_step = 1000;
    *result_dash_step = 1000;
    return 1000;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
BlockGenerator::predictSelfReachStep( const rcsc::WorldModel & wm,
                                      const rcsc::Vector2D & target_point,
                                      const int opponent_reach_step,
                                      int * result_turn_step,
                                      int * result_dash_step,
                                      double * result_stamina )
{
    const double dist_thr = 0.5; // Magic Number

    const ServerParam & SP = ServerParam::i();
    const PlayerType * ptype = wm.self().playerTypePtr();
    const double self_first_speed = wm.self().vel().r();

    const double recover_dec_thr = SP.staminaMax() * SP.recoverDecThr();
    const int max_cycle = std::min( 50, opponent_reach_step );

    for ( int cycle = 1; cycle < max_cycle; ++cycle )
    {
        const Vector2D inertia_pos = wm.self().inertiaPoint( cycle );
        const double target_dist = inertia_pos.dist( target_point );

        if ( target_dist - dist_thr > ptype->realSpeedMax() * cycle )
        {
            continue;
        }

        if ( target_dist < dist_thr )
        {
            *result_turn_step = 0;
            *result_dash_step = cycle;
            return cycle;
        }

        const AngleDeg dash_angle = ( target_point - inertia_pos ).th();

        int n_turn = FieldAnalyzer::predict_player_turn_cycle( ptype,
                                                               wm.self().body(),
                                                               self_first_speed,
                                                               target_dist,
                                                               dash_angle,
                                                               dist_thr,
                                                               false );

        if ( n_turn >= cycle )
        {
            continue;
        }

#ifdef DEBUG_PREDICT_BLOCKER
        dlog.addText( Logger::BLOCK,
                      "%d: (self) cycle=%d inertia=(%.2f %.2f) dist=%.2f turn=%d",
                      M_total_count,
                      cycle,
                      inertia_pos.x, inertia_pos.y, target_dist,
                      n_turn );
#endif

        StaminaModel stamina = wm.self().staminaModel();

        stamina.simulateWaits( *ptype, n_turn );

        double move_dist = 0.0;
        double speed = self_first_speed * std::pow( ptype->playerDecay(), n_turn );
        for ( int n_dash = 1; n_dash < cycle - n_turn; ++n_dash )
        {
            double dash_power = std::min( SP.maxDashPower(),
                                          stamina.stamina() + ptype->extraStamina() );
            if ( ! stamina.capacityIsEmpty()
                 && stamina.stamina() - dash_power < recover_dec_thr )
            {
                dash_power = std::max( 0.9, stamina.stamina() - recover_dec_thr );
            }

            double accel = dash_power * ptype->dashPowerRate() * stamina.effort();
            speed += accel;
            move_dist += speed;
            speed *= ptype->playerDecay();

            stamina.simulateDash( *ptype, dash_power );

#ifdef DEBUG_PREDICT_BLOCKER_LEVEL2
            dlog.addText( Logger::BLOCK,
                          "%d: (self) cycle=%d inertia=(%.2f %.2f) dist=%.2f(moved=%.2f) turn=%d dash=%d stamina=%.1f",
                          M_total_count,
                          cycle,
                          inertia_pos.x, inertia_pos.y,
                          target_dist, move_dist,
                          n_turn, n_dash, stamina.stamina() );
#endif

            if ( move_dist > target_dist )
            {
#ifdef DEBUG_PREDICT_BLOCKER
                dlog.addText( Logger::BLOCK,
                              "%d: (self) ok. cycle=%d inertia=(%.2f %.2f) dist=%.2f(moved=%.2f) turn=%d dash=%d stamina=%.1f",
                              M_total_count,
                              cycle,
                              inertia_pos.x, inertia_pos.y,
                              target_dist, move_dist,
                              n_turn, n_dash, stamina.stamina() );
#endif
                *result_turn_step = n_turn;
                *result_dash_step = n_dash;
                *result_stamina = stamina.stamina();
                return n_turn + n_dash;
            }
        }
    }

#ifdef DEBUG_PREDICT_BLOCKER
    dlog.addText( Logger::BLOCK,
                  "%d: (self) nerver reach.",
                  M_total_count );
#endif

    return 1000;
}

/*-------------------------------------------------------------------*/
/*!

 */
double
BlockGenerator::evaluate( const WorldModel & wm,
                          const BlockResult & result )
{
    (void)wm;
    (void)result;
    // const ServerParam & SP = ServerParam::i();
    double score = 0.0;


    return score;
}
