// -*-c++-*-

/*!
  \file block_generator.h
  \brief opponent block action generator Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 3 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef BLOCK_GENERATOR_H
#define BLOCK_GENERATOR_H

#include <rcsc/geom/vector_2d.h>

#include <vector>

namespace rcsc {
class AbstractPlayerObject;
class WorldModel;
}

/*!
  \brief block action simulator & generator
 */
class BlockGenerator {
public:

    struct BlockResult {
        rcsc::Vector2D target_point_;
        int opponent_unum_; //!< target opponent player's uniform number
        int opponent_kick_step_;
        int opponent_turn_step_;
        int opponent_dash_step_;
        int blocker_turn_step_;
        int blocker_dash_step_;
        double blocker_stamina_;
        double score_;

        BlockResult();
        BlockResult( const rcsc::Vector2D & target_point,
                     const int opponent_unum,
                     const int opponent_kick,
                     const int opponent_turn,
                     const int opponent_dash,
                     const int blocker_turn,
                     const int blocker_dash,
                     const double & blocker_stamina );
    };

    typedef std::vector< BlockResult > Container;

private:

    int M_total_count;

    const rcsc::AbstractPlayerObject * M_target_opponent;
    int M_opponent_ball_reach_step;
    rcsc::Vector2D M_first_ball_point;

    const rcsc::AbstractPlayerObject * M_blocker;

    Container M_candidates;
    BlockResult M_best_result;

    // private for singleton
    BlockGenerator();

    // not used
    BlockGenerator( const BlockGenerator & );
    BlockGenerator & operator=( const BlockGenerator & );
public:

    static
    BlockGenerator & instance();

    void generate( const rcsc::WorldModel & wm );

    const BlockResult & bestResult( const rcsc::WorldModel & wm )
      {
          generate( wm );
          return M_best_result;
      }

private:

    void clear();

    void updateTargetOpponent( const rcsc::WorldModel & wm );
    void updateBlocker( const rcsc::WorldModel & wm );

    void createCandidates( const rcsc::WorldModel & wm );

    int predictOpponentReachStep( const rcsc::AbstractPlayerObject * opponent,
                                  const int opponent_ball_reach_step,
                                  const rcsc::Vector2D & target_point,
                                  int * result_turn_step,
                                  int * result_dash_step );

    int predictSelfReachStep( const rcsc::WorldModel & wm,
                              const rcsc::Vector2D & target_point,
                              const int opponent_reach_step,
                              int * result_turn_step,
                              int * result_dash_step,
                              double * result_stamina );

    double evaluate( const rcsc::WorldModel & wm,
                     const BlockResult & result );

};

#endif
