// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <vector>
#include <fstream>
#include <string>
#include <cstdlib>
#include <cstdio>
#include <ios>
#include <sstream>

#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <rcsc/math_util.h>

#include "state_debug_client_parser.h"
#include "predict_state.h"
#include "cooperative_action.h"
#include "action_state_pair.h"
#include "variable_repository.h"
#include "file_evolutional_repository.h"
#include "param_field_evaluator.h"
#include "hold_ball.h"


#define NORMAL 0
#define POSTSCRIPT 1

#define PRINT_TYPE NORMAL
//#define PRINT_TYPE POSTSCRIPT

static const double GRID_SIZE = 0.5;


using namespace rcsc;

struct Color
{
    int r;
    int g;
    int b;

public:
    Color()
        : r( 0 ), g( 0 ), b( 0 )
    {
    }

    Color( const Color & c )
        : r( c.r ), g( c.g ), b( c.b )
    {
    }

    Color( const int r, const int g, const int b )
        : r( bound( 0, r, 255 ) ),
          g( bound( 0, g, 255 ) ),
          b( bound( 0, b, 255 ) )
    {
    }

    std::string toString() const
    {
        std::stringstream buf;
        buf << r << ' ' << g << ' ' << b;

        return buf.str();
    }

    static
    Color blend( const Color & c1, const Color & c2, const double c1_rate )
    {
        const double c2_rate = 1.0 - c1_rate;

        return Color( bound( 0, static_cast<int>( c1.r * c1_rate + c2.r * c2_rate ), 255 ),
                      bound( 0, static_cast<int>( c1.g * c1_rate + c2.g * c2_rate ), 255 ),
                      bound( 0, static_cast<int>( c1.b * c1_rate + c2.b * c2_rate) , 255 ) );
    }
};

static void print_point( const Vector2D & pos,
                         const double evaluation,
                         const double width );

static void print_players( const PredictState & state );
static void print_player( const AbstractPlayerObject & pl );
static void print_ball( const Vector2D & ball_pos );
static void print_field_lines();

static void print_header();
static void print_footer();

#if PRINT_TYPE == POSTSCRIPT
static Color get_rgb( double value );
static Color get_gradation( const std::vector<Color> & colors,
                            const double min, const double max,
                            const double value );
#endif

#if PRINT_TYPE == POSTSCRIPT
static const double MAGNIFY = 5.0;
static double convert_x( const double x );
static double convert_y( const double y );
#endif


static const std::string DLOG_FILE_PATH = "evaluation-printer.log";

static
void usage( const char * argv0 )
{
    std::cerr << "Usage: " << argv0 << " PARAM_FILE_PATH [WORLD_MODEL_FILE]"
              << std::endl;
}


int
main( int argc, char **argv )
{
    //
    // set C++ I/O environment
    //
    std::ios::sync_with_stdio();


    //
    // analyze options
    //
    std::string file_path;
    std::string param_file_path;
    bool has_file_path;
    {
        if ( argc == 3 )
        {
            param_file_path = argv[1];
            file_path = argv[2];
            has_file_path = true;
        }
        else if ( argc == 2 )
        {
            param_file_path = argv[1];
            has_file_path = false;
        }
        else
        {
            usage( argv[0] );

            return EXIT_FAILURE;
        }
    }



    //
    // prepare environment
    //
    {
        // parameter file
        if ( ! VariableRepository::initialize
             ( boost::shared_ptr<EvolutionalRepository>
               ( new FileEvolutionalRepository( param_file_path ) ) ) )
        {
            std::cerr << "Can't initalize variable repository" << std::endl;

            return EXIT_FAILURE;
        }

        const VariableRepository & rep = VariableRepository::i();
        if ( ! rep.isValid() )
        {
            std::cerr << "invaild variable repository, using default values"
                      << " for field evaluator" << std::endl;

            return EXIT_FAILURE;
        }

        // dlog
        {
            dlog.open( DLOG_FILE_PATH );
            if ( ! dlog.isOpen() )
            {
                std::cerr << "Can't open log file" << std::endl;

                return EXIT_FAILURE;
            }

            GameTime t( 1, 0 );

            dlog.setLogFlag( &t, Logger::ACTION_CHAIN, true );
        }
    }


    //
    // read world model string
    //
    std::string world_model_string;
    {
        if ( has_file_path )
        {
            std::ifstream file( file_path.c_str() );

            if ( ! file )
            {
                std::cerr << "Can't open [" << file_path << "]" << std::endl;

                return EXIT_FAILURE;
            }

            world_model_string.assign( (std::istreambuf_iterator<char>(file) ),
                                       std::istreambuf_iterator<char>() );
        }
        else
        {
            world_model_string.assign( (std::istreambuf_iterator<char>(std::cin) ),
                                       std::istreambuf_iterator<char>() );
        }
    }



    //
    // parse
    //
    StateDebugClientParser parser;
    const PredictState::Ptr state( new PredictState() );

    if ( ! parser.parse( world_model_string.c_str(), &(*state) ) )
    {
        std::cerr << "parse failed!!" << std::endl;

        return EXIT_FAILURE;
    }



    //
    // print grid point evaluation
    //
    print_header();

    const ParamFieldEvaluator evaluator;

    for ( double x = -52.0; x <= +52.0 + rcsc::EPS ; x += GRID_SIZE )
    {
        for ( double y = -33.0; y <= +33.0 + rcsc::EPS ; y += GRID_SIZE )
        {
            const Vector2D pos( x, y );

            const int unum = state->ballHolderUnum();

            StaticPredictState::Ptr s( new StaticPredictState( *state ) );
            CooperativeAction::Ptr action( new HoldBall( unum,
                                                         pos,
                                                         1,
                                                         "hold" ) );
            s->setBall( Vector2D( x, y ), Vector2D( 0.0, 0.0 ) );
            s->setBallHolderUnum( unum );
            s->setSelfPos( pos );
            state->update();

            action->setFinalAction( true );
            std::vector< ActionStatePair > path;
            path.push_back( ActionStatePair( action, s ) );

            const double evaluation = evaluator( *s, path );
            print_point( pos, evaluation, GRID_SIZE );
        }
    }

    print_field_lines();
    print_players( *state );
    print_ball( state->ball().pos() );
    print_footer();

    return EXIT_SUCCESS;
}

static
void
print_point( const Vector2D & pos,
             const double evaluation,
             const double width )
{
#if PRINT_TYPE == NORMAL
    static_cast< void >( width );

    std::cout << pos.x << ' ' << pos.y << ' ' << evaluation << std::endl;


#elif PRINT_TYPE == POSTSCRIPT
    const double wid = width / 2.0;

    Color c = get_rgb( evaluation );

    std::stringstream buf;

    buf << "newpath" << std::endl
        << convert_x(pos.x - wid) << ' ' << convert_y(pos.y - wid) << " moveto" << std::endl
        << width * MAGNIFY << " 0 rlineto" << std::endl
        << "0 " << width * MAGNIFY << " rlineto" << std::endl
        << - width * MAGNIFY << " 0 rlineto" << std::endl
        << "closepath" << std::endl;

    std::cout << static_cast<double>( c.r ) / 255 << ' '
              << static_cast<double>( c.g ) / 255 << ' '
              << static_cast<double>( c.b ) / 255 << " setrgbcolor" << std::endl
              << buf.str()
              << "fill" << std::endl;

    std::cout << "0 0 0 setrgbcolor" << std::endl
              << buf.str()
              << "stroke" << std::endl;

    std::cout << std::endl;
#endif
}


void
print_players( const PredictState & state )
{
    const PredictPlayerPtrCont::const_iterator t_end = state.allTeammates().end();
    for ( PredictPlayerPtrCont::const_iterator it = state.allTeammates().begin();
          it != t_end;
          ++it )
    {
        if ( (**it).isValid() )
        {
            print_player( **it );
        }
    }

    const AbstractPlayerCont::const_iterator o_end = state.allOpponents().end();
    for ( AbstractPlayerCont::const_iterator it = state.allOpponents().begin();
          it != o_end;
          ++it )
    {
        print_player( **it );
    }
}

void
print_player( const AbstractPlayerObject & pl )
{
#if PRINT_TYPE == POSTSCRIPT
    std::string color;

    if ( pl.side() == LEFT )
    {
        color = "255 215 0";
    }
    else if ( pl.side() == RIGHT )
    {
        color = "255 0 0";
    }
    else if ( pl.side() == NEUTRAL )
    {
        color = "127 127 127";
    }

    std::stringstream buf;

    buf << "newpath" << std::endl
        << convert_x( pl.pos().x ) << ' '
        << convert_y( pl.pos().y ) << ' '
        << 3.0 << ' '
        << "0 360 arc" << std::endl;

    std::cout << color << " setrgbcolor" << std::endl
              << buf.str() << std::endl
              << "fill" << std::endl;

    std::cout << "0 0 0 setrgbcolor" << std::endl
              << buf.str() << std::endl
              << "stroke" << std::endl;
#else
    static_cast< void >( pl );
#endif
}

void
print_ball( const Vector2D & ball_pos )
{
#if PRINT_TYPE == POSTSCRIPT

    std::cout << "255 255 255 setrgbcolor" << std::endl
              << "newpath" << std::endl
              << convert_x( ball_pos.x ) << ' '
              << convert_y( ball_pos.y ) << ' '
              << 3.0 << ' '
              << "0 360 arc" << std::endl
              << "fill" << std::endl;
#else
    static_cast<void>( ball_pos );
#endif
}

void print_field_lines()
{
#if PRINT_TYPE == POSTSCRIPT
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();

    std::cout << "255 255 255 setrgbcolor" << std::endl;

    // center circle
    std::cout << "newpath" << std::endl
              << convert_x( 0.0 ) << ' ' << convert_y( 0.0 ) << ' '
              << SP.centerCircleR() * MAGNIFY << ' '
              << "0 360 arc" << std::endl
              << "stroke" << std::endl;

    // cecter line
    std::cout << "newpath" << std::endl
              << convert_x( 0.0 ) << ' ' << convert_y( -SP.pitchHalfWidth() ) << " moveto" << std::endl
              << convert_x( 0.0 ) << ' ' << convert_y( +SP.pitchHalfWidth() ) << " lineto" << std::endl
              << "stroke" << std::endl;

    // field line
    std::cout << "newpath" << std::endl
              << convert_x( -SP.pitchHalfLength() ) << ' ' << convert_y( -SP.pitchHalfWidth() ) << " moveto" << std::endl
              << convert_x( -SP.pitchHalfLength() ) << ' ' << convert_y( +SP.pitchHalfWidth() ) << " lineto" << std::endl
              << convert_x( +SP.pitchHalfLength() ) << ' ' << convert_y( +SP.pitchHalfWidth() ) << " lineto" << std::endl
              << convert_x( +SP.pitchHalfLength() ) << ' ' << convert_y( -SP.pitchHalfWidth() ) << " lineto" << std::endl
              << "closepath" << std::endl
              << "stroke" << std::endl;

    // penalty area
    std::cout << "newpath" << std::endl
              << convert_x( -SP.pitchHalfLength() ) << ' ' << convert_y( -SP.penaltyAreaHalfWidth() ) << " moveto" << std::endl
              << convert_x( - ( SP.pitchHalfLength() - SP.penaltyAreaLength() ) ) << ' ' << convert_y( -SP.penaltyAreaHalfWidth() ) << " lineto" << std::endl
              << convert_x( - ( SP.pitchHalfLength() - SP.penaltyAreaLength() ) ) << ' ' << convert_y( +SP.penaltyAreaHalfWidth() ) << " lineto" << std::endl
              << convert_x( -SP.pitchHalfLength() ) << ' ' << convert_y( +SP.penaltyAreaHalfWidth() ) << " lineto" << std::endl
              << "closepath" << std::endl
              << "stroke" << std::endl;
    std::cout << "newpath" << std::endl
              << convert_x( SP.pitchHalfLength() ) << ' ' << convert_y( +SP.penaltyAreaHalfWidth() ) << " moveto" << std::endl
              << convert_x( + ( SP.pitchHalfLength() - SP.penaltyAreaLength() ) ) << ' ' << convert_y( +SP.penaltyAreaHalfWidth() ) << " lineto" << std::endl
              << convert_x( + ( SP.pitchHalfLength() - SP.penaltyAreaLength() ) ) << ' ' << convert_y( -SP.penaltyAreaHalfWidth() ) << " lineto" << std::endl
              << convert_x( SP.pitchHalfLength() ) << ' ' << convert_y( -SP.penaltyAreaHalfWidth() ) << " lineto" << std::endl
              << "closepath" << std::endl
              << "stroke" << std::endl;

    // goal area
    std::cout << "newpath" << std::endl
              << convert_x( -SP.pitchHalfLength() ) << ' ' << convert_y( -SP.goalAreaHalfWidth() ) << " moveto" << std::endl
              << convert_x( - ( SP.pitchHalfLength() - SP.goalAreaLength() ) ) << ' ' << convert_y( -SP.goalAreaHalfWidth() ) << " lineto" << std::endl
              << convert_x( - ( SP.pitchHalfLength() - SP.goalAreaLength() ) ) << ' ' << convert_y( +SP.goalAreaHalfWidth() ) << " lineto" << std::endl
              << convert_x( -SP.pitchHalfLength() ) << ' ' << convert_y( +SP.goalAreaHalfWidth() ) << " lineto" << std::endl
              << "closepath" << std::endl
              << "stroke" << std::endl;
    std::cout << "newpath" << std::endl
              << convert_x( SP.pitchHalfLength() ) << ' ' << convert_y( +SP.goalAreaHalfWidth() ) << " moveto" << std::endl
              << convert_x( + ( SP.pitchHalfLength() - SP.goalAreaLength() ) ) << ' ' << convert_y( +SP.goalAreaHalfWidth() ) << " lineto" << std::endl
              << convert_x( + ( SP.pitchHalfLength() - SP.goalAreaLength() ) ) << ' ' << convert_y( -SP.goalAreaHalfWidth() ) << " lineto" << std::endl
              << convert_x( SP.pitchHalfLength() ) << ' ' << convert_y( -SP.goalAreaHalfWidth() ) << " lineto" << std::endl
              << "closepath" << std::endl
              << "stroke" << std::endl;
#endif
}

void
print_header()
{
#if PRINT_TYPE == POSTSCRIPT
    std::cout << "%!PS-Adobe-3.0" << std::endl;
    std::cout << "%%BoundingBox: "
              << convert_x( -53.0 ) << ' ' << convert_y( -34.5 ) << ' '
              << convert_x( +53.0 ) << ' ' << convert_y( +34.5 ) << std::endl;
#endif
}

void
print_footer()
{
#if PRINT_TYPE == POSTSCRIPT
    std::cout << "showpage" << std::endl;
#endif
}



#if PRINT_TYPE == POSTSCRIPT
double
convert_x( const double x )
{
    return (x + 53.0) * MAGNIFY;
}

double
convert_y( const double y )
{
    return (y + 34.5) * MAGNIFY;
}
#endif


#if PRINT_TYPE == POSTSCRIPT
Color
get_rgb( double value )
{
    Color result;

    const double PENALTY_MAX = 1000.0;

    const double GOAL_EVAL_THRETHOLD = 1.0e+7 - PENALTY_MAX;
    const double CAN_SHOOT_FROM_EVAL_THRETHOLD = 2.0e+6;
    const double OVER_OFFSIDE_AND_FREE_THRESHOLD = 6.0e+5;
    const double OVER_OFFSIDE_LINE_THRESHOLD = 5.0e+5;
    const double FREE_SITUATION_NEAR_OFFSIDE_LINE_THRESHOLD = 1.0e+5;
    const double EVAL_THRETHOLD_1 = 70.0;
    const double EVAL_THRETHOLD_0 = -50.0;

    if ( value >= GOAL_EVAL_THRETHOLD )
    {
        result.r = 10;
        result.g = 10;
        result.b = 10;
    }
    else if ( value >= CAN_SHOOT_FROM_EVAL_THRETHOLD )
    {
        std::vector<Color> colors;
        colors.push_back( Color( 255, 255, 0 ) );
        colors.push_back( Color( 200, 0, 0 ) );

        result = get_gradation( colors,
                                CAN_SHOOT_FROM_EVAL_THRETHOLD + 50.0,
                                CAN_SHOOT_FROM_EVAL_THRETHOLD + 150.0,
                                value );
    }
    else if ( value >= OVER_OFFSIDE_AND_FREE_THRESHOLD )
    {
        std::vector<Color> colors;
        colors.push_back( Color( 0, 255, 0 ) );
        colors.push_back( Color( 200, 200, 0 ) );

        result = get_gradation( colors,
                                OVER_OFFSIDE_AND_FREE_THRESHOLD + 50.0,
                                OVER_OFFSIDE_AND_FREE_THRESHOLD + 100.0,
                                value );
    }
    else if ( value >= OVER_OFFSIDE_LINE_THRESHOLD )
    {
        std::vector<Color> colors;
        colors.push_back( Color( 0, 255, 255 ) );
        colors.push_back( Color( 0, 200, 0 ) );

        result = get_gradation( colors,
                                OVER_OFFSIDE_LINE_THRESHOLD,
                                OVER_OFFSIDE_LINE_THRESHOLD + 100.0,
                                value );
    }
    else if ( value >= FREE_SITUATION_NEAR_OFFSIDE_LINE_THRESHOLD - 100.0 )
    {
        std::vector<Color> colors;

        colors.push_back( Color( 0, 0, 255 ) );
        colors.push_back( Color( 0, 255, 255 ) );

        result = get_gradation( colors,
                                FREE_SITUATION_NEAR_OFFSIDE_LINE_THRESHOLD - 100.0,
                                FREE_SITUATION_NEAR_OFFSIDE_LINE_THRESHOLD,
                                value );
    }
    else if ( value >= EVAL_THRETHOLD_0 )
    {
        std::vector<Color> colors;
        colors.push_back( Color( 0, 0, 0 ) );
        colors.push_back( Color( 0, 0, 255 ) );
        colors.push_back( Color( 0, 255, 255 ) );
        colors.push_back( Color( 0, 255, 0 ) );
        colors.push_back( Color( 255, 255, 0 ) );
        colors.push_back( Color( 255, 0, 0 ) );

        result = get_gradation( colors, EVAL_THRETHOLD_0, EVAL_THRETHOLD_1, value );
    }
    else if ( value > -100.0 )
    {
        int n = (value + 100);
        n = rcsc::bound( 0, n, 255 );

        result.r = 0;
        result.g = 0;
        result.b = n;
    }
    else
    {
        result.r = 0;
        result.g = 0;
        result.b = 255;
    }

    return result;
}
#endif

Color
get_gradation( const std::vector<Color> & colors,
               const double min, const double max,
               const double value )
{
    const size_t n_colors = colors.size();

    if ( n_colors < 2 || max < min )
    {
        std::cerr << "internal error: invalid argument of get_gradation()"
                  << std::endl;

        return Color();
    }
    const int n_color_range = static_cast<int>( colors.size() - 1 );
    const double value_range = max - min;

    const double val = bound( min, value, max );
    const double rate = ( val - min ) / value_range;

    const double rate_split_width = 1.0 / ( n_color_range );

    int n = static_cast<int>( rate / rate_split_width );

    if ( n < 0 )
    {
        n = 0;
    }
    else if ( n > n_color_range )
    {
        n = n_color_range;
    }

    const Color & c1 = colors[ n ];
    const Color & c2 = colors[ n + 1 ];

    const double normalized_value = ( rate - (n * rate_split_width) ) / rate_split_width;

    return Color::blend( c1, c2, ( 1.0 - normalized_value ) );
}
