// -*-c++-*-

/*!
  \file state_debug_client_parser.cpp
  \brief state debug client paket parser class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "state_debug_client_parser.h"
#include "static_predict_state.h"
#include "predict_player_object.h"

#include <iostream>
#include <cstring>
#include <cctype>
#include <climits>
#include <cmath>
#include <cstdlib>
#include <cstdio>

using namespace rcsc;

namespace {

enum PlayerRecog {
    TEAMMATE,
    OPPONENT,
    UNKNOWN_TEAMMATE,
    UNKNOWN_OPPONENT,
    UNKNOWN_PLAYER,
    MAX_PLAYER_RECOG,
};


/*-------------------------------------------------------------------*/
/*!
  \brief msg pointer is moved untill c.
  if '"' is appeared, all character is skipped until next '"' will be
  appeared.
*/
inline
int
go_to_next_char( const char ** msg, const char c )
{
    int i = 0;
    bool no_paren = true;
    while ( **msg != '\0' )
    {
        ++i;
        ++(*msg);
        if ( **msg == '\"' )
        {
            no_paren = ! no_paren;
        }
        else if ( no_paren )
        {
            if ( **msg == c )
            {
                break;
            }
        }
    }
    return i;
}
}


///////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
bool
StateDebugClientParser::parse( const char * msg,
                               StaticPredictState * holder )
{
    M_side = 'n';
    M_unum = 0;
    M_cycle = -1;

    int	version = 0;
    int	n_read = 0;

    bool result = false;

    if ( std::sscanf( msg,
                      " ( ( debug ( format-version %d ) ) %n" ,
                      &version, &n_read ) == 1 )
    {
        switch( version ) {
        case 1:
            std::cerr
                << "debug message version 1 is not supported anymore."
                << std::endl;
            break;
        case 2:
        case 3:
            msg += n_read;
            result = parseV2( msg, holder );
            break;
        default:
            break;
        }
    }

    holder->update();

    return result;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse debug client message body.
*/
bool
StateDebugClientParser::parseV2( const char * msg,
                            StaticPredictState * debug_view )
{
    if ( msg[0] == '(' && msg[1] == '(' )
    {
        ++msg;
    }

    // read all S-Expression token that is contained in 'msg;.
    while ( *msg != '\0' && *msg != ')' )
    {
        // if first character is not '(', it may be illegal token.
        // try to skip to the next '('.
        if ( *msg != '(' )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " bad message [" << std::string( msg, 16 ) << "]"
                      << std::endl;
            // skip to next paren
            go_to_next_char( &msg, '(' );
            continue;
        }

        // read token ID
        char id[32];
        if ( std::sscanf( msg,
                          " ( %s ",
                          id ) != 1 )
        {
            std::cerr << "StateDebugClientParser::parse(). "
                      << "Bad message ID [" << std::string( msg, 16 ) << "]"
                      << std::endl;
            ++msg;
            continue;
        }

        int n_read = 0;

        if ( ! std::strcmp( id, "t" )
             || ! std::strcmp( id, "o" )
             || ! std::strcmp( id, "ut" )
             || ! std::strcmp( id, "uo" )
             || ! std::strcmp( id, "u" )
             )
        {
            // other players info
            if ( ( n_read = parsePlayer( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parsePlayer" << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "line" ) )
        {
            // line draw request
            if ( ( n_read = parseLine( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseLine ["
                          << std::string( msg, 32 )
                          << "]"
                          << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "tri" ) )
        {
            // triangle draw request
            if ( ( n_read = parseTriangle( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseTriangle ["
                          << std::string( msg, 32 )
                          << "]"
                          << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "rect" ) )
        {
            // rectanble draw request
            if ( ( n_read = parseRectangle( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseRectangle["
                          << std::string( msg, 32 )
                          << "]"
                          << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "circle" ) )
        {
            // line draw request
            if ( ( n_read = parseCircle( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseCircle["
                          << std::string( msg, 32 )
                          << "]" << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "s" ) )
        {
            // self info
            if ( ( n_read = parseSelf( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseSelf" << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "b" ) )
        {
            // ball info
            if ( ( n_read = parseBall( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseBall" << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "time" ) )
        {
            // time info of this message
            if ( ( n_read = parseTime( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseTime" << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "target-teammate" ) )
        {
            if ( ( n_read = parseTargetTeammate( msg, debug_view ) ) == 0  )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseTargetTeammate" << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "target-point" ) )
        {
            if ( ( n_read = parseTargetPoint( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseTargetPoint" << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "say" ) )
        {
            if ( ( n_read = parseSayMessage( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseSayMessage" << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "hear" ) )
        {
            if ( ( n_read = parseHearMessage( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseHearMessage" << std::endl;
                return false;
            }
        }
        else if ( ! std::strcmp( id, "message" ) )
        {
            // misc. message
            if ( ( n_read = parseMessage( msg, debug_view ) ) == 0 )
            {
                std::cerr << "StateDebugClientParser::parse(). "
                          << "Error parseMessage" << std::endl;
                return false;
            }
        }
        else if ( id[0] == '(' )
        {
            ++msg;
            continue;
        }
        else
        {
            std::cerr << "StateDebugClientParser::parse(). "
                      << "Unsupported id [" << id << "]"
                      << std::endl;
            go_to_next_char( &msg, '(' );
            continue;
        }

        msg += n_read;

        if ( msg[0] != '(' )
        {
            go_to_next_char( &msg, '(' );
        }
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse time token
  (time <long>)
  \retval 0 if failed.
  \return the length.of read byte.
*/
int
StateDebugClientParser::parseTime( const char * tok,
                              StaticPredictState * debug_view )
{
    int n_read = 0;

    int cycle = 0;
    if ( std::sscanf( tok,
                      " ( time %d ) %n",
                      &cycle,
                      &n_read ) != 1
         || n_read == 0 )
    {
        std::cout << "StateDebugClientParser::parseTime. Failed. ["
                  << std::string( tok, 16 )
                  << "]" << std::endl;
        return 0;
    }

#if 0
    std::cout << "StateDebugClientParser::parserTime. time = " << cycle
              << "  offset=" << n_read << std::endl;
#endif
    M_cycle = cycle;

    debug_view->setGameTime( M_cycle );

    return n_read;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse ball token
  (b <x> <y>[ <vx> <vy>][ (c "<comment>")])
  \retval 0 if failed.
  \return the length.of read byte.
*/
int
StateDebugClientParser::parseBall( const char * tok,
                                   StaticPredictState * debug_view )
{
    const char * start = tok;
    char * next;

    while ( *tok != '\0' && *tok == '(' ) ++tok;
    while ( *tok != '\0' && *tok == ' ' ) ++tok;
    while ( *tok != '\0' && *tok != ' ' ) ++tok; // skip 'b'

    const double x = std::strtod( tok, &next );
    if ( x == -HUGE_VAL || x == HUGE_VAL ) return 0;
    tok = next;
    const double y = std::strtod( tok, &next );
    if ( y == -HUGE_VAL || y == HUGE_VAL ) return 0;
    tok = next;

    while ( *tok != '\0' && *tok == ' ' ) ++tok;

    double vx = HUGE_VAL;
    double vy = HUGE_VAL;
    if ( std::isdigit( *tok )
         || *tok == '-'
         || *tok == '.' )
    {
        vx = std::strtod( tok, &next );
        if ( vx == -HUGE_VAL || vx == HUGE_VAL ) return 0;
        tok = next;
        vy = std::strtod( tok, &next );
        if ( vy == -HUGE_VAL || vy == HUGE_VAL ) return 0;
        tok = next;

        while ( *tok != '\0' && *tok == ' ' ) ++tok;
    }

    char comment[256+1];
    std::strcpy( comment, "" );
    if ( *tok == '('
         && *(tok+1) == 'c' )
    {
        int n_read = 0;
        if ( std::sscanf( tok,
                          " ( c \"%256[^\"]\" ) %n" ,
                          comment,
                          &n_read ) == 1
             && n_read != 0 )
        {
            tok += n_read;
        }
    }

    while ( *tok != '\0' && *tok != ')' ) ++tok;
    // skip this token
    while ( *tok != '\0' && ( *tok == ' ' || *tok == ')' ) ) ++tok;
#if 0
    std::cout << "StateDebugClientParser::parserBall. all:"
              << " x=" << x
              << " y=" << y
              << " vx=" << vx
              << " vy=" << vy
              << " c=[" << comment << "]"
              << std::endl;
#endif
    if ( vx != HUGE_VAL )
    {
        debug_view->setBallVel( Vector2D( vx, vy ) );
    }

#if 0
    if ( comment[0] != '\0' )
    {
        ball->comment_.assign( comment );
    }
#endif

    debug_view->setBallPos( Vector2D( x, y ) );

    return tok - start;
}

/*-------------------------------------------------------------------*/
/*!
  (s <side> <unum> <x> <y> <vx> <vy> <body> <neck>[ (c "<comment>")])
*/
int
StateDebugClientParser::parseSelf( const char * tok,
                                   StaticPredictState * debug_view )
{
    const char * start = tok;
    char * next;

    while ( *tok != '\0' && *tok == '(' ) ++tok;
    while ( *tok != '\0' && *tok == ' ' ) ++tok;
    while ( *tok != '\0' && *tok != ' ' ) ++tok; // skip 's'

    while ( *tok != '\0' && *tok == ' ' ) ++tok; // goto <side>
    const char side = *tok;
    if ( side != 'l' && side != 'r' )
    {
        std::cerr << "StateDebugClientParser::parseSelf. Error. unknown side char ["
                  << side << "]" << std::endl;
        return 0;
    }

    ++tok; // skip side

    const long unum = std::strtol( tok, &next, 10 );
    if ( unum == LONG_MIN || unum == LONG_MAX ) return 0;
    tok = next;
    double x = std::strtod( tok, &next );
    if ( x == -HUGE_VAL || x == HUGE_VAL ) return 0;
    tok = next;
    double y = std::strtod( tok, &next );
    if ( y == -HUGE_VAL || y == HUGE_VAL ) return 0;
    tok = next;
    double vx = std::strtod( tok, &next );
    if ( vx == -HUGE_VAL || vx == HUGE_VAL ) return 0;
    tok = next;
    double vy = std::strtod( tok, &next );
    if ( vy == -HUGE_VAL || vy == HUGE_VAL ) return 0;
    tok = next;
    double body = std::strtod( tok, &next );
    if ( body == -HUGE_VAL || body == HUGE_VAL ) return 0;
    tok = next;
    double neck = std::strtod( tok, &next );
    if ( neck == -HUGE_VAL || neck == HUGE_VAL ) return 0;
    tok = next;

    while ( *tok != '\0' && *tok == ' ' ) ++tok;

    char comment[256 + 1];
    std::strcpy( comment, "" );

    if ( *tok == '(' )
    {
        int n_read = 0;
        if ( std::sscanf( tok,
                          " ( c \"%256[^\"]\" ) %n" ,
                          comment,
                          &n_read ) == 1
             && n_read != 0 )
        {
            tok += n_read;
        }
    }

    while ( *tok != '\0' && *tok != ')' ) ++tok;
    // skip this token
    while ( *tok != '\0' && ( *tok == ' ' || *tok == ')' ) ) ++tok;

#if 0
    std::cout << "StateDebugClientParser::parserSelf. Self:"
              << " side=" << side
              << " unum=" << unum
              << " x=" << x
              << " y=" << y
              << " vx=" << vx
              << " vy=" << vy
              << " b=" << body
              << " n=" << neck
              << " c=[" << comment << "]"
              << std::endl;
#endif

    M_side = side;
    M_unum = static_cast< int >( unum );
    // create new instance
#if 0
    boost::shared_ptr< DebugViewData::SelfT >
        self( new DebugViewData::SelfT( M_side,
                                        M_unum,
                                        static_cast< float >( x ),
                                        static_cast< float >( y ),
                                        static_cast< float >( vx ),
                                        static_cast< float >( vy ),
                                        static_cast< float >( body ),
                                        static_cast< float >( neck ),
                                        comment ) );
    debug_view->setSelf( self );
#else
    SideID self_side;
    switch( M_side )
    {
    case 'l':
        self_side = LEFT;
        break;

    case 'r':
        self_side = RIGHT;
        break;

    default:
        self_side = NEUTRAL;
        break;
    }

    debug_view->setSelf( self_side,
                         M_unum,
                         Vector2D( x, y ),
                         Vector2D( vx, vy ),
                         body,
                         neck );
#endif
    return tok - start;
}

/*-------------------------------------------------------------------*/
/*!
  ({t|o} <unum> <x> <y>[ (bd <body>)][ (c "<comment>")])
  ({ut|uo|u} <x> <y>[ (bd <body>)][ (c "<comment>")])
*/
int
StateDebugClientParser::parsePlayer( const char * tok,
                                     StaticPredictState * debug_view )
{
    const char * start = tok;
    char * next;

    while ( *tok != '\0' && *tok == '(' ) ++tok;
    while ( *tok != '\0' && *tok == ' ' ) ++tok;

    // read side
    PlayerRecog recog_type = MAX_PLAYER_RECOG;
    if ( *tok != 'u' )
    {
        if ( *tok == 't' ) recog_type = TEAMMATE;
        if ( *tok == 'o' ) recog_type = OPPONENT;
    }
    else
    {
        if ( *(tok+1) == 't' ) recog_type = UNKNOWN_TEAMMATE;
        if ( *(tok+1) == 'o' ) recog_type = UNKNOWN_OPPONENT;
        if ( *(tok+1) == ' ' ) recog_type = UNKNOWN_PLAYER;
    }
    while ( *tok != '\0' && *tok != ' ' ) ++tok; // skip side info


    // read unum, if known type
    long unum = 0;
    if ( recog_type == TEAMMATE
         || recog_type == OPPONENT )
    {
        unum = std::strtol( tok, &next, 10 );
        if ( unum == LONG_MIN || unum == LONG_MAX ) return 0;
        tok = next;
    }
    const double x = std::strtod( tok, &next );
    if ( x == -HUGE_VAL || x == HUGE_VAL ) return 0;
    tok = next;
    const double y = std::strtod( tok, &next );
    if ( y == -HUGE_VAL || y == HUGE_VAL ) return 0;
    tok = next;

    while ( *tok != '\0' && *tok == ' ' ) ++tok;

    float body = -360.0;
    char comment[256+1];
    std::strcpy( comment, "" );

    while ( *tok != '\0' && *tok == '(' )
    {
        int n_read = 0;

        if ( std::sscanf( tok,
                          " ( bd %f ) %n",
                          &body,
                          &n_read ) == 1
             && n_read != 0 )
        {
            tok += n_read;
            while ( *tok != '\0' && *tok == ' ' ) ++tok;
        }
        else if ( std::sscanf( tok,
                               " ( c \"%256[^\"]\" ) %n" ,
                               comment,
                               &n_read ) == 1
                  && n_read != 0 )
        {
            tok += n_read;
            while ( *tok != '\0' && *tok == ' ' ) ++tok;
        }
        else
        {
            std::cerr << "parserPlayer. unsupported option. ["
                      << std::string( tok, 5 ) << "]" << std::endl;
            break;
        }
    }

    while ( *tok != '\0' && *tok != ')' ) ++tok;
    // skip this token
    while ( *tok != '\0' && ( *tok == ' ' || *tok == ')' ) ) ++tok;

#if 0
    boost::shared_ptr< DebugViewData::PlayerT >
        player( new DebugViewData::PlayerT( unum,
                                            static_cast< float >( x ),
                                            static_cast< float >( y ),
                                            body,
                                            comment ) );
#else
    if ( unum == 0 )
    {
        unum = Unum_Unknown;
    }

    SideID teammate_side;
    SideID opponent_side;

    switch( M_side )
    {
    case 'l':
        teammate_side = LEFT;
        opponent_side = RIGHT;
        break;

    case 'r':
        teammate_side = RIGHT;
        opponent_side = LEFT;
        break;

    default:
        teammate_side = NEUTRAL;
        opponent_side = NEUTRAL;
        break;
    }

    SideID pl_side;
    switch ( recog_type ) {
    case TEAMMATE:
    case UNKNOWN_TEAMMATE:
        pl_side = teammate_side;
        break;
    case OPPONENT:
    case UNKNOWN_OPPONENT:
        pl_side = opponent_side;
        break;
    case UNKNOWN_PLAYER:
    default:
        pl_side = NEUTRAL;
        break;
    }

    PredictPlayerObject::Ptr player( new PredictPlayerObject( pl_side,
                                                              static_cast<int>( unum ),
                                                              false,
                                                              Vector2D( x, y ),
                                                              AngleDeg( body ) ) );
    switch ( recog_type ) {
    case TEAMMATE:
    case UNKNOWN_TEAMMATE:
        debug_view->addTeammate( player );
        break;
    case OPPONENT:
    case UNKNOWN_OPPONENT:
        debug_view->addOpponent( player );
        break;
    case UNKNOWN_PLAYER:
        debug_view->addUnknownPlayer( player );
        break;
    default:
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error. Illegal Player ID. pos = ("
                  << x  << ", "<< y << ")"
                  << std::endl;
        break;
    }
#endif
    return tok - start;
}

/*-------------------------------------------------------------------*/
/*!
  (line <x1> <y1> <x2> <y2>)
*/
int
StateDebugClientParser::parseLine( const char * tok,
                                   StaticPredictState * )
{
    const char * start = tok;
    char * next;

    while ( *tok != '\0' && *tok == '(' ) ++tok;
    while ( *tok != '\0' && *tok == ' ' ) ++tok;
    while ( *tok != '\0' && *tok != ' ' ) ++tok; // skip 'line'

    const double x1 = std::strtod( tok, &next );
    if ( x1 == -HUGE_VAL || x1 == HUGE_VAL ) return 0;
    tok = next;
    const double y1 = std::strtod( tok, &next );
    if ( y1 == -HUGE_VAL || y1 == HUGE_VAL ) return 0;
    tok = next;
    const double x2 = std::strtod( tok, &next );
    if ( x2 == -HUGE_VAL || x2 == HUGE_VAL ) return 0;
    tok = next;
    const double y2 = std::strtod( tok, &next );
    if ( y2 == -HUGE_VAL || y2 == HUGE_VAL ) return 0;
    tok = next;

    while ( *tok != '\0' && *tok != ')' ) ++tok;
    // skip this token
    while ( *tok != '\0' && ( *tok == ' ' || *tok == ')' ) ) ++tok;
#if 0
    std::cout << "StateDebugClientParser::parseLine. Line:"
              << " x1=" << x1
              << " y1=" << y1
              << " x2=" << x2
              << " y2=" << y2
              << std::endl;
#endif
#if 0
    debug_view->addLine( static_cast< float >( x1 ),
                         static_cast< float >( y1 ),
                         static_cast< float >( x2 ),
                         static_cast< float >( y2 ) );
#endif

    return tok - start;
}

/*-------------------------------------------------------------------*/
/*!
  (tri <x1> <y1> <x2> <y2> <x3> <y3>)
*/
int
StateDebugClientParser::parseTriangle( const char * tok,
                                       StaticPredictState * )
{
    const char * start = tok;
    char * next;

    while ( *tok != '\0' && *tok == '(' ) ++tok;
    while ( *tok != '\0' && *tok == ' ' ) ++tok;
    while ( *tok != '\0' && *tok != ' ' ) ++tok; // skip 'tri'

    const double x1 = std::strtod( tok, &next );
    if ( x1 == -HUGE_VAL || x1 == HUGE_VAL ) return 0;
    tok = next;
    const double y1 = std::strtod( tok, &next );
    if ( y1 == -HUGE_VAL || y1 == HUGE_VAL ) return 0;
    tok = next;
    const double x2 = std::strtod( tok, &next );
    if ( x2 == -HUGE_VAL || x2 == HUGE_VAL ) return 0;
    tok = next;
    const double y2 = std::strtod( tok, &next );
    if ( y2 == -HUGE_VAL || y2 == HUGE_VAL ) return 0;
    tok = next;
    const double x3 = std::strtod( tok, &next );
    if ( x3 == -HUGE_VAL || x3 == HUGE_VAL ) return 0;
    tok = next;
    const double y3 = std::strtod( tok, &next );
    if ( y3 == -HUGE_VAL || y3 == HUGE_VAL ) return 0;
    tok = next;

    while ( *tok != '\0' && *tok != ')' ) ++tok;
    // skip this token
    while ( *tok != '\0' && ( *tok == ' ' || *tok == ')' ) ) ++tok;

#if 0
    debug_view->addTriangle( static_cast< float >( x1 ),
                             static_cast< float >( y1 ),
                             static_cast< float >( x2 ),
                             static_cast< float >( y2 ),
                             static_cast< float >( x3 ),
                             static_cast< float >( y3 ) );
#endif

    return tok - start;
}

/*-------------------------------------------------------------------*/
/*!
  (rect <left_x> <top_y> <right_x> <bottom_y>)
*/
int
StateDebugClientParser::parseRectangle( const char * tok,
                                        StaticPredictState * )
{
    const char * start = tok;
    char * next;

    while ( *tok != '\0' && *tok == '(' ) ++tok;
    while ( *tok != '\0' && *tok == ' ' ) ++tok;
    while ( *tok != '\0' && *tok != ' ' ) ++tok; // skip 'rect'

    const double x1 = std::strtod( tok, &next );
    if ( x1 == -HUGE_VAL || x1 == HUGE_VAL ) return 0;
    tok = next;
    const double y1 = std::strtod( tok, &next );
    if ( y1 == -HUGE_VAL || y1 == HUGE_VAL ) return 0;
    tok = next;
    const double x2 = std::strtod( tok, &next );
    if ( x2 == -HUGE_VAL || x2 == HUGE_VAL ) return 0;
    tok = next;
    const double y2 = std::strtod( tok, &next );
    if ( y2 == -HUGE_VAL || y2 == HUGE_VAL ) return 0;
    tok = next;

    while ( *tok != '\0' && *tok != ')' ) ++tok;
    // skip this token
    while ( *tok != '\0' && ( *tok == ' ' || *tok == ')' ) ) ++tok;

#if 0
    debug_view->addRectangle( static_cast< float >( x1 ),
                              static_cast< float >( y1 ),
                              static_cast< float >( x2 ),
                              static_cast< float >( y2 ) );
#endif

    return tok - start;
}


/*-------------------------------------------------------------------*/
/*!
  (circle <center_x> <center_y> <radius>)
*/
int
StateDebugClientParser::parseCircle( const char * tok,
                                     StaticPredictState * )
{
    const char * start = tok;
    char * next;

    while ( *tok != '\0' && *tok == '(' ) ++tok;
    while ( *tok != '\0' && *tok == ' ' ) ++tok;
    while ( *tok != '\0' && *tok != ' ' ) ++tok; // skip 'rect'

    const double x = std::strtod( tok, &next );
    if ( x == -HUGE_VAL || x == HUGE_VAL ) return 0;
    tok = next;
    const double y = std::strtod( tok, &next );
    if ( y == -HUGE_VAL || y == HUGE_VAL ) return 0;
    tok = next;
    const double r = std::fabs( std::strtod( tok, &next ) );
    if ( r == -HUGE_VAL || r == HUGE_VAL ) return 0;
    tok = next;

    while ( *tok != '\0' && *tok != ')' ) ++tok;
    // skip this token
    while ( *tok != '\0' && ( *tok == ' ' || *tok == ')' ) ) ++tok;

#if 0
    debug_view->addCircle( static_cast< float >( x ),
                           static_cast< float >( y ),
                           static_cast< float >( r ) );
#endif

    return tok - start;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
StateDebugClientParser::parseTargetTeammate( const char * tok,
                                             StaticPredictState * )
{
    int n_read = 0;

    int number = 0;
    if ( std::sscanf( tok,
                      " ( target-teammate %d ) %n",
                      &number,
                      &n_read ) != 1
         || n_read == 0 )
    {
        std::cout << "StateDebugClientParser::parseTargetTeammate. Failed ["
                  << std::string( tok, 16 )
                  << "]"  << std::endl;
        return 0;
    }
#if 0
    std::cout << "StateDebugClientParser::parseTagetTeammats. target teammate = "
              << number << std::endl;
#endif
    if ( 0 < number && number < 12 )
    {
#if 0
        debug_view->setTargetTeammate( number );
#endif
    }

    return n_read;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
StateDebugClientParser::parseTargetPoint( const char * tok,
                                          StaticPredictState * )
{
    int n_read = 0;

    double x = 0.0;
    double y = 0.0;
    if ( std::sscanf( tok,
                      " ( target-point %lf %lf ) %n",
                      &x, &y,
                      &n_read ) != 2
         || n_read == 0 )
    {
        std::cout << "StateDebugClientParser::parseTargetPoint. Failed. ["
                  << std::string( tok, 16 )
                  << "]"  << std::endl;
        return 0;
    }
#if 0
    std::cout << "StateDebugClientParser::parseTargetPoint. target point = "
              << x << ", " << y << std::endl;
#endif
#if 0
    debug_view->setTargetPoint( x, y );
#endif

    return n_read;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
StateDebugClientParser::parseMessage( const char * tok,
                                      StaticPredictState * )
{
    int n_read = 0;

    char message[1024];
    std::strcpy( message, "" );
    if ( std::sscanf( tok,
                      " ( message \"%1023[^\"]\" ) %n" ,
                      message,
                      &n_read ) != 1
         || n_read == 0 )
    {
        std::cout << "StateDebugClientParser::parseMessage. Failed. ["
                  << std::string( tok, 16 )
                 << "]"  << std::endl;
        return 0;
    }
#if 0
    std::cout << "StateDebugClientParser::parseMessage. message = ["
              << message << "]" << std::endl;
#endif
#if 0
    debug_view->setMessage( message );
#endif

//     if ( std::strstr( message, "Say[" ) != NULL )
//     {
//         debug_view->setHasSayMessage( true );
//     }

    return n_read;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
StateDebugClientParser::parseSayMessage( const char * tok,
                                         StaticPredictState * )
{
    int n_read = 0;

    char message[1024];
    std::strcpy( message, "" );
    if ( std::sscanf( tok,
                      " ( say \"%1023[^\"]\" ) %n" ,
                      message,
                      &n_read ) != 1
         || n_read == 0 )
    {
        std::cout << "StateDebugClientParser::parseSayMessage. Failed. ["
                  << std::string( tok, 16 )
                 << "]"  << std::endl;
        return 0;
    }

#if 0
    debug_view->setSayMessage( message );
#endif

    return n_read;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
StateDebugClientParser::parseHearMessage( const char * tok,
                                          StaticPredictState * )
{
    const char * start = tok;

    while ( *tok != '\0' && *tok == ' ' ) ++tok;
    while ( *tok != '\0' && *tok == '(' ) ++tok;
    while ( *tok != '\0' && *tok == ' ' ) ++tok;
    while ( *tok != '\0' && *tok != ' ' && *tok != ')' ) ++tok;  // skip 'hear'

    while ( *tok != '\0' )
    {
        if ( *tok == ')' )
        {
            ++tok;
            break;
        }

        int n_read = 0;
        int unum = -1;
        char msg[256];

        if ( std::sscanf( tok, " ( %d \"%255[^\"]\" ) %n ",
                          &unum, msg, &n_read ) != 2 )
        {
            std::cerr << "StateDebugClientParser::parseHearMessage()"
                      << " Illegal message [" << std::string( start, 16 ) << "]"
                      << std::endl;
            return 0;
        }
        tok += n_read;

#if 0
        debug_view->addHearMessage( unum, msg );
#endif

        while ( *tok != '\0' && *tok == ' ' ) ++tok;
    }

    while ( *tok != '\0' && *tok == ' ' ) ++tok;

    return tok - start;
}
