// -*-c++-*-

/*!
  \file static predict_state.cpp
  \brief static predicted field state class Source File
*/

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA, Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "static_predict_state.h"
#include <rcsc/common/server_param.h>
#include <algorithm>
#include <boost/scoped_ptr.hpp>

namespace rcsc {

const int StaticPredictState::VALID_PLAYER_THRESHOLD = 8;

/*-------------------------------------------------------------------*/
/*!

 */
StaticPredictState::StaticPredictState()
    :
      M_game_time( -1, 0 ),
      M_spend_time( 0 ),
      M_ball_holder_unum( Unum_Unknown ),
      M_ball(),
      M_self_unum( Unum_Unknown ),
      M_our_side( NEUTRAL ),
      M_all_teammates(),
      M_our_defense_line_x( 0.0 ),
      M_our_offense_player_line_x( 0.0 ),
      M_their_defense_player_line_x( 0.0 ),
      M_offside_line_x( 0.0 )
{
    init();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::setBall( const Vector2D & pos, const Vector2D & vel )
{
    M_ball.assign( pos, vel );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::setBallPos( const Vector2D & pos )
{
    M_ball.setPos( pos );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::setBallVel( const Vector2D & vel )
{
    M_ball.setVel( vel );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::setSpendTime( const unsigned long spend_time )
{
    M_spend_time = spend_time;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::setGameTime( const long cycle )
{
    M_game_time.assign( cycle, 0 );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::setSelf( const SideID side,
                             const int unum,
                             const Vector2D & pos,
                             const Vector2D &,
                             const AngleDeg body,
                             const AngleDeg )
{
    if ( ! ( 1 <= unum && unum <= MAX_PLAYER ) )
    {
        std::cerr << "internal error: bad unum of self" << std::endl;
        return;
    }

    M_our_side = side;
    M_self_unum = unum;

    M_all_teammates[unum - 1]
        = PredictPlayerObject::Ptr( new PredictPlayerObject( side,
                                                             unum,
                                                             true,
                                                             pos,
                                                             body ) );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::setSelfPos( const Vector2D & pos )
{
    M_all_teammates[ M_self_unum - 1 ]
        = PredictPlayerObject::Ptr( new PredictPlayerObject( M_our_side,
                                                             M_self_unum,
                                                             true,
                                                             pos,
                                                             M_all_teammates[ M_self_unum - 1 ]->body() ) );
}


/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::addTeammate( const PredictPlayerObject::Ptr & player )
{
    const int unum = player->unum();
    if ( unum == Unum_Unknown )
    {
        return;
    }

    if ( ! ( 1 <= unum && unum <= MAX_PLAYER ) )
    {
        std::cerr << "internal error: bad unum of teammate" << std::endl;
        return;
    }

    M_all_teammates[ unum - 1 ] = player;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::addOpponent( const PredictPlayerObject::Ptr & player )
{
    const int unum = player->unum();
    if ( unum == Unum_Unknown )
    {
        return;
    }

    if ( ! ( 1 <= unum && unum <= MAX_PLAYER ) )
    {
        std::cerr << "internal error: bad unum of opponent" << std::endl;
        return;
    }

    M_opponents[ unum - 1 ] = player;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::addUnknownPlayer( const PredictPlayerObject::Ptr & )
{
    // ignore unknown player
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::setBallHolderUnum( const int ball_holder_unum )
{
    M_ball_holder_unum = ball_holder_unum;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::update()
{
    if ( M_ball_holder_unum == Unum_Unknown )
    {
        M_ball_holder_unum = M_self_unum;
    }

    updateLinesAndBallHolder();


    //
    // update M_opponents_player_cont from M_opponents
    //
    {
        M_opponents_player_cont.clear();

        const PredictPlayerPtrCont::const_iterator end = M_opponents.end();
        for ( PredictPlayerPtrCont::const_iterator it = M_opponents.begin();
              it != end; ++it )
        {
            if ( ! (*it)->isValid() )
            {
                continue;
            }

            M_opponents_player_cont.push_back( &(**it) );
        }
    }

    // XXX: update M_opponents_from_self
}


/*-------------------------------------------------------------------*/
/*!

 */
void
StaticPredictState::init()
{
    //
    // initialize spend_time
    //
    M_spend_time = 0;

    //
    // initialize ball pos & vel
    //
    M_ball.assign( Vector2D( 0.0, 0.0 ), Vector2D( 0.0, 0.0 ) );

    //
    // initialize self_unum
    //
    M_self_unum = Unum_Unknown;


    //
    // initialize teammates, opponent
    //
    M_all_teammates.reserve( MAX_PLAYER );

    for ( int i = 1; i <= MAX_PLAYER; ++i )
    {
        M_all_teammates.push_back( PredictPlayerObject::Ptr
                                   ( new PredictPlayerObject() ) );
        M_opponents.push_back( PredictPlayerObject::Ptr
                                   ( new PredictPlayerObject() ) );
    }

    update();
}

void
StaticPredictState::updateLinesAndBallHolder()
{
    if ( M_self_unum != Unum_Unknown )
    {
        M_ball_holder_unum = M_self_unum;
        double min_dist = ( M_all_teammates[ M_self_unum - 1 ]->pos() - M_ball.pos() ).r();

        const PredictPlayerPtrCont::const_iterator t_end = M_all_teammates.end();
        for ( PredictPlayerPtrCont::const_iterator it = M_all_teammates.begin();
              it != t_end;
              ++it )
        {
            if ( ! (*it)->isValid() )
            {
                continue;
            }

            const double dist = ( (*it)->pos() - M_ball.pos() ).r();

            if ( dist < min_dist )
            {
                min_dist = dist;
                M_ball_holder_unum =(*it)->unum();
            }
        }
    }


    //
    // update our line and ball holder
    //
    {
        M_our_offense_player_line_x = ServerParam::i().ourTeamGoalLineX();

        double first = 0.0;
        double second = 0.0;

        const PredictPlayerPtrCont::const_iterator t_end = M_all_teammates.end();
        for ( PredictPlayerPtrCont::const_iterator it = M_all_teammates.begin();
              it != t_end;
              ++it )
        {
            if ( ! (*it)->isValid() )
            {
                continue;
            }

            const double x = (*it)->pos().x;

            if ( M_our_offense_player_line_x < x )
            {
                M_our_offense_player_line_x = x;
            }

            if ( x < second )
            {
                second = x;
                if ( second < first )
                {
                    std::swap( first, second );
                }
            }
        }

        M_our_defense_line_x = second;
    }


    //
    // update their line
    //
    {
        double first = 0.0;
        double second = 0.0;

        const PredictPlayerPtrCont::const_iterator t_end = M_opponents.end();
        for ( PredictPlayerPtrCont::const_iterator it = M_opponents.begin();
              it != t_end;
              ++it )
        {
            if ( ! (*it)->isValid() )
            {
                continue;
            }

            const double x = (*it)->pos().x;

            if ( x > second )
            {
                second = x;
                if ( second > first )
                {
                    std::swap( first, second );
                }
            }
        }

        M_their_defense_player_line_x = second;
        M_offside_line_x = std::max( M_their_defense_player_line_x, M_ball.pos().x );
    }
}


/*-------------------------------------------------------------------*/
/*!

 */
AbstractPlayerCont
StaticPredictState::getPlayerCont( const PlayerPredicate * predicate ) const
{
    boost::scoped_ptr< const PlayerPredicate > pred( predicate );

    AbstractPlayerCont ret;

    if ( ! pred )
    {
        return ret;
    }

    const PredictPlayerPtrCont::const_iterator t_end = M_all_teammates.end();
    for ( PredictPlayerPtrCont::const_iterator it = M_all_teammates.begin();
          it != t_end;
          ++it )
    {
        if ( (*pred)( **it ) )
        {
            ret.push_back( &(**it) );
        }
    }

    const PredictPlayerPtrCont::const_iterator o_end = M_opponents.end();
    for ( PredictPlayerPtrCont::const_iterator it = M_opponents.begin();
          it != o_end;
          ++it )
    {
        if ( (*pred)( **it ) )
        {
            ret.push_back( &(**it) );
        }
    }

    return ret;
}

}
