// -*-c++-*-

/*!
  \file static_predict_state.h
  \brief static predicted field state class Header File
*/

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA, Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_STATIC_PREDICT_STATE_H
#define RCSC_PLAYER_STATIC_PREDICT_STATE_H

#include "predict_player_object.h"
#include "predict_ball_object.h"

#include <rcsc/player/world_model.h>
#include <rcsc/player/abstract_player_object.h>
#include <rcsc/player/player_predicate.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/vector_2d.h>
#include <rcsc/common/audio_memory.h>

#include <boost/shared_ptr.hpp>

#include <algorithm>

namespace rcsc {

class StaticPredictState {

public:
    static const int VALID_PLAYER_THRESHOLD;

    typedef boost::shared_ptr< StaticPredictState > Ptr; //!< pointer type alias
    typedef boost::shared_ptr< const StaticPredictState > ConstPtr; //!< const pointer type alias

private:
    GameTime M_game_time;

    unsigned long M_spend_time;

    int M_ball_holder_unum;

    PredictBallObject M_ball;

    int M_self_unum;
    SideID M_our_side;

    PredictPlayerPtrCont M_all_teammates;
    PredictPlayerPtrCont M_opponents;
    AbstractPlayerCont M_opponents_player_cont;


    // XXX: not used
    PlayerPtrCont M_opponents_from_self;

    double M_our_defense_line_x;
    double M_our_offense_player_line_x;
    double M_their_defense_player_line_x;
    double M_offside_line_x;

public:
    StaticPredictState();

private:
    void init();
    void updateLinesAndBallHolder();

public:
    void setSpendTime( const unsigned long spend_time );


    void setGameTime( long cycle );
    void setBall( const Vector2D & pos, const Vector2D & vel );
    void setBallPos( const Vector2D & pos );
    void setBallVel( const Vector2D & vel );

    void setSelf( const SideID side,
                  const int self_unum,
                  const Vector2D & pos,
                  const Vector2D & vel,
                  const AngleDeg body,
                  const AngleDeg neck );
    void setSelfPos( const Vector2D & pos );

    void addTeammate( const PredictPlayerObject::Ptr & player );
    void addOpponent( const PredictPlayerObject::Ptr & player );
    void addUnknownPlayer( const PredictPlayerObject::Ptr & player );

    void setBallHolderUnum( const int ball_holder_unum );

    void update();

public:
    unsigned long spendTime() const
      {
          return M_spend_time;
      }

    int ballHolderUnum() const
      {
          return M_ball_holder_unum;
      }

    const AbstractPlayerObject * ballHolder() const
      {
          if ( ballHolderUnum() == Unum_Unknown )
          {
              return static_cast<AbstractPlayerObject *>( 0 );
          }
          else
          {
              return teammate( ballHolderUnum() );
          }
      }

    const PredictBallObject & ball() const
      {
          return M_ball;
      }

    const AbstractPlayerObject & self() const
      {
          if ( M_self_unum < 1 || 11 < M_self_unum )
          {
              std::cerr << "internal error: "
                        << __FILE__ << ":" << __LINE__
                        << "invalid self unum " << M_self_unum << std::endl;
              return *(M_all_teammates[0]);
          }

          return *(M_all_teammates[ M_self_unum - 1 ]);
      }

    const AbstractPlayerObject * teammate( const int unum ) const
      {
          if ( unum < 1 || 11 < unum  )
          {
              std::cerr << "internal error: "
                        << __FILE__ << ":" << __LINE__ << ": "
                        << "invalid unum " << unum << std::endl;
              return static_cast< const AbstractPlayerObject * >( 0 );
          }

          return &( *(M_all_teammates[ unum - 1 ]) );
      }

    const AbstractPlayerObject * opponent( const int unum ) const
      {
          if ( unum < 1 || 11 < unum  )
          {
              std::cerr << "internal error: "
                        << __FILE__ << ":" << __LINE__ << ": "
                        << "invalid unum " << unum << std::endl;
              return static_cast< const AbstractPlayerObject * >( 0 );
          }

          return &( *(M_opponents[ unum - 1 ]) );
      }

    const PredictPlayerPtrCont & allTeammates() const
      {
          return M_all_teammates;
      }

    const AbstractPlayerCont & allOpponents() const
      {
          return M_opponents_player_cont;
      }

    const PlayerPtrCont & opponentsFromSelf() const
      {
          return M_opponents_from_self;
      }

    const AbstractPlayerObject * getTeammateGoalie() const
      {
          return static_cast< AbstractPlayerObject * >( 0 );
      }

    const GameTime & currentTime() const
      {
          return M_game_time;
      }

    SideID ourSide() const
      {
          return M_our_side;
      }

    double offsideLineX() const
      {
          return M_offside_line_x;
      }

    double ourDefenseLineX() const
      {
          return M_our_defense_line_x;
      }

    double ourOffensePlayerLineX() const
      {
          return M_our_offense_player_line_x;
      }

    double theirDefensePlayerLineX() const
      {
          return M_their_defense_player_line_x;
      }

    const AudioMemory & audioMemory() const
      {
          static const AudioMemory audioMemory;
          return audioMemory;
      }

    AbstractPlayerCont getPlayerCont( const PlayerPredicate * predicate ) const;
};

}

#endif
