// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_EVS_EVOLUTIONAL_VARIABLE_H
#define RCSC_EVS_EVOLUTIONAL_VARIABLE_H

#include "evolutional_repository.h"
#include <boost/lexical_cast.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/algorithm/string.hpp>

#include <string>
#include <vector>
#include <map>
#include <iostream>

namespace rcsc {

class EvolutionalVariable {

public:
    enum Type { FloatType, StructType };

private:
    boost::shared_ptr<EvolutionalRepository> M_rep;
    std::string M_name;
    std::string M_id;
    bool M_evaluated;

protected:
    virtual
    void getValue( const std::string & name, std::string * str_val )
      {
          M_rep->getValue( name, str_val, &M_id );
      }

public:
    EvolutionalVariable( const boost::shared_ptr<EvolutionalRepository> & rep,
                         const std::string & name )
        : M_rep( rep )
        , M_name( name )
        , M_id( "" )
        , M_evaluated( false )
      {

      }

    virtual
    ~EvolutionalVariable()
      {
          if ( ! M_evaluated )
          {
              M_rep->evaluationCancel( M_name, M_id );
          }
      }

    virtual
    void evaluate( double evaluation_value )
      {
          M_rep->evaluate( M_name, M_id, evaluation_value );
          M_evaluated = true;
      }
};


class EvolutionalFloat : public EvolutionalVariable {

private:
    double M_value;

public:
    EvolutionalFloat( const boost::shared_ptr<EvolutionalRepository> & rep,
                      const std::string & name )
        : EvolutionalVariable( rep, name )
        , M_value( 0.0 )
      {
          std::string str_val;

          this->EvolutionalVariable::getValue( name, &str_val );

          try
          {
              M_value = boost::lexical_cast<double>( str_val );
          }
          catch( const boost::bad_lexical_cast & e )
          {
              std::cerr << str_val << ": invalid value, real number requested."
                        << std::endl;

              M_value = 0.0;
          }
      }

    ~EvolutionalFloat()
      {

      }

    operator double () const
      {
          return M_value;
      }
};

class EvolutionalStruct : public EvolutionalVariable {

private:
    struct Child
    {
        Type type;
        double float_value;

        Child()
            : type( FloatType )
            , float_value( 0.0 )
        {
        }

        Child( const Type & type, double float_value )
            : type( type )
            , float_value( float_value )
        {
        }

        Child & operator = ( const Child & c )
        {
            this->type = c.type;
            this->float_value = c.float_value;

            return *this;
        }
    };

    typedef std::map<std::string, Child> ChildMapType;

    ChildMapType M_child;

public:
    EvolutionalStruct( const boost::shared_ptr<EvolutionalRepository> & rep,
                       const std::string & name )
        : EvolutionalVariable( rep, name )
        , M_child()
      {
          std::string struct_str;

          this->EvolutionalVariable::getValue( name, &struct_str );

          if ( struct_str.length() < 2
               || *(struct_str.begin()) != '{'
               || *(struct_str.rbegin()) != '}' )
          {
              std::cerr << struct_str
                        << ": invalid value, struct value requested."
                        << std::endl;
              return;
          }

          const std::string members_str( struct_str, 1 , struct_str.length() - 2 );

          std::vector< std::string > entries;
          boost::algorithm::split( entries, members_str, boost::is_any_of( "," ),
                                   boost::algorithm::token_compress_off );

          for ( size_t i = 0; i < entries.size(); ++i )
          {
              std::vector< std::string > name_value_str;
              boost::algorithm::split( name_value_str, entries[i],
                                       boost::is_any_of( ":" ),
                                       boost::algorithm::token_compress_off );

              if ( name_value_str.size() != 2 )
              {
                  std::cerr << struct_str
                            << ": invalid value, struct value requested."
                            << std::endl;
                  return;
              }


              const std::string & name_str = name_value_str[0];
              const std::string & value_str = name_value_str[1];

              double value;
              try
              {
                  value = boost::lexical_cast<double>( value_str );
              }
              catch( const boost::bad_lexical_cast & e )
              {
                  std::cerr << value_str
                            << ": invalid value of [" << name_str << "]"
                            << " in struct [" << struct_str << "], "
                            << "real number requested" << std::endl;
                  return;
              }

              M_child[ name_str ] = Child( FloatType, value );
          }
      }

    ~EvolutionalStruct()
      {

      }

    double getFloatValue( const std::string & name ) const
      {
          const ChildMapType::const_iterator it = M_child.find( name );

          if ( it == M_child.end() )
          {
              std::cerr << name << ": not found in struct." << std::endl;

              return 0.0;
          }

          if ( ((*it).second).type != FloatType )
          {
              std::cerr << name << ": not a float value." << std::endl;

              return 0.0;
          }

          return ((*it).second).float_value;
      }
};

}

#endif
