// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "param_field_evaluator.h"
#include "variable_repository.h"
#include "field_analyzer.h"
#include "simple_pass_checker.h"

#include <rcsc/player/player_evaluator.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <rcsc/math_util.h>

#include <iostream>
#include <algorithm>
#include <cmath>
#include <cfloat>

#define DEBUG_PRINT

using namespace rcsc;

namespace {
static bool s_params_initialized = false;

// default values
// (evaluator-4231-devel-gen-18)
static double THEIR_GOAL_DIST_RATE = 47.5738497206819;
static double NEAR_OFFSIDE_LINE_DIST_THRETHOLD = 2.66091416713824;
static double NEAR_OFFSIDE_BONUS_RATE = 19.4287968238018;
static double NEAR_DEFENSE_LINE_DIST_THRETHOLD = 5.10828671749136;
static double ATTACK_LINE_X_THRESHOLD = 29.6245831971171;
static double OVER_ATTACK_LINE_RATE = 3.47329040685356;
static double OPP_DIST_THRETHOLD = 9.59137452980102;
static double OPP_DIST_FACTOR = 5.73872810031228;
static double PASS_COUNT_BONUS_RATE = 0.637027058115708;
static double CHAIN_SPEND_TIME_PENALTY_RATE = 0.640933805909443;
static double CHAIN_PATH_LENGTH_PENALTY_RATE = 0.186817585577638;
}


static const int VALID_PLAYER_THRESHOLD = 8;
static const double PASS_BALL_FIRST_SPEED = 2.5;


/*-------------------------------------------------------------------*/
/*!

 */
static double evaluate_state( const PredictState & state );

/*-------------------------------------------------------------------*/
/*!

 */
static
double
get_opponent_dist( const PredictState & wm,
                   const Vector2D & point,
                   int opponent_additional_chase_time = 0,
                   int valid_opponent_thr = -1 )
{
    static const double MAX_DIST = 65535.0;

    double min_dist = MAX_DIST;

    const AbstractPlayerCont::const_iterator end = wm.allOpponents().end();
    for ( AbstractPlayerCont::const_iterator it = wm.allOpponents().begin();
          it != end;
          ++it )
    {
        if ( (*it)->ghostCount() >= 3 )
        {
            continue;
        }

        if ( valid_opponent_thr != -1 )
        {
            if ( (*it)->posCount() > valid_opponent_thr )
            {
                continue;
            }
        }

        double dist = (*it)->pos().dist( point );
        dist -= ( bound( 0, (*it)->posCount() - 2, 2 )
                  + opponent_additional_chase_time )
            * (*it)->playerTypePtr()->realSpeedMax();

        if ( (*it)->goalie()
             && dist > 5.0 )
        {
            continue;
        }

        if ( dist < min_dist )
        {
            min_dist = dist;
        }
    }

    return min_dist;
}

/*-------------------------------------------------------------------*/
/*!

 */
ParamFieldEvaluator::ParamFieldEvaluator()
{

}

/*-------------------------------------------------------------------*/
/*!

 */
ParamFieldEvaluator::~ParamFieldEvaluator()
{

}

/*-------------------------------------------------------------------*/
/*!

 */
bool
ParamFieldEvaluator::read_parameters( const std::string & prefix )
{
    const VariableRepository & rep = VariableRepository::i();
    const std::string realm = "evol_default_field_evaluator";

    if ( ! rep.isValid() )
    {
        std::cerr << "invaild variable repository, using default values"
                  << " for field evaluator" << std::endl;
        return false;
    }


    THEIR_GOAL_DIST_RATE             = rep.getFloatValue( realm, "THEIR_GOAL_DIST_RATE" );
    NEAR_OFFSIDE_LINE_DIST_THRETHOLD = rep.getFloatValue( realm, "NEAR_OFFSIDE_LINE_DIST_THRETHOLD" );
    NEAR_OFFSIDE_BONUS_RATE          = rep.getFloatValue( realm, "NEAR_OFFSIDE_BONUS_RATE" );
    NEAR_DEFENSE_LINE_DIST_THRETHOLD = rep.getFloatValue( realm, "NEAR_DEFENSE_LINE_DIST_THRETHOLD" );
    ATTACK_LINE_X_THRESHOLD          = rep.getFloatValue( realm, "ATTACK_LINE_X_THRESHOLD" );
    OVER_ATTACK_LINE_RATE            = rep.getFloatValue( realm, "OVER_ATTACK_LINE_RATE" );
    OPP_DIST_THRETHOLD               = rep.getFloatValue( realm, "OPP_DIST_THRETHOLD" );
    OPP_DIST_FACTOR                  = rep.getFloatValue( realm, "OPP_DIST_FACTOR" );
    PASS_COUNT_BONUS_RATE            = rep.getFloatValue( realm, "PASS_COUNT_BONUS_RATE" );
    CHAIN_SPEND_TIME_PENALTY_RATE    = rep.getFloatValue( realm, "CHAIN_SPEND_TIME_PENALTY_RATE" );
    CHAIN_PATH_LENGTH_PENALTY_RATE   = rep.getFloatValue( realm, "CHAIN_PATH_LENGTH_PENALTY_RATE" );

    std::cerr << prefix
              << ": initialized parameters." << std::endl;

    s_params_initialized = true;

    return true;

}

/*-------------------------------------------------------------------*/
/*!

 */
double
ParamFieldEvaluator::operator()( const PredictState & state,
                                 const std::vector< ActionStatePair > & path ) const
{
    if ( ! s_params_initialized )
    {
        read_parameters( "" );
    }

    return get_chain_evaluation( state, path );
}

/*-------------------------------------------------------------------*/
/*!

 */
#if 0
double
ParamFieldEvaluator::get_chain_evaluation( const rcsc::PredictState & state,
                                           const std::vector< rcsc::ActionStatePair > & path )
{
    const double final_state_evaluation = evaluate_state( state );
    const double chain_length_penalty = get_chain_length_penalty( path, state );

    const double result = final_state_evaluation + chain_length_penalty;

#ifdef DEBUG_PRINT
    rcsc::dlog.addText( rcsc::Logger::ACTION_CHAIN,
                        "(eval) chain length penalty %f (%f)",
                        chain_length_penalty, result );
#endif

    return result;
}
#else
double
ParamFieldEvaluator::get_chain_evaluation( const rcsc::PredictState & state,
                                           const std::vector< rcsc::ActionStatePair > & path )
{
    //const double PATH_OPPENENT_DIST_PENALTY = -1.0e+4;
    const double PATH_OPPENENT_DIST_PENALTY = -1.0;

    const double PATH_OPPENENT_DIST_THRESHOLD = 5.0;
    const double PATH_OPPENENT_DIST_THRESHOLD_1ST = PATH_OPPENENT_DIST_THRESHOLD;
    const double PATH_OPPENENT_DIST_THRESHOLD_2ND = PATH_OPPENENT_DIST_THRESHOLD;
    // const double OPPONENT_ADDITIONAL_CHASE_RATE = 0.2;


    double result = evaluate_state( state );
    const double chain_length_penalty = get_chain_length_penalty( path, state );
    result += chain_length_penalty;

#ifdef DEBUG_PRINT
    rcsc::dlog.addText( rcsc::Logger::ACTION_CHAIN,
                        "(eval) chain length penalty %f (%f)",
                        chain_length_penalty, result );
#endif

    if ( path.empty() )
    {
        return result;
    }

    double penalty = 0.0;

    for ( int i = 1; i < static_cast< int >( path.size() ); ++i )
    {
        if ( state.spendTime() == 0 )
        {
            continue;
        }

        // if ( ( state.ball().pos() - ServerParam::i().theirTeamGoalPos() ).r() < 30.0
        //      || state.ball().pos().x > ServerParam::i().theirPenaltyAreaLineX()  )
        // {
        //     continue;
        // }


        //
        // get free radius
        //
        const double opponent_additional_chase_time
            = path[i - 1].action().durationStep();
        // const double opponent_additional_chase_time
        //     = state.spendTime() * OPPONENT_ADDITIONAL_CHASE_RATE;

        const double r = get_opponent_dist( path[i].state(),
                                            path[i].state().ballHolder()->pos(),
                                            opponent_additional_chase_time );

        double thr;

        switch( i )
        {
        case 1:
            thr = PATH_OPPENENT_DIST_THRESHOLD_1ST;
            break;

        case 2:
            thr = PATH_OPPENENT_DIST_THRESHOLD_2ND;
            break;

        default:
            thr = PATH_OPPENENT_DIST_THRESHOLD;
            break;
        }

        if ( r < thr )
        {

            penalty += ( PATH_OPPENENT_DIST_PENALTY * ( thr - r ) );
            // penalty += ( PATH_OPPENENT_DIST_PENALTY * ( thr - r ) )
            //            / ( path.size() - 1 );
        }
    }

    result += penalty;

#ifdef DEBUG_PRINT
    rcsc::dlog.addText( rcsc::Logger::ACTION_CHAIN,
                        "(eval) chain opp dist penalty %f (%f)",
                        penalty, result );
#endif

    return result;
}
#endif


/*-------------------------------------------------------------------*/
/*!

 */
static
double
evaluate_state( const PredictState & state )
{
    const ServerParam & SP = ServerParam::i();

    const AbstractPlayerObject * holder = state.ballHolder();

#ifdef DEBUG_PRINT
    static GameTime s_last_call_time( 0, 0 );
    if ( s_last_call_time != state.currentTime() )
    {
        s_last_call_time = state.currentTime();

        dlog.addText( Logger::ACTION_CHAIN,
                  "========= "__FILE__" ==========" );
        dlog.addText( Logger::ACTION_CHAIN,
                      "THEIR_GOAL_DIST_RATE %f", THEIR_GOAL_DIST_RATE );
        dlog.addText( Logger::ACTION_CHAIN,
                      "NEAR_OFFSIDE_LINE_DIST_THRETHOLD %f", NEAR_OFFSIDE_LINE_DIST_THRETHOLD );
        dlog.addText( Logger::ACTION_CHAIN,
                      "NEAR_OFFSIDE_BONUS_RATE %f", NEAR_OFFSIDE_BONUS_RATE );
        dlog.addText( Logger::ACTION_CHAIN,
                      "NEAR_DEFENSE_LINE_DIST_THRETHOLD %f", NEAR_DEFENSE_LINE_DIST_THRETHOLD );
        dlog.addText( Logger::ACTION_CHAIN,
                      "ATTACK_LINE_X_THRESHOLD %f", ATTACK_LINE_X_THRESHOLD );
        dlog.addText( Logger::ACTION_CHAIN,
                      "OPP_DIST_THRETHOLD %f", OPP_DIST_THRETHOLD );
        dlog.addText( Logger::ACTION_CHAIN,
                      "OPP_DIST_FACTOR %f", OPP_DIST_FACTOR );
        dlog.addText( Logger::ACTION_CHAIN,
                      "PASS_COUNT_BONUS_RATE %f", PASS_COUNT_BONUS_RATE );
        dlog.addText( Logger::ACTION_CHAIN,
                      "CHAIN_SPEND_TIME_PENALTY_RATE %f", CHAIN_SPEND_TIME_PENALTY_RATE );
        dlog.addText( Logger::ACTION_CHAIN,
                      "CHAIN_PATH_LENGTH_PENALTY_RATE %f", CHAIN_PATH_LENGTH_PENALTY_RATE );
    }

    dlog.addText( Logger::ACTION_CHAIN,
                  "========= (evaluate_state) ==========" );
#endif

    //
    // if holder is invalid, return bad evaluation
    //
    if ( ! holder )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) XXX null holder" );
#endif
        return - DBL_MAX / 2.0;
    }

    const int holder_unum = holder->unum();


    //
    // ball is in opponent goal
    //
    if ( state.ball().pos().x > + ( SP.pitchHalfLength() - 0.1 )
         && state.ball().pos().absY() < SP.goalHalfWidth() + 2.0 )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) *** in opponent goal" );
#endif
        return +1.0e+7;
    }

    //
    // ball is in our goal
    //
    if ( state.ball().pos().x < - ( SP.pitchHalfLength() - 0.1 )
         && state.ball().pos().absY() < SP.goalHalfWidth() )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) XXX in our goal" );
#endif

        return -1.0e+7;
    }


    //
    // out of pitch
    //
    if ( state.ball().pos().absX() > SP.pitchHalfLength()
         || state.ball().pos().absY() > SP.pitchHalfWidth() )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) XXX out of pitch" );
#endif

        return - DBL_MAX / 2.0;
    }


    //
    // set basic evaluation
    //
    double point = state.ball().pos().x;

#ifdef DEBUG_PRINT
    dlog.addText( Logger::ACTION_CHAIN,
                  "(eval) ball pos (%f, %f)",
                  state.ball().pos().x, state.ball().pos().y );

    dlog.addText( Logger::ACTION_CHAIN,
                  "(eval) initial value (%f)", point );
#endif


    //
    // add bonus for goal, free situation near offside line
    //
    if ( FieldAnalyzer::can_shoot_from
         ( holder->unum() == state.self().unum(),
           holder->pos(),
           state.getPlayerCont( new OpponentOrUnknownPlayerPredicate( state.ourSide() ) ),
           VALID_PLAYER_THRESHOLD ) )
    {
        point += 1.0e+6;
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) bonus for goal %f (%f)", 1.0e+6, point );
#endif

        if ( holder_unum == state.self().unum() )
        {
            point += 5.0e+5;
#ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          "(eval) bonus for goal self %f (%f)", 5.0e+5, point );
#endif
        }
    }
    else if ( ParamFieldEvaluator::is_free_situation_near_offside_line( state, state.ballHolder() ) )
    {
        const double near_offside_line_bonus
            = 1.0e+5 + ( state.ball().pos().x - SP.pitchHalfLength() ) * 2.0;

        point += near_offside_line_bonus;

#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) bonus for free near offside line %f (%f)",
                      near_offside_line_bonus,
                      point );
#endif
    }


    if ( ( state.ball().pos().x > SP.theirPenaltyAreaLineX()
           //&& state.ball().pos().absY() < SP.penaltyAreaHalfWidth()
           )
         || state.ball().pos().x > state.theirDefensePlayerLineX() )
    {
        double over_offside_line_bonus = 5.0e+5;

        //over_offside_line_bonus += ( state.ball().pos().x - state.offsideLineX() ) * 5.0;
        over_offside_line_bonus -= std::fabs( state.ball().pos().x - 45.0 );

        // const double their_goal_dist = ( holder->pos() - SP.theirTeamGoalPos() ).r();

        // if ( their_goal_dist < 50.0 )
        // {
        //     over_offside_line_bonus += ( 50.0 - their_goal_dist ) * 100.0;
        // }

        // center of field is higher, side is lower
        //over_offside_line_bonus += ( SP.pitchHalfWidth() - state.ball().pos().absY() ) * 100.0;

        if ( state.ball().pos().absY() > SP.penaltyAreaHalfWidth() - 6.0 )
        {
            over_offside_line_bonus -= state.ball().pos().absY() - ( SP.penaltyAreaHalfWidth() - 6.0 );
        }

        if ( state.ball().pos().x > SP.pitchHalfLength() - 10.0 )
        {
            over_offside_line_bonus -= std::fabs( state.ball().pos().x - ( SP.pitchHalfLength() - 10.0 ) );
        }

        point += over_offside_line_bonus;


#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) bonus for over offside line %f (%f)",
                      over_offside_line_bonus, point );
#endif
    }



    //
    // add distance bonus when near opponent goal
    //
    {
        const double their_goal_dist = ( holder->pos() - SP.theirTeamGoalPos() ).r();
        if ( their_goal_dist <= 30.0 )
        {
            const double bonus = ( 30.0 - their_goal_dist ) * THEIR_GOAL_DIST_RATE;

            point += bonus;

#ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          "(eval) bonus for opponent goal dist %f (%f)",
                          bonus, point );
#endif
        }
    }


    //
    // add distance penalty/bonus when near our goal
    //
    {
        const double our_goal_dist = ( holder->pos() - SP.ourTeamGoalPos() ).r();
        static const double OUR_GOAL_DIST_RATE = THEIR_GOAL_DIST_RATE;

        if ( our_goal_dist <= 30.0 )
        {
            const double penalty = - ( ( 30.0 - our_goal_dist ) * OUR_GOAL_DIST_RATE );
            point += penalty;

#ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          "(eval) penalty for our goal near %f (%f)",
                          penalty, point );
#endif
        }
    }



    //
    // add bonus near offside line
    //
    const double near_offside_line_dist_threthold = NEAR_OFFSIDE_LINE_DIST_THRETHOLD;

    if ( state.ball().pos().x > state.theirDefensePlayerLineX() - near_offside_line_dist_threthold )
    {
        const double bonus = ( state.ball().pos().x - ( state.offsideLineX() - near_offside_line_dist_threthold ) )
            * NEAR_OFFSIDE_BONUS_RATE;
        point += bonus;

#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) bonus for near offside line %f (%f)",
                      bonus, point );
#endif
    }


    //
    // add penalty near defense line
    //
    const double near_defense_line_dist_threthold = NEAR_DEFENSE_LINE_DIST_THRETHOLD;
    if ( state.ball().pos().x < state.ourDefenseLineX() + near_defense_line_dist_threthold )
    {
        double bonus = ( state.ball().pos().x - ( state.ourDefenseLineX() + near_defense_line_dist_threthold ) )
            * NEAR_DEFENSE_LINE_DIST_THRETHOLD;
        point += bonus;

#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) penalty for defense line %f (%f)",
                      bonus, point );
#endif
    }


    //
    // add penalty center of field when our goalie not in penalty area
    //
    const AbstractPlayerObject * our_goalie = state.getTeammateGoalie();
    if ( our_goalie != static_cast<AbstractPlayerObject *>( 0 ) )
    {
        const bool our_goalie_is_in_side_area
            = ( our_goalie->pos().absY() > SP.penaltyAreaHalfWidth() - 5.0
                && our_goalie->posCount() < VALID_PLAYER_THRESHOLD
                && our_goalie->pos().x < -20.0
                && state.ball().pos().x < -20.0 );

        if ( our_goalie_is_in_side_area )
        {
            double side_line_dist;

            if ( our_goalie->pos().y > 0.0 )
            {
                side_line_dist = ( SP.penaltyAreaHalfWidth()
                                   - state.ball().pos().y );
            }
            else
            {
                side_line_dist = ( state.ball().pos().y
                                   - (- SP.penaltyAreaHalfWidth() ) );
            }

            const double penalty = -30.0 + side_line_dist * 10.0;
            point += penalty;

#ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          "(eval) penalty for our goalie is in side area %f (%f)",
                          penalty, point );
#endif
        }
    }


    //
    // add absolute x pos bonus
    //
    const double attack_line_x_threshold = ATTACK_LINE_X_THRESHOLD;

    if ( holder->pos().x >= attack_line_x_threshold )
    {
        const double attack_line_bonus
            = ( holder->pos().x - attack_line_x_threshold ) * OVER_ATTACK_LINE_RATE;

        point += attack_line_bonus;

#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "(eval) bonus for over attack line %f (%f)",
                      attack_line_bonus,
                      point );
#endif
    }



    //
    // add free radius penalty
    //
    {
        const double r = get_opponent_dist( state, holder->pos() );

        const double opp_dist_threthold = OPP_DIST_THRETHOLD;
        const double factor = OPP_DIST_FACTOR;

#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "check opponent dist = %f thr=%f diff=%f factor=%f",
                      r, opp_dist_threthold,
                      opp_dist_threthold - r,
                      factor );
#endif

        if ( r < opp_dist_threthold )
        {
            double penalty = -factor * ( opp_dist_threthold - r );
            point += penalty;

#ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          "(eval) penalty for free radius %f (%f)",
                          penalty, point );
#endif
        }
    }


    //
    // add pass count bonus
    //
    const SimplePassChecker pass_checker;
    const int pass_count = FieldAnalyzer::get_pass_count( state,
                                                          pass_checker,
                                                          PASS_BALL_FIRST_SPEED,
                                                          3 );

    const double pass_count_bonus = std::min( pass_count, 3 ) * PASS_COUNT_BONUS_RATE;
    point += pass_count_bonus;

#ifdef DEBUG_PRINT
    dlog.addText( Logger::ACTION_CHAIN,
                  "(eval) bonus for pass count(%d) %f (%f)",
                  pass_count,
                  pass_count_bonus,
                  point );
#endif

    return point;
}

double
ParamFieldEvaluator::get_chain_length_penalty( const std::vector< ActionStatePair > & path,
                                               const PredictState & state )
{
    //
    // penalty by path length
    //
    double path_length_penalty = 0.0;
    if ( ! path.empty() )
    {
        path_length_penalty = -( ( path.size() - 1 ) * CHAIN_PATH_LENGTH_PENALTY_RATE );
    }
#ifdef DEBUG_PRINT
    dlog.addText( Logger::ACTION_CHAIN,
                  "(eval) penalty for path length=(%d) %f",
                  path.size(),
                  path_length_penalty );
#endif


    //
    // penalty by spend time
    //
    const double spend_time_penalty = - ( state.spendTime() * CHAIN_SPEND_TIME_PENALTY_RATE );

#ifdef DEBUG_PRINT
    dlog.addText( Logger::ACTION_CHAIN,
                  "(eval) penalty for spend time=(%d) %f",
                  state.spendTime(),
                  spend_time_penalty );
#endif

    return path_length_penalty + spend_time_penalty;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
ParamFieldEvaluator::is_free_situation_near_offside_line( const PredictState & wm,
                                                          const AbstractPlayerObject * from )
{
    const long VALID_PLAYER_ACCURACY = 20;
    const long VALID_OPPONENT_ACCURACY = VALID_PLAYER_ACCURACY;
    const double FREE_WIDTH = 1.75;

    if ( ! from
         || from->posCount() > VALID_PLAYER_ACCURACY
         || from->isGhost() )
    {
        return false;
    }

    if ( from->pos().x > wm.offsideLineX() )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "checking push %d: offside position",
                      from->unum() );
#endif
        return false;
    }

    if ( from->pos().x < ServerParam::i().theirPenaltyAreaLineX()
         && ( from->pos().x < wm.offsideLineX() - 10.0
              || from->pos().x < wm.ourOffensePlayerLineX() - 10.0 ) )
    {
#ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "checking push %d: far from offside line",
                      from->unum() );
#endif
        return false;
    }


    static const double WIDE_FREE_WIDTH = 10.0;

    if ( from->isSelf()
         || from->pos().x >= wm.offsideLineX() - 20.0 )
    {
        AbstractPlayerCont opp_set;
        opp_set = wm.getPlayerCont( new AndPlayerPredicate
                                    ( new OpponentOrUnknownPlayerPredicate( wm.ourSide() ),
                                      new CoordinateAccuratePlayerPredicate
                                      ( VALID_PLAYER_ACCURACY ),
                                      new XCoordinateForwardPlayerPredicate
                                      ( from->pos().x ),
                                      new XCoordinateBackwardPlayerPredicate
                                      ( wm.offsideLineX() + 1.0 ),
                                      new YCoordinatePlusPlayerPredicate
                                      ( from->pos().y - WIDE_FREE_WIDTH ),
                                      new YCoordinateMinusPlayerPredicate
                                      ( from->pos().y + WIDE_FREE_WIDTH ) ) );

        const double free_length
            = AbstractPlayerObject::get_minimum_evaluation( opp_set,
                                                            new XPosPlayerEvaluator )
            - from->pos().x;

        if ( free_length >= 12.5 )
        {
#ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          "checking push %d: first check ok",
                          from->unum() );
#endif
            return true;
        }
    }


    if ( from->pos().x <= wm.ourDefenseLineX() + 10.0 )
    {
# ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "checking push %d: near our defense line",
                      from->unum() );
# endif
        return false;
    }

    if ( from->pos().x < wm.ourOffensePlayerLineX() - 5.0 )
    {
# ifdef DEBUG_PRINT
        dlog.addText( Logger::ACTION_CHAIN,
                      "checking push %d: too far from our forward line: x = %.2f, forward line = %.2f",
                      from->unum(),
                      from->pos().x, wm.ourOffensePlayerLineX() );
# endif
        return false;
    }

    const AbstractPlayerCont::const_iterator o_end = wm.allOpponents().end();
    for ( AbstractPlayerCont::const_iterator opp = wm.allOpponents().begin();
          opp != o_end;
          ++opp )
    {
        if ( (*opp)->posCount() > VALID_OPPONENT_ACCURACY )
        {
            continue;
        }

        if ( (from->pos().x < (*opp)->pos().x
              && (*opp)->pos().x < from->pos().x + 5.0)
             && std::fabs( (*opp)->pos().y - from->pos().y ) < FREE_WIDTH )
        {
# ifdef DEBUG_PRINT
            dlog.addText( Logger::ACTION_CHAIN,
                          "checking push %d: too narrow free width",
                          from->unum() );
# endif
            return false;
        }
    }

    return true;
}
