// -*-c++-*-

/*!
  \file player_graph.cpp
  \brief player position graph Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "player_graph.h"

#include <rcsc/player/world_model.h>
#include <rcsc/player/player_predicate.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/player_type.h>

#include <cstdlib>

using namespace rcsc;


/*-------------------------------------------------------------------*/
/*!

 */
Vector2D
PlayerGraph::Node::randomPos() const
{
    Vector2D p = this->pos();
    p.x += -0.001 + 0.002 * ( static_cast< double >( std::rand() ) / static_cast< double >( RAND_MAX ) );
    p.y += -0.001 + 0.002 * ( static_cast< double >( std::rand() ) / static_cast< double >( RAND_MAX ) );

    return p;
}

/*-------------------------------------------------------------------*/
/*!

 */
PlayerGraph::PlayerGraph()
{
    M_nodes.reserve( 24 );
    //M_triangulation.setUseTriangles( false );
}

/*-------------------------------------------------------------------*/
/*!

 */
PlayerGraph::PlayerGraph( const rcsc::PlayerPredicate * predicate )
    : M_predicate( predicate )
{
    M_nodes.reserve( 24 );
    //M_triangulation.setUseTriangles( false );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerGraph::setPredicate( const PlayerPredicate * predicate )
{
    M_predicate = boost::shared_ptr< const PlayerPredicate >( predicate );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerGraph::updateNode( const WorldModel & wm )
{
    if ( ! M_predicate )
    {
        return;
    }

    AbstractPlayerCont players;
    wm.getPlayerCont( players, M_predicate );

    M_nodes.reserve( players.size() );
    const AbstractPlayerCont::const_iterator p_end = players.end();
    for ( AbstractPlayerCont::const_iterator p = players.begin();
          p != p_end;
          ++p )
    {
        M_nodes.push_back( Node( *p ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
PlayerGraph::update( const WorldModel & wm )
{
    static GameTime s_update_time( 0, 0 );

    if ( s_update_time == wm.time() )
    {
        return;
    }
    s_update_time = wm.time();

    M_nodes.clear();
    M_connections.clear();

    updateNode( wm );

    //
    // create triangulation
    //
    M_triangulation.clear();
    const std::vector< Node >::const_iterator n_end = M_nodes.end();
    for ( std::vector< Node >::const_iterator n = M_nodes.begin();
          n != n_end;
          ++n )
    {
        M_triangulation.addPoint( n->randomPos() );
    }

    M_triangulation.compute();

    //
    // create connections
    //
    const Triangulation::SegmentCont::const_iterator e_end = M_triangulation.edges().end();
    for ( Triangulation::SegmentCont::const_iterator e = M_triangulation.edges().begin();
          e != e_end;
          ++e )
    {
        Node * first = &M_nodes[e->first];
        Node * second = &M_nodes[e->second];

        first->addConnection( second );
        second->addConnection( first );

        M_connections.push_back( Connection( first, second ) );
    }
}
