// -*-c++-*-

/*!
  \file predict_state.h
  \brief predicted field state class Header File
*/

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA, Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_PREDICT_STATE_H
#define RCSC_PLAYER_PREDICT_STATE_H

#ifdef EVALUATION_PRINTER
#include "static_predict_state.h"
namespace rcsc {
    typedef StaticPredictState PredictState;
}
#else

#include "predict_player_object.h"
#include "predict_ball_object.h"

#include <rcsc/player/world_model.h>
#include <rcsc/player/abstract_player_object.h>
#include <rcsc/player/player_predicate.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/vector_2d.h>

#include <boost/shared_ptr.hpp>

#include <algorithm>

namespace rcsc {

class PredictState {
public:
    static const int VALID_PLAYER_THRESHOLD;

    typedef boost::shared_ptr< PredictState > Ptr; //!< pointer type alias
    typedef boost::shared_ptr< const PredictState > ConstPtr; //!< const pointer type alias

private:
    const WorldModel * M_world;
    unsigned long M_spend_time;

    int M_ball_holder_unum;

    PredictBallObject M_ball;

    int M_self_unum;

    PredictPlayerPtrCont M_all_teammates;


    double M_our_defense_line_x;
    double M_our_offense_player_line_x;


public:
    PredictState( const WorldModel & wm );

    PredictState( const PredictState & rhs,
                  unsigned long append_spend_time = 0 );

    PredictState( const PredictState & rhs,
                  unsigned long append_spend_time,
                  int ball_holder_unum,
                  const Vector2D & ball_and_holder_pos );

    PredictState( const PredictState & rhs,
                  unsigned long append_spend_time,
                  const Vector2D & ball_pos );

private:

    void init( const WorldModel & wm );
    void updateLines();

public:

    unsigned long spendTime() const
      {
          return M_spend_time;
      }

    int ballHolderUnum() const
      {
          return M_ball_holder_unum;
      }

    const AbstractPlayerObject * ballHolder() const
      {
          if ( ballHolderUnum() == Unum_Unknown )
          {
              return static_cast<AbstractPlayerObject *>( 0 );
          }
          else
          {
              return teammate( ballHolderUnum() );
          }
      }

    const PredictBallObject & ball() const
      {
          return M_ball;
      }

    const AbstractPlayerObject & self() const
      {
          if ( M_self_unum < 1 || 11 < M_self_unum )
          {
              std::cerr << "internal error: "
                        << __FILE__ << ":" << __LINE__
                        << "invalid self unum " << M_self_unum << std::endl;
              return *(M_all_teammates[0]);
          }

          return *(M_all_teammates[ M_self_unum - 1 ]);
      }

    const AbstractPlayerObject * teammate( const int unum ) const
      {
          if ( unum < 1 || 11 < unum  )
          {
              std::cerr << "internal error: "
                        << __FILE__ << ":" << __LINE__ << ": "
                        << "invalid unum " << unum << std::endl;
              return static_cast< const AbstractPlayerObject * >( 0 );
          }

          return &( *(M_all_teammates[ unum - 1 ]) );
      }

    const AbstractPlayerObject * opponent( const int unum ) const
      {
          return M_world->opponent( unum );
      }

    const PredictPlayerPtrCont & allTeammates() const
      {
          return M_all_teammates;
      }

    const PlayerCont & opponents() const
      {
          return M_world->opponents();
      }

    const AbstractPlayerCont & allOpponents() const
      {
          return M_world->allOpponents();
      }

    const PlayerPtrCont & opponentsFromSelf() const
      {
          return M_world->opponentsFromSelf();
      }

    const PlayerObject * getOpponentNearestTo( const Vector2D & point,
                                               const int count_thr,
                                               double * dist_to_point ) const
      {
          return M_world->getOpponentNearestTo( point, count_thr, dist_to_point );
      }

    const PlayerType * teammatePlayerType( const int unum ) const
      {
          return M_world->teammatePlayerType( unum );
      }

    const AbstractPlayerObject * getTeammateGoalie() const
      {
          if ( M_world->teammateGoalieUnum() == Unum_Unknown )
          {
              return static_cast< AbstractPlayerObject * >( 0 );
          }

          return teammate( M_world->teammateGoalieUnum() );
      }

    const AbstractPlayerObject * getOpponentGoalie() const
      {
          return M_world->getOpponentGoalie();
      }

    const GameMode & gameMode() const
      {
          return M_world->gameMode();
      }

    const GameTime & currentTime() const
      {
          return M_world->time();
      }

    SideID ourSide() const
      {
          return M_world->ourSide();
      }

    double offsideLineX() const
      {
          return std::max( M_world->offsideLineX(), M_ball.pos().x );
      }

    double ourDefenseLineX() const
      {
          return M_our_defense_line_x;
      }

    double ourOffensePlayerLineX() const
      {
          return M_our_offense_player_line_x;
      }

    double theirDefensePlayerLineX() const
      {
          return M_world->theirDefensePlayerLineX();
      }

    int dirCount( const AngleDeg & angle )
      {
          return M_world->dirCount( angle );
      }

    int dirRangeCount( const AngleDeg & angle,
                       const double & width,
                       int * max_count,
                       int * sun_count,
                       int * ave_count ) const
      {
          return M_world->dirRangeCount( angle, width,
                                         max_count, sun_count, ave_count );
      }

    const AudioMemory & audioMemory() const
      {
          return M_world->audioMemory();
      }

    AbstractPlayerCont getPlayerCont( const PlayerPredicate * predicate ) const;
};

}
#endif

#endif
