// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "role_defensive_half.h"

#include "strategy.h"

#include "bhv_basic_offensive_kick.h"
#include "bhv_basic_move.h"
#include "bhv_basic_tackle.h"
#include "bhv_chain_action.h"
#include "bhv_center_half_cross_block_move.h"
#include "bhv_center_half_danger_move.h"
#include "bhv_center_half_defensive_move.h"
#include "bhv_defensive_half_offensive_move.h"
#include "bhv_offensive_half_offensive_move.h"
#include "bhv_offensive_half_defensive_move.h"
#include "neck_offensive_intercept_neck.h"
#include "neck_default_intercept_neck.h"

#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_intercept.h>
#include <rcsc/action/body_turn_to_angle.h>
#include <rcsc/action/neck_turn_to_ball.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>

#include <rcsc/formation/formation.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

using namespace rcsc;

const std::string RoleDefensiveHalf::NAME( "DefensiveHalf" );

/*-------------------------------------------------------------------*/
/*!

 */
namespace {
rcss::RegHolder role = SoccerRole::creators().autoReg( &RoleDefensiveHalf::create,
                                                       RoleDefensiveHalf::NAME );
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
RoleDefensiveHalf::execute( PlayerAgent * agent )
{
    bool kickable = agent->world().self().isKickable();
    if ( agent->world().existKickableTeammate()
         && agent->world().teammatesFromBall().front()->distFromBall()
         < agent->world().ball().distFromSelf() )
    {
        kickable = false;
    }

    if ( kickable )
    {
        doKick( agent );
    }
    else
    {
        doMove( agent );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
RoleDefensiveHalf::doKick( PlayerAgent * agent )
{
    if ( Bhv_ChainAction().execute( agent ) )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (execute) do chain action" );
        agent->debugClient().addMessage( "ChainAction" );
        return;
    }

    Bhv_BasicOffensiveKick().execute( agent );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
RoleDefensiveHalf::doMove( PlayerAgent * agent )
{
    switch ( Strategy::get_ball_area( agent->world() ) ) {
    case Strategy::BA_CrossBlock:
        Bhv_CenterHalfCrossBlockMove().execute( agent );
        break;
    case Strategy::BA_Stopper:
        doCrossBlockMove( agent );
        break;
    case Strategy::BA_Danger:
        Bhv_CenterHalfDangerMove().execute( agent );
        break;
    case Strategy::BA_DribbleBlock:
        doDribbleBlockMove( agent );
        break;
    case Strategy::BA_DefMidField:
        Bhv_CenterHalfDefensiveMove().execute( agent );
        break;
    case Strategy::BA_DribbleAttack:
        //Bhv_OffensiveHalfDefensiveMove().execute( agent );
        //Bhv_OffensiveHalfOffensiveMove().execute( agent );
        Bhv_DefensiveHalfOffensiveMove().execute( agent );
        //Bhv_BasicMove().execute( agent );
        break;
    case Strategy::BA_OffMidField:
        //Bhv_OffensiveHalfDefensiveMove().execute( agent );
        Bhv_OffensiveHalfOffensiveMove().execute( agent );
        break;
    case Strategy::BA_Cross:
        //Bhv_OffensiveHalfDefensiveMove().execute( agent );
        //Bhv_OffensiveHalfOffensiveMove().execute( agent );
        Bhv_DefensiveHalfOffensiveMove().execute( agent );
        //Bhv_BasicMove().execute( agent );
        break;
    case Strategy::BA_ShootChance:
        //Bhv_OffensiveHalfDefensiveMove().execute( agent );
        //Bhv_OffensiveHalfOffensiveMove().execute( agent );
        Bhv_DefensiveHalfOffensiveMove().execute( agent );
        //Bhv_BasicMove().execute( agent );
        break;
    default:
        dlog.addText( Logger::ROLE,
                            __FILE__": unknown ball area" );
        Bhv_BasicMove().execute( agent );
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
RoleDefensiveHalf::doCrossBlockMove( PlayerAgent * agent )
{
    dlog.addText( Logger::ROLE,
                        __FILE__": doCrossBlockMove" );

    //-----------------------------------------------
    // tackle
    if ( Bhv_BasicTackle( 0.8, 80.0 ).execute( agent ) )
    {
        return;
    }

    const WorldModel & wm = agent->world();
    const Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    //if ( wm.interceptTable()->isSelfFastestPlayer() )
    if ( ! wm.existKickableTeammate()
         && ! wm.existKickableOpponent()
         && self_min < mate_min
         && self_min < opp_min )
    {
        Body_Intercept().execute( agent );

        if ( opp_min >= self_min + 3 )
        {
            agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        }
        else
        {
            agent->setNeckAction( new Neck_DefaultInterceptNeck
                                  ( new Neck_TurnToBallOrScan() ) );
        }

        return;
    }
    double dash_power = 0.0;

    if ( wm.self().staminaModel().capacityIsEmpty() )
    {
        dash_power = std::min( ServerParam::i().maxDashPower(),
                               wm.self().stamina() + wm.self().playerType().extraStamina() );
    }
    else if ( wm.ball().distFromSelf() > 30.0 )
    {
        dash_power = wm.self().playerType().staminaIncMax() * 0.9;
        dlog.addText( Logger::ROLE,
                      __FILE__": ball is too far. dash_power=%.1f",
                      dash_power );
    }
    else if ( wm.ball().distFromSelf() > 20.0 )
    {
        dash_power = ServerParam::i().maxDashPower() * 0.5;
        dlog.addText( Logger::ROLE,
                      __FILE__": ball far. dash_power=%.1f",
                      dash_power );
    }
    else
    {
        dash_power = ServerParam::i().maxDashPower() * 0.9;
        dlog.addText( Logger::ROLE,
                      __FILE__": near. dash_power = %.1f",
                      dash_power );
    }

    if ( wm.existKickableTeammate() )
    {
        dash_power = std::min( ServerParam::i().maxDashPower() * 0.5,
                               dash_power );
    }

    dash_power = wm.self().getSafetyDashPower( dash_power );

    double dist_thr = wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    agent->debugClient().addMessage( "CrossBlock" );
    agent->debugClient().setTarget( home_pos );
    agent->debugClient().addCircle( home_pos, dist_thr );

    if ( ! Body_GoToPoint( home_pos, dist_thr, dash_power ).execute( agent ) )
    {
        AngleDeg body_angle( wm.ball().angleFromSelf().abs() < 80.0
                                   ? 0.0
                                   : 180.0 );
        Body_TurnToAngle( body_angle ).execute( agent );
    }

    agent->setNeckAction( new Neck_TurnToBall() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
RoleDefensiveHalf::doDribbleBlockMove( PlayerAgent * agent )
{
    dlog.addText( Logger::ROLE,
                        __FILE__": doDribbleBlockMove()" );

    ///////////////////////////////////////////////////
    // tackle
    if ( Bhv_BasicTackle( 0.8, 80.0 ).execute( agent ) )
    {
        return;
    }

    const WorldModel & wm = agent->world();
    const Vector2D home_pos = Strategy::i().getPosition( wm.self().unum() );

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( ! wm.existKickableTeammate()
         && ! wm.existKickableOpponent()
         && self_min <= mate_min + 1
         && self_min <= opp_min + 2 )
    {
        dlog.addText( Logger::ROLE,
                            __FILE__": doDribbleBlockMove() I am fastest. intercept" );
        Body_Intercept().execute( agent );

        if ( opp_min >= self_min + 3 )
        {
            agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        }
        else
        {
            agent->setNeckAction( new Neck_DefaultInterceptNeck
                                  ( new Neck_TurnToBallOrScan() ) );
        }

        return;
    }

    ///////////////////////////////////////////////////
    double dash_power;

    if ( wm.self().staminaModel().capacityIsEmpty() )
    {
        dash_power = std::min( ServerParam::i().maxDashPower(),
                               wm.self().stamina() + wm.self().playerType().extraStamina() );
    }
    else if ( wm.self().pos().x + 5.0 < wm.ball().pos().x )
    {
        if ( wm.self().stamina() < ServerParam::i().staminaMax() * 0.7 )
        {
            dash_power
                = ServerParam::i().maxDashPower() * 0.7
                * ( wm.self().stamina() / ServerParam::i().staminaMax() );
            dlog.addText( Logger::ROLE,
                                __FILE__": doDribbleBlockMove() dash_power=%.1f",
                                dash_power );
        }
        else
        {
            dash_power
                = ServerParam::i().maxDashPower() * 0.75
                - std::min( 30.0, wm.ball().distFromSelf() );
            dlog.addText( Logger::ROLE,
                                __FILE__": doDribbleBlockMove() dash_power=%.1f",
                                dash_power );
        }
    }
    else
    {
        dash_power
            = ServerParam::i().maxDashPower()
            + 10.0
            - wm.ball().distFromSelf();
        dash_power = min_max( 0.0, dash_power, ServerParam::i().maxDashPower() );
        dlog.addText( Logger::ROLE,
                            __FILE__": doDribbleBlockMove() dash_power=%.1f",
                            dash_power );
    }



    if ( wm.existKickableTeammate() )
    {
        dash_power = std::min( ServerParam::i().maxDashPower() * 0.5,
                               dash_power );
    }

    dash_power = wm.self().getSafetyDashPower( dash_power );

    double dist_thr = wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    agent->debugClient().addMessage( "DribBlockMove" );
    agent->debugClient().setTarget( home_pos );
    agent->debugClient().addCircle( home_pos, dist_thr );

    if ( ! Body_GoToPoint( home_pos, dist_thr, dash_power ).execute( agent ) )
    {
        // face to front or side
        AngleDeg body_angle = 0.0;
        if ( wm.ball().angleFromSelf().abs() > 90.0 )
        {
            body_angle = ( wm.ball().pos().y > wm.self().pos().y
                           ? 90.0
                           : -90.0 );
        }
        Body_TurnToAngle( body_angle ).execute( agent );
    }
    agent->setNeckAction( new Neck_TurnToBallOrScan() );
}
