// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "role_side_back.h"

#include "strategy.h"
#include "defense_system.h"

#include "bhv_chain_action.h"

#include "bhv_basic_offensive_kick.h"
#include "bhv_basic_move.h"
#include "bhv_basic_tackle.h"

#include "bhv_side_back_aggressive_cross_block.h"
#include "bhv_side_back_danger_move.h"
#include "bhv_side_back_defensive_move.h"
#include "bhv_side_back_offensive_move.h"
#include "bhv_side_back_stopper_move.h"
#include "bhv_mark_pass_line.h"

#include "neck_default_intercept_neck.h"
#include "neck_offensive_intercept_neck.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_intercept.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>

#include <rcsc/formation/formation.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

using namespace rcsc;

const std::string RoleSideBack::NAME( "SideBack" );

/*-------------------------------------------------------------------*/
/*!

 */
namespace {
rcss::RegHolder role = SoccerRole::creators().autoReg( &RoleSideBack::create,
                                                       RoleSideBack::NAME );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
RoleSideBack::execute( PlayerAgent * agent )
{
    bool kickable = agent->world().self().isKickable();
    if ( agent->world().existKickableTeammate()
         && agent->world().teammatesFromBall().front()->distFromBall()
         < agent->world().ball().distFromSelf() )
    {
        kickable = false;
    }

    if ( kickable )
    {
        doKick( agent );
    }
    else
    {
        doMove( agent );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
RoleSideBack::doKick( PlayerAgent * agent )
{
    if ( Bhv_ChainAction().execute( agent ) )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (execute) do chain action" );
        agent->debugClient().addMessage( "ChainAction" );
        return;
    }

    Bhv_BasicOffensiveKick().execute( agent );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
RoleSideBack::doMove( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    int ball_step = 1000;
    ball_step = std::min( ball_step, wm.interceptTable()->teammateReachCycle() );
    ball_step = std::min( ball_step, wm.interceptTable()->opponentReachCycle() );
    ball_step = std::min( ball_step, wm.interceptTable()->selfReachCycle() );

    Vector2D ball_pos = wm.ball().inertiaPoint( ball_step );

    PositionType position_type = Strategy::i().getPositionType( wm.self().unum() );


    switch ( Strategy::get_ball_area( wm ) ) {
    case Strategy::BA_CrossBlock:
        if ( ( ball_pos.y < 0.0
               && position_type != Position_Right )
             || ( ball_pos.y > 0.0
                  && position_type != Position_Left ) )
        {
            Bhv_SideBackAggressiveCrossBlock().execute( agent );
        }
        else
        {
            Bhv_SideBackDefensiveMove().execute( agent );
        }
        break;
    case Strategy::BA_Stopper:
        Bhv_SideBackStopperMove().execute( agent );
        break;
    case Strategy::BA_Danger:
        Bhv_SideBackDangerMove().execute( agent );
        break;
    case Strategy::BA_DribbleBlock:
        Bhv_SideBackDefensiveMove().execute( agent );
        break;
    case Strategy::BA_DefMidField:
        Bhv_SideBackDefensiveMove().execute( agent );
        break;
    case Strategy::BA_DribbleAttack:
    case Strategy::BA_OffMidField:
    case Strategy::BA_Cross:
    case Strategy::BA_ShootChance:
        Bhv_SideBackOffensiveMove().execute( agent );
        //doBasicMove( agent );
        break;
    default:
        dlog.addText( Logger::ROLE,
                      __FILE__": unknown ball area" );
        Bhv_BasicMove().execute( agent );
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
RoleSideBack::doBasicMove( PlayerAgent * agent )
{
    dlog.addText( Logger::ROLE,
                  __FILE__": doBasicMove" );

    //-----------------------------------------------
    // tackle
    if ( Bhv_BasicTackle( 0.85, 60.0 ).execute( agent ) )
    {
        return;
    }

    const WorldModel & wm = agent->world();

    //
    // intercept
    //

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( ! wm.existKickableTeammate()
         && ! wm.existKickableOpponent()
         && self_min <= mate_min + 1
         && self_min <= opp_min + 3 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": doBasicMove() intercept" );
        Body_Intercept().execute( agent );

        if ( opp_min >= self_min + 3 )
        {
            agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        }
        else
        {
            agent->setNeckAction( new Neck_DefaultInterceptNeck
                                  ( new Neck_TurnToBallOrScan() ) );
        }

        return;
    }

    if ( opp_min <= self_min
         && opp_min <= mate_min )
    {
        Vector2D opp_trap_pos = wm.ball().inertiaPoint( opp_min );
        if ( opp_trap_pos.x < 12.0 )
        {
            //if ( Bhv_DefensiveMark().execute( agent ) )
            //{
            //    return;
            //}

            dlog.addText( Logger::ROLE,
                          __FILE__": (doBasicMove) switch to the defensive move" );
            Bhv_SideBackDefensiveMove().execute( agent );
            return;
        }
    }

    if ( Bhv_MarkPassLine().execute( agent ) )
    {
        return;
    }

    // decide move target point
    Vector2D target_point = Strategy::i().getPosition( wm.self().unum() );

    // decide dash power
    double dash_power = DefenseSystem::get_defender_dash_power( wm, target_point );
    // decide threshold
    double dist_thr = std::fabs( wm.ball().pos().x - wm.self().pos().x ) * 0.1; //wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 1.0 ) dist_thr = 1.0;

    agent->debugClient().addMessage( "SBBasic%.0f", dash_power );
    agent->debugClient().setTarget( target_point );
    agent->debugClient().addCircle( target_point, dist_thr );

    if ( ! Body_GoToPoint( target_point, dist_thr, dash_power ).execute( agent ) )
    {
        AngleDeg body_angle = ( wm.ball().pos().y < wm.self().pos().y
                                ? -90.0
                                : 90.0 );
        Body_TurnToAngle( body_angle ).execute( agent );
    }

    if ( ( wm.existKickableOpponent()
           || opp_min <= 3 )
         && ( wm.ball().distFromSelf() < 10.0 //20.0
              || wm.ball().posCount() >= 2 ) )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": check ball" );
        agent->debugClient().addMessage( "CheckBall" );
        agent->setNeckAction( new Neck_TurnToBall() );
    }
    else
    {
        int count_thr = -1;
        switch ( agent->effector().queuedNextViewWidth().type() ) {
        case ViewWidth::NARROW:
            count_thr = 1;
            break;
        case ViewWidth::NORMAL:
            count_thr = 2;
            break;
        case ViewWidth::WIDE:
            count_thr = 3;
            break;
        default:
            break;
        }

        dlog.addText( Logger::ROLE,
                      __FILE__": check ball or scan" );
        agent->debugClient().addMessage( "CheckBallOrScan" );
        agent->setNeckAction( new Neck_TurnToBallOrScan( count_thr ) );
    }
}
