// -*-c++-*-

/*!
  \file attacker_positioning_generator.cpp
  \brief attacker's positioning action generator Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "attacker_positioning_generator.h"

#include <rcsc/player/world_model.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <rcsc/geom/rect_2d.h>
#include <rcsc/soccer_math.h>
#include <rcsc/timer.h>

#define DEBUG_PROFILE

#define DEBUG_UPDATE_PASSER

using namespace rcsc;

namespace {
const double shootable_dist2 = std::pow( 19.0, 2 ); // Magic Number
}

/*-------------------------------------------------------------------*/
/*!

 */
AttackerPositioningGenerator::AttackerPositioningGenerator()
{
    M_actions.reserve( 256 );

    clear();
}

/*-------------------------------------------------------------------*/
/*!

 */
AttackerPositioningGenerator &
AttackerPositioningGenerator::instance()
{
    static AttackerPositioningGenerator s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AttackerPositioningGenerator::clear()
{
    M_total_count = 0;
    M_passer = static_cast< AbstractPlayerObject * >( 0 );
    M_start_time.assign( -1, 0 );
    M_first_point.invalidate();
    M_actions.clear();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AttackerPositioningGenerator::update( const WorldModel & wm )
{
    static GameTime s_update_time( -1, 0 );
    if ( s_update_time == wm.time() )
    {
        return;
    }
    s_update_time = wm.time();

    clear();

    if ( wm.time().stopped() > 0
         || wm.gameMode().isPenaltyKickMode() )
    {
        return;
    }

#ifdef DEBUG_PROFILE
    Timer timer;
#endif

    updatePasser( wm );

    if ( ! M_passer
         || ! M_first_point.isValid() )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__" (update) passer not found." );
        return;
    }

    createActions( wm );

#ifdef DEBUG_PROFILE
    dlog.addText( Logger::TEAM,
                  __FILE__" (update) action_size=%d/%d elapsed %f [ms]",
                  (int)M_actions.size(),
                  M_total_count,
                  timer.elapsedReal() );
    //     std::cerr << "PassGenerator: total_count=" << M_total_count
    //               << " success_count=" << M_courses.size()
    //               << std::endl;
#endif
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AttackerPositioningGenerator::updatePasser( const WorldModel & wm )
{
    if ( wm.self().isKickable() )
    {
        M_passer = &wm.self();
        M_start_time = wm.time();
        M_first_point = wm.ball().pos();
#ifdef DEBUG_UPDATE_PASSER
        dlog.addText( Logger::TEAM,
                      __FILE__" (updatePasser) self kickable." );
#endif
        return;
    }

    int s_min = wm.interceptTable()->selfReachCycle();
    int t_min = wm.interceptTable()->teammateReachCycle();
    int o_min = wm.interceptTable()->opponentReachCycle();

    int our_min = std::min( s_min, t_min );
    if ( o_min < std::min( our_min - 4, (int)rint( our_min * 0.9 ) ) )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__" (updatePasser) opponent ball." );
        return;
    }

    if ( s_min <= t_min )
    {
        M_passer = &wm.self();
    }
    else
    {
        M_passer = wm.interceptTable()->fastestTeammate();
    }

    if ( ! M_passer )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__" (updatePasser) NULL passer teammate." );
        return;
    }

    M_start_time = wm.time();
    if ( ! wm.gameMode().isServerCycleStoppedMode() )
    {
        M_start_time.addCycle( t_min );
    }

    M_first_point = wm.ball().inertiaPoint( t_min );

#ifdef DEBUG_UPDATE_PASSER
    dlog.addText( Logger::TEAM,
                  __FILE__" (updatePasser) passer=%d(%.1f %.1f) reachStep=%d startPos=(%.1f %.1f)",
                  M_passer->unum(),
                  M_passer->pos().x, M_passer->pos().y,
                  t_min,
                  M_first_point.x, M_first_point.y );
#endif

}

/*-------------------------------------------------------------------*/
/*!

 */
void
AttackerPositioningGenerator::createActions( const rcsc::WorldModel &  )
{

}
