// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_attacker_offensive_kick.h"

#include "field_analyzer.h"

#include "bhv_self_pass.h"

#include <rcsc/action/body_dribble.h>
#include <rcsc/action/body_hold_ball.h>
#include <rcsc/action/body_pass.h>
#include <rcsc/action/neck_scan_field.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/sector_2d.h>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

 */
bool
Bhv_AttackerOffensiveKick::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::TEAM,
                  __FILE__": Bhv_AttackerOffensiveKick" );

    const WorldModel & wm = agent->world();

    const PlayerObject * nearest_opp = wm.getOpponentNearestToSelf( 6 );

    if ( nearest_opp )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": doMiddleAreaKick. nearest opp=(%.1f %.1f)",
                      nearest_opp->pos().x, nearest_opp->pos().y );
    }

    Vector2D receive_point = Vector2D::INVALIDATED;
    const bool exist_pass = Body_Pass::get_best_pass( wm, &receive_point, NULL, NULL );

    if ( exist_pass
         && FieldAnalyzer::can_shoot_from( wm, receive_point, 5 ) )
    {
        double receive_point_opponent_dist = 1000.0;
        wm.getOpponentNearestTo( receive_point,
                                 5,
                                 &receive_point_opponent_dist );
        if ( receive_point_opponent_dist > 4.0
             || ( nearest_opp
                  && nearest_opp->distFromSelf() < receive_point_opponent_dist )
             )
        {
            agent->debugClient().addMessage( "LastPass" );
            dlog.addText( Logger::ROLE,
                          __FILE__": (execute) receive point (%.1f %.1f)",
                          receive_point.x,
                          receive_point.y );
            Body_Pass().execute( agent );
            agent->setNeckAction( new Neck_ScanField() );
            return true;
        }
    }

    if ( Bhv_SelfPass().execute( agent ) )
    {
        return true;
    }

    if ( exist_pass )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": pass receive point (%.1f %.1f)",
                      receive_point.x,
                      receive_point.y );

        Rect2D pass_front_rect = Rect2D::from_corners( receive_point.x,
                                                       receive_point.y - 5.0,
                                                       receive_point.x + 20.0,
                                                       receive_point.y + 5.0 );
        if ( receive_point.x > wm.self().pos().x - 5.0
             && receive_point.absY() < 15.0
             && ! wm.existOpponentIn( pass_front_rect, 10, false ) ) // without goalie
        {
            agent->debugClient().addMessage( "PassChance" );
            Body_Pass().execute( agent );
            agent->setNeckAction( new Neck_ScanField() );
            return true;
        }
    }

    // TODO: strict forward dribble prediction

    // dribble straight
    if ( wm.self().stamina() > ServerParam::i().recoverDecThrValue() + 200.0
         && ( ! nearest_opp
              || nearest_opp->pos().x < wm.self().pos().x + 1.5
              || std::fabs( nearest_opp->pos().y -  wm.self().pos().y ) > 2.0
              || nearest_opp->distFromSelf() > 4.0 )
         )
    {
        const Rect2D target_rect
            = Rect2D::from_center( wm.ball().pos().x + 5.5,
                                   wm.ball().pos().y,
                                   10.0, 12.0 );
        if ( ! wm.existOpponentIn( target_rect, 10, false ) )
        {
            Vector2D drib_target( 47.0, wm.self().pos().y );
            if ( wm.self().body().abs() < 20.0 )
            {
                drib_target
                    = wm.self().pos()
                    + Vector2D::polar2vector( 10.0, wm.self().body() );
            }

            int dash_step = 10;

            const Rect2D safety_rect
                = Rect2D::from_center( wm.ball().pos().x + 7.0,
                                       wm.ball().pos().y,
                                       14.0, 15.0 );
            if ( wm.existOpponentIn( target_rect, 10, false ) )
            {
                dash_step = 3;
            }

            agent->debugClient().addMessage( "DribbleStraight" );
            agent->debugClient().addRectangle( target_rect );
            agent->debugClient().addRectangle( safety_rect );
            dlog.addText( Logger::ROLE,
                          __FILE__": dribble straight to (%.1f %.1f)",
                          drib_target.x, drib_target.y );
            Body_Dribble( drib_target,
                          1.0,
                          ServerParam::i().maxDashPower(),
                          dash_step, // dash count
                          false // no dodge
                          ).execute( agent );
            agent->setNeckAction( new Neck_ScanField() );
            return true;
        }
    }


    // pass in order to avoid opponent
    if ( exist_pass
         && nearest_opp
         && nearest_opp->distFromSelf() < 2.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": pass(1)" );
        agent->debugClient().addMessage( "Pass(1)" );
        Body_Pass().execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    Vector2D drib_target( 50.0, 0.0 );
    drib_target.y = ( agent->world().self().pos().y > 0.0
                      ? 5.0 : -5.0 );

    // opp's is behind of me
    if ( ! nearest_opp
         || ( nearest_opp->distFromSelf()
              > ServerParam::i().defaultPlayerSize() * 2.0 + 2.0
              && nearest_opp->pos().x < wm.self().pos().x )
         )
    {
        AngleDeg drib_angle = ( drib_target - wm.self().pos() ).th();
        const Sector2D sector( agent->world().self().pos(),
                               0.5, 15.0,
                               drib_angle - 30.0, drib_angle + 30.0 );
        // opponent check with goalie
        if ( ! wm.existOpponentIn( sector, 10, true ) )
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": fast dribble" );
            agent->debugClient().addMessage( "DribbleFast" );
            Body_Dribble( drib_target,
                          1.0,
                          ServerParam::i().maxDashPower(),
                          5
                          ).execute( agent );
        }
        else
        {
            dlog.addText( Logger::ROLE,
                          __FILE__": low dribble" );
            agent->debugClient().addMessage( "DribbleSlow" );
            Body_Dribble( drib_target,
                          1.0,
                          ServerParam::i().maxDashPower() * 0.5,
                          5
                          ).execute( agent );
        }
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    //------------------------------------------------
    // check pass point
    if ( exist_pass
         && ( receive_point.x > wm.offsideLineX() + 5.0
              || ( receive_point.x > 39.0
                   && receive_point.absY() < 15.0 ) )
         )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": chance pass" );
        agent->debugClient().addMessage( "ThroughPass" );
        Body_Pass().execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    //------------------------------------------------
    // opp is far from me
    if ( ! nearest_opp
         || nearest_opp->distFromSelf() > 4.0 )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": doMiddleAreaKick() opp far. dribble" );
        agent->debugClient().addMessage( "MidDribD1" );
        Body_Dribble( drib_target,
                      1.0,
                      ServerParam::i().maxDashPower(),
                      1
                      ).execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    //------------------------------------------------
    // opp is close
    // pass
    if ( exist_pass )
    {
        dlog.addText( Logger::ROLE,
                      __FILE__": doMiddleAreaKick() pass" );
        agent->debugClient().addMessage( "NormalPass" );
        Body_Pass().execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    //
    // near opponent may exist.
    //

    dlog.addText( Logger::ROLE,
                  __FILE__": doMiddleAreaKick() opponent is near. keep" );
    agent->debugClient().addMessage( "MidAreaHold" );
    Vector2D face_point = ServerParam::i().theirTeamGoalPos();
    Body_HoldBall( true, face_point ).execute( agent );
    agent->setNeckAction( new Neck_ScanField() );

    return true;
}
