// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_pre_process.h"

#include "strategy.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/bhv_before_kick_off.h>
#include <rcsc/action/bhv_emergency.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_intercept.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/bhv_shoot.h>
#include <rcsc/action/neck_scan_field.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/view_synch.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/audio_memory.h>
#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!
  execute action
*/
bool
Bhv_PreProcess::execute( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    dlog.addText( Logger::TEAM,
                  __FILE__": Bhv_PreProcess" );

    //////////////////////////////////////////////////////////////
    // freezed by tackle effect
    if ( wm.self().isFrozen() )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": tackle wait. expires= %d",
                      wm.self().tackleExpires() );
        // face neck to ball
        agent->setViewAction( new View_Synch() );
        agent->setNeckAction( new Neck_TurnToBallOrScan() );
        return true;
    }

    //////////////////////////////////////////////////////////////
    // BeforeKickOff or AfterGoal. should jump to the initial position
    if ( wm.gameMode().type() == GameMode::BeforeKickOff
         || wm.gameMode().type() == GameMode::AfterGoal_ )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__":before_kick_off" );
        Vector2D move_point =  Strategy::i().getPosition( wm.self().unum() );
        //agent->setViewAction( new View_Synch() );
        agent->setViewAction( new View_Wide() );
        Bhv_BeforeKickOff( move_point ).execute( agent );
        return true;
    }

    //////////////////////////////////////////////////////////////
    // my pos is unknown
    if ( ! wm.self().posValid() )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": invalid my pos" );
        // included change view
        Bhv_Emergency().execute( agent );
        return true;
    }
    //////////////////////////////////////////////////////////////
    // ball search
    // included change view
    int count_thr = 5;
    if ( wm.self().goalie() ) count_thr = 10;
    if ( wm.ball().posCount() > count_thr
         || wm.ball().rposCount() > count_thr + 3 )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": search ball" );
        //agent->setViewAction( new View_Synch() );
        agent->setViewAction( new View_Wide() );
        Bhv_NeckBodyToBall().execute( agent );
        return true;
    }

    //////////////////////////////////////////////////////////////
    // default change view

    agent->setViewAction( new View_Synch() );


    //////////////////////////////////////////////////////////////
    // check queued action
    if ( agent->doIntention() )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": do queued intention" );
        return true;
    }

    //////////////////////////////////////////////////////////////
    // check shoot chance
    if ( wm.gameMode().type() != GameMode::IndFreeKick_
         && wm.time().stopped() == 0
         && wm.self().isKickable()
         && Bhv_Shoot().execute( agent ) )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": shooted" );

        // reset intention
        agent->setIntention( static_cast< SoccerIntention * >( 0 ) );
        return true;
    }

    //////////////////////////////////////////////////////////////
    // check simultaneous kick
    if ( wm.gameMode().type() == GameMode::PlayOn
         && ! wm.self().goalie()
         && wm.self().isKickable()
         && wm.existKickableOpponent() )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": simultaneous kick" );
        agent->debugClient().addMessage( "SimultaneousKick" );
        Vector2D goal_pos( ServerParam::i().pitchHalfLength(), 0.0 );

        if ( wm.self().pos().x > 36.0
             && wm.self().pos().absY() > 10.0 )
        {
            goal_pos.x = 45.0;
            dlog.addText( Logger::TEAM,
                          __FILE__": simultaneous kick cross type" );
        }
        Body_KickOneStep( goal_pos,
                          ServerParam::i().ballSpeedMax()
                          ).execute( agent );
        agent->setNeckAction( new Neck_ScanField() );
        return true;
    }

    //////////////////////////////////////////////////////////////
    // check communication intention
    if ( wm.audioMemory().passTime() == wm.time()
         && ! wm.audioMemory().pass().empty()
         && wm.audioMemory().pass().front().receiver_ == wm.self().unum()
         )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": communication receive" );
        doReceiveMove( agent );

        return true;
    }

    return false;
}


/*-------------------------------------------------------------------*/
/*!

 */
void
Bhv_PreProcess::doReceiveMove( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    int self_min = wm.interceptTable()->selfReachCycle();

    if ( ( ! wm.existKickableTeammate()
           && self_min < 6 )
         || wm.audioMemory().pass().empty() )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (doReceiveMove) intercept cycle=%d"
                      ,__FILE__, __LINE__,
                      self_min );
        agent->debugClient().addMessage( "Comm:Receive:Intercept" );
        Body_Intercept().execute( agent );
        agent->setNeckAction( new Neck_TurnToBall() );
        return;
    }

    Vector2D receive_pos = wm.audioMemory().pass().front().receive_pos_;

    dlog.addText( Logger::TEAM,
                  __FILE__": (doReceiveMove) intercept cycle=%d. go to receive point"
                  ,__FILE__, __LINE__,
                  self_min );
    agent->debugClient().setTarget( receive_pos );
    agent->debugClient().addMessage( "Comm:Receive:GoTo" );
    Body_GoToPoint( receive_pos,
                    0.5,
                    ServerParam::i().maxDashPower()
                    ).execute( agent );
    agent->setNeckAction( new Neck_TurnToBall() );
}
