// -*-c++-*-

/*!
  \file cross_generator.cpp
  \brief cross pass generator Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "cross_generator.h"

#include <rcsc/player/world_model.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <rcsc/geom/rect_2d.h>
#include <rcsc/soccer_math.h>
#include <rcsc/timer.h>

#define DEBUG_PROFILE

using namespace rcsc;

namespace {
const double shootable_dist2 = std::pow( 19.0, 2 ); // Magic Number
}

/*-------------------------------------------------------------------*/
/*!

 */
CrossGenerator::CrossGenerator()
{
    M_courses.reserve( 1024 );

    clear();
}

/*-------------------------------------------------------------------*/
/*!

 */
CrossGenerator &
CrossGenerator::instance()
{
    static CrossGenerator s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
CrossGenerator::clear()
{
    M_total_count = 0;
    M_passer = static_cast< AbstractPlayerObject * >( 0 );
    M_start_time.assign( -1, 0 );
    M_first_point.invalidate();
    M_receiver_candidates.clear();
    M_courses.clear();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
CrossGenerator::update( const WorldModel & wm )
{
    static GameTime s_update_time( -1, 0 );
    if ( s_update_time == wm.time() )
    {
        return;
    }
    s_update_time = wm.time();

    clear();

    if ( wm.time().stopped() > 0
         || wm.gameMode().isPenaltyKickMode() )
    {
        return;
    }

#ifdef DEBUG_PROFILE
    Timer timer;
#endif


    updatePasser( wm );

    if ( ! M_passer
         || ! M_first_point.isValid() )
    {
        dlog.addText( Logger::CROSS,
                      __FILE__" (update) passer not found." );
        return;
    }

    createReceivers( wm );

    if ( M_receiver_candidates.empty() )
    {
        dlog.addText( Logger::CROSS,
                      __FILE__" (update) no receiver." );
        return;
    }

    createCourses( wm );

#ifdef DEBUG_PROFILE
    dlog.addText( Logger::CROSS,
                  __FILE__" (update) course_size=%d/%d elapsed %f [ms]",
                  (int)M_courses.size(),
                  M_total_count,
                  timer.elapsedReal() );
    //     std::cerr << "PassGenerator: total_count=" << M_total_count
    //               << " success_count=" << M_courses.size()
    //               << std::endl;
#endif
}

/*-------------------------------------------------------------------*/
/*!

 */
void
CrossGenerator::updatePasser( const WorldModel & wm )
{
    if ( wm.self().isKickable() )
    {
        M_passer = &wm.self();
        M_start_time = wm.time();
        M_first_point = wm.ball().pos();
#ifdef DEBUG_UPDATE_PASSER
        dlog.addText( Logger::PASS,
                      __FILE__" (updatePasser) self kickable." );
#endif
        return;
    }


}

/*-------------------------------------------------------------------*/
/*!

 */
void
CrossGenerator::createReceivers( const WorldModel & wm )
{
    const ServerParam & SP = ServerParam::i();
    const double max_cross_dist2 = std::pow( 30.0, 2 ); // Magic Number
    const Vector2D goal = SP.theirTeamGoalPos();

    const AbstractPlayerCont::const_iterator end = wm.allTeammates().end();
    for ( AbstractPlayerCont::const_iterator p = wm.allTeammates().begin();
          p != end;
          ++p )
    {
        if ( *p == M_passer ) continue;

        if ( M_passer->unum() == wm.self().unum() )
        {
            if ( (*p)->isGhost() ) continue;
            if ( (*p)->unumCount() > 10 ) continue;
            if ( (*p)->posCount() > 10 ) continue;
            if ( (*p)->pos().x > wm.offsideLineX() ) continue;
        }

        if ( (*p)->pos().dist2( goal ) > shootable_dist2 ) continue;
        if ( (*p)->pos().dist2( M_first_point ) > max_cross_dist2 ) continue;

        M_receiver_candidates.push_back( *p );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
CrossGenerator::createCourses( const WorldModel & wm )
{
    for ( AbstractPlayerCont::const_iterator p = M_receiver_candidates.begin();
          p != M_receiver_candidates.end();
          ++p )
    {
        createCourses( wm, *p );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
CrossGenerator::createCourses( const WorldModel & wm,
                               const AbstractPlayerObject * receiver )
{
    const ServerParam & SP = ServerParam::i();
    const double min_first_speed = std::max( SP.defaultPlayerSpeedMax(),
                                             SP.ballSpeedMax() * std::pow( SP.ballDecay(), 15 ) );
    const double max_last_speed = SP.ballSpeedMax() * std::pow( SP.ballDecay(), 5 );
    const double min_last_speed = SP.defaultPlayerSpeedMax();
    const Vector2D goal = SP.theirTeamGoalPos();

    const AngleDeg receiver_angle = ( ( receiver->pos() + receiver->vel() ) - M_first_point ).th();
    const double angle_step = 3.0;

    for ( int a = -2; a < 3; ++a )
    {
        const AngleDeg cross_angle = receiver_angle + ( angle_step * a );
        const Vector2D unit_vel = Vector2D::from_polar( 1.0, cross_angle );

        for ( double first_speed = SP.ballSpeedMax();
              first_speed > min_first_speed;
              first_speed -= 0.3 )
        {
            const Vector2D first_vel = unit_vel * first_speed;

            int our_step = predictReceiverReachStep( receiver, M_first_point, first_vel );
            if ( our_step < 0 )
            {
                continue;
            }

            double last_speed = first_speed * std::pow( SP.ballDecay(), our_step );
            if ( last_speed > max_last_speed
                 || last_speed < min_last_speed )
            {
                continue;
            }

            Vector2D ball_pos = inertia_n_step_point( M_first_point, first_vel, our_step, SP.ballDecay() );
            if ( ball_pos.dist2( goal ) > shootable_dist2 )
            {
                continue;
            }

            int opp_step = predictOpponentReachStep( wm, M_first_point, first_vel, our_step );

            ++M_total_count;

            if ( opp_step < 0
                 || opp_step > our_step + 1 )
            {
                // M_courses.push_back( Pass::Ptr( new Pass() ) );
#ifdef DEBUG_CREATE_PASS_COURSES
                dlog.addText( Logger::CROSS,
                              "OOO (createCourses) %d start=(%.1f %.1f) end=(%.1f %.1f) angle=%.1f speed=(%.2f->%.2f) receiver=%d step=%d opp=%d",
                              M_total_count,
                              M_first_point.x, M_first_point.y,
                              ball_pos.x, ball_pos.y,
                              cross_angle.degree(),
                              first_speed,
                              last_speed,
                              receiver->unum(),
                              our_step,
                              opp_step );
#endif
            }
#ifdef DEBUG_CREATE_PASS_COURSES
            else
            {
                // opponent can intercept.
                dlog.addText( Logger::CROSS,
                              "XXX (createCourses) %d start=(%.1f %.1f) end=(%.1f %.1f) angle=%.1f speed=(%.2f->%.2f) receiver=%d step=%d opp=%d",
                              M_total_count,
                              M_first_point.x, M_first_point.y,
                              ball_pos.x, ball_pos.y,
                              cross_angle.degree(),
                              first_speed,
                              last_speed,
                              receiver->unum(),
                              our_step,
                              opp_step );
            }
#endif
        }
    }

}

/*-------------------------------------------------------------------*/
/*!

 */
int
CrossGenerator::predictReceiverReachStep( const AbstractPlayerObject * receiver,
                                          const Vector2D & first_ball_pos,
                                          const Vector2D & first_ball_vel )
{
    return -1;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
CrossGenerator::predictOpponentReachStep( const WorldModel & wm,
                                          const Vector2D & first_ball_pos,
                                          const Vector2D & first_ball_vel,
                                          const int max_step )
{
    return -1;
}

/*-------------------------------------------------------------------*/
/*!

 */
int
CrossGenerator::predictOpponentReachStep( const PlayerObject & opponent,
                                          const Vector2D & first_ball_pos,
                                          const Vector2D & first_ball_vel,
                                          const AngleDeg & ball_move_angle,
                                          const int max_step )
{
    return -1;
}
