// -*-c++-*-

/*!
  \file default_communication.cpp
  \brief default communication planner Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include "default_communication.h"

#include "strategy.h"

#include <rcsc/formation/formation.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/say_message_builder.h>
#include <rcsc/player/freeform_parser.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/player_param.h>
#include <rcsc/common/audio_memory.h>
#include <rcsc/common/say_message_parser.h>

using namespace rcsc;

namespace {

struct AbstractPlayerSelfDistCmp {
    bool operator()( const AbstractPlayerObject * lhs,
                     const AbstractPlayerObject * rhs ) const
      {
          return lhs->distFromSelf() < rhs->distFromSelf();
      }
};



struct AbstractPlayerXCmp {
    bool operator()( const AbstractPlayerObject * lhs,
                     const AbstractPlayerObject * rhs ) const
      {
          return lhs->pos().x < rhs->pos().x;
      }
};

}

/*-------------------------------------------------------------------*/
/*!

 */
DefaultCommunication::DefaultCommunication()
    : M_ball_send_time( 0, 0 )
{
    for ( int i = 0; i < 12; ++i )
    {
        M_teammate_send_time[i].assign( 0, 0 );
        M_opponent_send_time[i].assign( 0, 0 );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
DefaultCommunication::~DefaultCommunication()
{

}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::execute( PlayerAgent * agent )
{
    if ( ! agent->config().useCommunication()
         || agent->world().gameMode().type() == GameMode::BeforeKickOff
         || agent->world().gameMode().type() == GameMode::AfterGoal_
         || agent->world().gameMode().isPenaltyKickMode() )
    {
        return false;
    }


    sayBall( agent );
    sayDefenseLine( agent );
    sayGoalie( agent );
    //sayOffsideLine( agent );
    saySelf( agent );
    sayPlayers( agent );
    //sayOpponents( agent );
    //sayIntercept( agent );

#if 0
    const int len = effector().getSayMessageLength();
    switch ( len ) {
    case 7:
        addSayMessage( new FreeMessage< 2 >( "10" ) );
        break;
    case 6:
        addSayMessage( new FreeMessage< 3 >( "210" ) );
        break;
    case 5:
        addSayMessage( new FreeMessage< 4 >( "3210" ) );
        break;
    case 4:
        addSayMessage( new FreeMessage< 5 >( "43210" ) );
        break;
    case 3:
        addSayMessage( new FreeMessage< 6 >( "543210" ) );
        break;
    case 2:
        addSayMessage( new FreeMessage< 7 >( "6543210" ) );
        break;
    case 1:
        addSayMessage( new FreeMessage< 8 >( "76543210" ) );
        break;
    default:
        break;
    }
#endif

    attentiontoSomeone( agent );

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
DefaultCommunication::updatePlayerSendTime( const WorldModel & wm,
                                            const SideID side,
                                            const int unum )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << ": Illegal player number. " << unum
                  << std::endl;
        return;
    }

    if ( side == wm.ourSide() )
    {
        M_teammate_send_time[unum] = wm.time();
    }
    else
    {
        M_opponent_send_time[unum] = wm.time();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::sayBall( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + BallMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    if ( wm.ball().seenPosCount() > 0
         || wm.ball().seenVelCount() > 1 )
    {
        // not seen at current cycle
        return false;
    }

    if ( wm.gameMode().type() != GameMode::PlayOn )
    {
        if ( agent->effector().queuedNextBallVel().r2() < 0.5*0.5 )
        {
            return false;
        }
    }

#if 0
    if ( wm.existKickableTeammate()
         || wm.existKickableOpponent() )
    {
        return false;
    }
#endif

    const PlayerObject * ball_nearest_teammate = NULL;
    const PlayerObject * second_ball_nearest_teammate = NULL;

    const PlayerPtrCont::const_iterator t_end = wm.teammatesFromBall().end();
    for ( PlayerPtrCont::const_iterator t = wm.teammatesFromBall().begin();
          t != t_end;
          ++t )
    {
        if ( (*t)->isGhost() || (*t)->posCount() >= 10 ) continue;

        if ( ! ball_nearest_teammate )
        {
            ball_nearest_teammate = *t;
        }
        else if ( ! second_ball_nearest_teammate )
        {
            second_ball_nearest_teammate = *t;
            break;
        }
    }

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int our_min = std::min( self_min, mate_min );
    int opp_min = wm.interceptTable()->opponentReachCycle();


    bool send_ball = false;
    if ( wm.self().isKickable()  // I am kickable
         || ( ! ball_nearest_teammate
              || ( ball_nearest_teammate->distFromBall()
                   > wm.ball().distFromSelf() - 3.0 ) // I am nearest to ball
              || ( wm.ball().distFromSelf() < 20.0
                   && ball_nearest_teammate->distFromBall() < 6.0
                   && ball_nearest_teammate->distFromBall() > 1.0
                   && opp_min <= mate_min + 1 ) )
         || ( second_ball_nearest_teammate
              && ( ball_nearest_teammate->distFromBall()  // nearest to ball teammate is
                   > ServerParam::i().visibleDistance() - 0.5 ) // over vis dist
              && ( second_ball_nearest_teammate->distFromBall()
                   > wm.ball().distFromSelf() - 5.0 ) ) ) // I am second
    {
        send_ball = true;
    }

//     const PlayerObject * nearest_opponent = wm.getOpponentNearestToBall( 10 );
//     if ( nearest_opponent
//          && ( nearest_opponent->distFromBall() >= 10.0
//               || nearest_opponent->distFromBall() < nearest_opponent->playerTypePtr()->kickableArea() + 0.1 ) )
//     {
//         send_ball = false;
//         return false;
//     }

    Vector2D ball_vel = agent->effector().queuedNextBallVel();
    if ( wm.self().isKickable()
         && agent->effector().queuedNextBallKickable() )
    {
        // set ball velocity to 0.
        ball_vel.assign( 0.0, 0.0 );
        dlog.addText( Logger::TEAM,
                      __FILE__": say ball status. next cycle kickable." );
    }

    Vector2D ball_trap_pos = wm.ball().inertiaPoint( our_min );

    //
    // ball & opponent goalie
    //
    if ( send_ball
         && ( wm.existKickableTeammate()
              || our_min <= opp_min + 1 )
         && ball_trap_pos.x > 34.0
         && ball_trap_pos.absY() < 20.0
         && static_cast< int >( len + BallGoalieMessage::slength() ) <= ServerParam::i().playerSayMsgSize() )
    {
        const PlayerObject * opp_goalie = wm.getOpponentGoalie();
        if ( opp_goalie
             && opp_goalie->seenPosCount() == 0
             && opp_goalie->bodyCount() == 0
             && opp_goalie->unum() != Unum_Unknown
             && opp_goalie->unumCount() == 0
             && opp_goalie->distFromSelf() < 25.0
             && opp_goalie->pos().x > 52.5 - 16.0
             && opp_goalie->pos().x < 52.5
             && opp_goalie->pos().absY() < 20.0 )
        {
            agent->addSayMessage( new BallGoalieMessage( agent->effector().queuedNextBallPos(),
                                                         ball_vel,
                                                         opp_goalie->pos(),
                                                         opp_goalie->body() ) );
            M_ball_send_time = wm.time();
            M_opponent_send_time[opp_goalie->unum()] = wm.time();

            dlog.addText( Logger::TEAM,
                          __FILE__": say ball/goalie status" );
            return true;
        }
    }

    //
    // ball & opponent
    //
    if ( send_ball
//          && ball_nearest_teammate
//          && mate_min <= 3
         && static_cast< int >( len + BallPlayerMessage::slength() ) <= ServerParam::i().playerSayMsgSize() )
    {
//         static GameTime s_last_opponent_send_time( 0, 0 );
//         static int s_last_sent_opponent_unum = Unum_Unknown;

        const PlayerObject * opponent = static_cast< const PlayerObject * >( 0 );

        Vector2D opp_trap_pos = wm.ball().inertiaPoint( opp_min );

        const PlayerPtrCont::const_iterator o_end = wm.opponentsFromBall().end();
        for ( PlayerPtrCont::const_iterator o = wm.opponentsFromBall().begin();
              o != o_end;
              ++o )
        {
            if ( (*o)->seenPosCount() > 0 ) continue;
            if ( (*o)->unum() == Unum_Unknown ) continue;
            if ( (*o)->unumCount() > 0 ) continue;
            if ( (*o)->bodyCount() > 0 ) continue;
            if ( (*o)->pos().dist2( opp_trap_pos ) > 15.0*15.0 ) continue;
            //if ( (*o)->distFromBall() > 20.0 ) continue;
            //if ( (*o)->distFromSelf() > 20.0 ) continue;

//             if ( (*o)->unum() == s_last_sent_opponent_unum
//                  && s_last_opponent_send_time.cycle() >= wm.time().cycle() - 1 )
//             {
//                 continue;
//             }

            opponent = *o;
            break;
        }

        if ( opponent )
        {
//             s_last_opponent_send_time = wm.time();
//             s_last_sent_opponent_unum = opponent->unum();

            agent->addSayMessage( new BallPlayerMessage( agent->effector().queuedNextBallPos(),
                                                         ball_vel,
                                                         opponent->unum() + 11,
                                                         opponent->pos(),
                                                         opponent->body() ) );
            M_opponent_send_time[opponent->unum()] = wm.time();

            dlog.addText( Logger::TEAM,
                          __FILE__": say ball/opponent status. opponent=%d",
                          opponent->unum() );
            return true;
        }
    }

    if ( send_ball )
    {
        agent->addSayMessage( new BallMessage( agent->effector().queuedNextBallPos(),
                                               ball_vel ) );
        M_ball_send_time = wm.time();
        dlog.addText( Logger::TEAM,
                      __FILE__": say ball status" );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::sayGoalie( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    // goalie info: ball is in chance area
    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + GoalieMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    if ( wm.ball().pos().x < 34.0
         || wm.ball().pos().absY() > 20.0 )
    {
        return false;
    }

    const PlayerObject * opp_goalie = wm.getOpponentGoalie();
    if ( opp_goalie
         && opp_goalie->seenPosCount() == 0
         && opp_goalie->bodyCount() == 0
         && opp_goalie->unum() != Unum_Unknown
         && opp_goalie->distFromSelf() < 25.0 )
    {
        agent->addSayMessage( new GoalieMessage( opp_goalie->unum(),
                                                 opp_goalie->pos(),
                                                 opp_goalie->body() ) );
        M_ball_send_time = wm.time();
        M_opponent_send_time[opp_goalie->unum()] = wm.time();

        dlog.addText( Logger::TEAM,
                      __FILE__": say goalie info: %d pos=(%.1f %.1f) body=%.1f",
                      opp_goalie->unum(),
                      opp_goalie->pos().x,
                      opp_goalie->pos().y,
                      opp_goalie->body().degree() );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::sayIntercept( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    // intercept info
    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + InterceptMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    if ( wm.gameMode().type() != GameMode::PlayOn )
    {
        return false;
    }

    if ( wm.ball().posCount() > 0
         || wm.ball().velCount() > 0 )
    {
        return false;
    }

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( wm.self().isKickable() )
    {
        double next_dist =  agent->effector().queuedNextMyPos().dist( agent->effector().queuedNextBallPos() );
        if ( next_dist > wm.self().playerType().kickableArea() )
        {
            self_min = 10000;
        }
        else
        {
            self_min = 0;
        }
    }

    if ( self_min <= mate_min
         && self_min <= opp_min
         && self_min <= 10 )
    {
        agent->addSayMessage( new InterceptMessage( true,
                                                    wm.self().unum(),
                                                    self_min ) );
        dlog.addText( Logger::TEAM,
                      __FILE__": say self intercept info %d",
                      self_min );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::sayOffsideLine( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + OffsideLineMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    if ( wm.seeTime() != wm.time()
         || wm.offsideLineCount() > 1
         || wm.opponentsFromSelf().size() < 10 )
    {
        return false;
    }

    int s_min = wm.interceptTable()->selfReachCycle();
    int t_min = wm.interceptTable()->teammateReachCycle();
    int o_min = wm.interceptTable()->opponentReachCycle();

    if ( o_min < t_min
         && o_min < s_min )
    {
        return false;
    }

    int min_step = std::min( s_min, t_min );

    Vector2D ball_pos = wm.ball().inertiaPoint( min_step );

    if ( 0.0 < ball_pos.x
         && ball_pos.x < 37.0
         && ball_pos.x > wm.offsideLineX() - 20.0
         && std::fabs( wm.self().pos().x - wm.offsideLineX() ) < 20.0
         )
    {
        agent->addSayMessage( new OffsideLineMessage( wm.offsideLineX() ) );
        dlog.addText( Logger::TEAM,
                      __FILE__": say offside line %.1f",
                      wm.offsideLineX() );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::sayDefenseLine( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + DefenseLineMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    if ( std::fabs( wm.self().pos().x - wm.defenseLineX() ) > 40.0 )
    {
        return false;
    }

    if ( wm.seeTime() != wm.time() )
    {
        return false;
    }

    int opp_min = wm.interceptTable()->opponentReachCycle();

    Vector2D opp_trap_pos = wm.ball().inertiaPoint( opp_min );

    if ( wm.self().goalie()
         && wm.gameMode().type() == GameMode::PlayOn
         && -40.0 < opp_trap_pos.x
         && opp_trap_pos.x < -20.0
         && wm.time().cycle() % 3 == 0 )
    {
        agent->addSayMessage( new DefenseLineMessage( wm.defenseLineX() ) );
        dlog.addText( Logger::TEAM,
                      __FILE__": say defense line %.1f",
                      wm.defenseLineX() );
        //agent->debugClient().addMessage( "Say_d" );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::sayPlayers( PlayerAgent * agent )
{
    static GameTime s_last_time( -1, 0 );

    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + OnePlayerMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    const WorldModel & wm = agent->world();

//     if ( s_last_time.cycle() >= wm.time().cycle() - 3 )
//     {
//         return false;
//     }

    if ( len == 0
         && wm.time().cycle() % 11 != wm.self().unum() - 1 )
    {
        return false;
    }

    bool opponent_attack = false;

    int opp_min = wm.interceptTable()->opponentReachCycle();
    int mate_min = wm.interceptTable()->opponentReachCycle();
    int self_min = wm.interceptTable()->opponentReachCycle();

    Vector2D opp_trap_pos = wm.ball().inertiaPoint( opp_min );

    if ( opp_min <= mate_min
         && opp_min <= self_min )
    {
        if ( opp_trap_pos.x < 10.0 )
        {
            opponent_attack = true;
        }
    }

    AbstractPlayerCont candidates;
    bool include_self = false;

    // set self as candidate
    if ( M_teammate_send_time[wm.self().unum()] != wm.time()
         && ( ! opponent_attack
              || wm.self().pos().x < opp_trap_pos.x + 10.0 )
         && wm.time().cycle() % 11 == ( wm.self().unum() - 1 ) )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (sayPlayers) candidate self(1)" );
        candidates.push_back( &(wm.self()) );
        include_self = true;
    }

    if ( ! opponent_attack )
    {
        // set teammate candidates
        const PlayerPtrCont::const_iterator t_end = wm.teammatesFromSelf().end();
        for ( PlayerPtrCont::const_iterator t = wm.teammatesFromSelf().begin();
              t != t_end;
              ++t )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": (sayPlayers) __ check(1) teammate %d(%.1f %.1f)",
                          (*t)->unum(),
                          (*t)->pos().x, (*t)->pos().y );

            if ( (*t)->seenPosCount() > 0 ) continue;
            //if ( (*t)->seenVelCount() > 0 ) continue;
            if ( (*t)->unumCount() > 0 ) continue;
            if ( (*t)->unum() == Unum_Unknown ) continue;

            if ( M_teammate_send_time[(*t)->unum()].cycle() >= wm.time().cycle() - 1 )
            {
                continue;
            }

            dlog.addText( Logger::TEAM,
                          __FILE__": (sayPlayers) candidate teammate(1) %d",
                          (*t)->unum() );
            candidates.push_back( *t );
        }
    }

    // set opponent candidates
    const PlayerPtrCont::const_iterator o_end = wm.opponentsFromSelf().end();
    for ( PlayerPtrCont::const_iterator o = wm.opponentsFromSelf().begin();
          o != o_end;
          ++o )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (sayPlayers) __ check opponent %d(%.1f %.1f)",
                      (*o)->unum(),
                      (*o)->pos().x, (*o)->pos().y );

        if ( (*o)->seenPosCount() > 0 ) continue;
        //if ( (*o)->seenVelCount() > 0 ) continue;
        if ( (*o)->unumCount() > 0 ) continue;
        if ( (*o)->unum() == Unum_Unknown ) continue;

        if ( ! opponent_attack )
        {
            if ( M_opponent_send_time[(*o)->unum()].cycle() >= wm.time().cycle() - 1 )
            {
                dlog.addText( Logger::TEAM,
                              __FILE__": (sayPlayers) __ check opponent %d. recently sent",
                              (*o)->unum() );
                continue;
            }
        }
        else
        {
            if ( (*o)->pos().x > 0.0
                 && (*o)->pos().x > wm.defenseLineX() + 15.0 )
            {
                dlog.addText( Logger::TEAM,
                              __FILE__": (sayPlayers) __ check opponent %d. x>0 && x>defense_line+15",
                              (*o)->unum() );
                continue;
            }
        }

        dlog.addText( Logger::TEAM,
                      __FILE__": (sayPlayers) candidate opponent %d",
                      (*o)->unum() );
        candidates.push_back( *o );
    }

    if ( //opponent_attack &&
        ! candidates.empty()
         && candidates.size() < 3 )
    {
        // set self
        if ( ! include_self
             && M_teammate_send_time[wm.self().unum()] != wm.time()
             && wm.time().cycle() % 11 == ( wm.self().unum() - 1 ) )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": (sayPlayers) candidate self(2)" );
            candidates.push_back( &(wm.self()) );
        }

        // set teammate candidates
        const PlayerPtrCont::const_iterator t_end = wm.teammatesFromSelf().end();
        for ( PlayerPtrCont::const_iterator t = wm.teammatesFromSelf().begin();
              t != t_end;
              ++t )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": (sayPlayers) __ check(2) teammate %d(%.1f %.1f)",
                          (*t)->unum(),
                          (*t)->pos().x, (*t)->pos().y );

            if ( (*t)->seenPosCount() > 0 ) continue;
            //if ( (*t)->seenVelCount() > 0 ) continue;
            if ( (*t)->unumCount() > 0 ) continue;
            if ( (*t)->unum() == Unum_Unknown ) continue;

            if ( M_teammate_send_time[(*t)->unum()].cycle() >= wm.time().cycle() - 1 )
            {
                continue;
            }

            dlog.addText( Logger::TEAM,
                          __FILE__": (sayPlayers) candidate teammate(2) %d",
                          (*t)->unum() );
            candidates.push_back( *t );
        }
    }


    if ( candidates.empty() )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (sayPlayers) no candidate" );
        return false;
    }


    if ( opponent_attack )
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (sayPlayers) opponent attack. sorted by X." );
        std::sort( candidates.begin(), candidates.end(), AbstractPlayerXCmp() );
    }
    else
    {
        dlog.addText( Logger::TEAM,
                      __FILE__": (sayPlayers) NO opponent attack. sorted by distance from self." );
        std::sort( candidates.begin(), candidates.end(), AbstractPlayerSelfDistCmp() );
    }

    AbstractPlayerCont::const_iterator first = candidates.begin();
    int first_unum = ( (*first)->side() == wm.ourSide()
                       ? (*first)->unum()
                       : (*first)->unum() + 11 );

    if ( candidates.size() >= 3
         && static_cast< int >( len + ThreePlayerMessage::slength() ) <= ServerParam::i().playerSayMsgSize() )
    {
        AbstractPlayerCont::const_iterator second = first; ++second;
        int second_unum = ( (*second)->side() == wm.ourSide()
                            ? (*second)->unum()
                            : (*second)->unum() + 11 );
        AbstractPlayerCont::const_iterator third = second; ++third;
        int third_unum = ( (*third)->side() == wm.ourSide()
                           ? (*third)->unum()
                           : (*third)->unum() + 11 );

        agent->addSayMessage( new ThreePlayerMessage( first_unum,
                                                      (*first)->pos() + (*first)->vel(),
                                                      second_unum,
                                                      (*second)->pos() + (*second)->vel(),
                                                      third_unum,
                                                      (*third)->pos() + (*third)->vel() ) );
        updatePlayerSendTime( wm, (*first)->side(), (*first)->unum() );
        updatePlayerSendTime( wm, (*second)->side(), (*second)->unum() );
        updatePlayerSendTime( wm, (*third)->side(), (*third)->unum() );

        dlog.addText( Logger::TEAM,
                      __FILE__": say three players %c%d %c%d %c%d",
                      (*first)->side() == wm.ourSide() ? 'T' : 'O', (*first)->unum(),
                      (*second)->side() == wm.ourSide() ? 'T' : 'O', (*second)->unum(),
                      (*third)->side() == wm.ourSide() ? 'T' : 'O', (*third)->unum() );

        //agent->debugClient().addMessage( "Say_%c%d_%c%d_%c%d",
        //                                 (*first)->side() == wm.ourSide() ? 'T' : 'O', (*first)->unum(),
        //                                 (*second)->side() == wm.ourSide() ? 'T' : 'O', (*second)->unum(),
        //                                 (*third)->side() == wm.ourSide() ? 'T' : 'O', (*third)->unum() );
    }
    else if ( candidates.size() >= 2
              && static_cast< int >( len + TwoPlayerMessage::slength() ) <= ServerParam::i().playerSayMsgSize() )
    {
        AbstractPlayerCont::const_iterator second = first; ++second;
        int second_unum = ( (*second)->side() == wm.ourSide()
                            ? (*second)->unum()
                            : (*second)->unum() + 11 );
        agent->addSayMessage( new TwoPlayerMessage( first_unum,
                                                    (*first)->pos() + (*first)->vel(),
                                                    second_unum,
                                                    (*second)->pos() + (*second)->vel() ) );
        updatePlayerSendTime( wm, (*first)->side(), (*first)->unum() );
        updatePlayerSendTime( wm, (*second)->side(), (*second)->unum() );

        dlog.addText( Logger::TEAM,
                      __FILE__": say two players %c%d %c%d",
                      (*first)->side() == wm.ourSide() ? 'T' : 'O', (*first)->unum(),
                      (*second)->side() == wm.ourSide() ? 'T' : 'O', (*second)->unum() );

        //agent->debugClient().addMessage( "Say_%c%d_%c%d",
        //                                 (*first)->side() == wm.ourSide() ? 'T' : 'O', (*first)->unum(),
        //                                 (*second)->side() == wm.ourSide() ? 'T' : 'O', (*second)->unum() );
    }
    else if ( len > 0 )
    {
        agent->addSayMessage( new OnePlayerMessage( first_unum,
                                                    (*first)->pos() + (*first)->vel() ) );
        updatePlayerSendTime( wm, (*first)->side(), (*first)->unum() );

        dlog.addText( Logger::TEAM,
                      __FILE__": say two players %c%d %c%d",
                      (*first)->side() == wm.ourSide() ? 'T' : 'O', (*first)->unum() );

        //agent->debugClient().addMessage( "Say_%c%d",
        //                                 (*first)->side() == wm.ourSide() ? 'T' : 'O', (*first)->unum() );
    }

    s_last_time = wm.time();
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::sayOpponents( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    if ( sayThreeOpponents( agent ) )
    {
        return true;
    }

    if ( sayTwoOpponents( agent ) )
    {
        return true;
    }

    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + OpponentMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( opp_min > self_min + 10
         && opp_min > mate_min + 10 )
    {
        return false;
    }

    const PlayerObject * fastest_opponent = wm.interceptTable()->fastestOpponent();
    if ( fastest_opponent
         && fastest_opponent->unum() != Unum_Unknown
         && fastest_opponent->unumCount() == 0
         && fastest_opponent->seenPosCount() == 0
         && fastest_opponent->bodyCount() == 0
         && 10.0 < fastest_opponent->distFromSelf()
         && fastest_opponent->distFromSelf() < 30.0 )
    {
        agent->addSayMessage( new OpponentMessage( fastest_opponent->unum(),
                                                   fastest_opponent->pos(),
                                                   fastest_opponent->body() ) );
        M_opponent_send_time[fastest_opponent->unum()] = wm.time();

        dlog.addText( Logger::TEAM,
                      __FILE__": say opponent status" );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::sayTwoOpponents( PlayerAgent * agent )
{
    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + TwoPlayerMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    const WorldModel & wm = agent->world();

    std::vector< const PlayerObject * > candidates;

    const PlayerPtrCont::const_iterator o_end = wm.opponentsFromSelf().end();
    for ( PlayerPtrCont::const_iterator o = wm.opponentsFromSelf().begin();
          o != o_end;
          ++o )
    {
        if ( (*o)->seenPosCount() > 0 ) continue;
        if ( (*o)->unumCount() > 0 ) continue;

        if ( M_opponent_send_time[(*o)->unum()].cycle() >= wm.time().cycle() - 1 )
        {
            continue;
        }

        candidates.push_back( *o );
    }

    if ( candidates.size() < 2 )
    {
        return false;
    }

    std::vector< const PlayerObject * >::const_iterator first = candidates.begin();
    std::vector< const PlayerObject * >::const_iterator second = first; ++second;

    agent->addSayMessage( new TwoPlayerMessage( (*first)->unum() + 11,
                                                (*first)->pos() + (*first)->vel(),
                                                (*second)->unum() + 11,
                                                (*second)->pos() + (*second)->vel() ) );
    M_opponent_send_time[(*first)->unum()] = wm.time();
    M_opponent_send_time[(*second)->unum()] = wm.time();


    dlog.addText( Logger::TEAM,
                  __FILE__": say two oppoinents %d %d",
                  (*first)->unum(),
                  (*second)->unum() );
    //agent->debugClient().addMessage( "Say_O%d_O%d",
    //                                 (*first)->unum(),
    //                                 (*second)->unum() );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::sayThreeOpponents( PlayerAgent * agent )
{
    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + ThreePlayerMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    const WorldModel & wm = agent->world();

    std::vector< const PlayerObject * > candidates;

    const PlayerPtrCont::const_iterator o_end = wm.opponentsFromSelf().end();
    for ( PlayerPtrCont::const_iterator o = wm.opponentsFromSelf().begin();
          o != o_end;
          ++o )
    {
        if ( (*o)->seenPosCount() > 0 ) continue;
        if ( (*o)->unumCount() > 0 ) continue;

        if ( M_opponent_send_time[(*o)->unum()].cycle() >= wm.time().cycle() - 1 )
        {
            continue;
        }

        candidates.push_back( *o );
    }

    if ( candidates.size() < 3 )
    {
        return false;
    }

    std::vector< const PlayerObject * >::const_iterator first = candidates.begin();
    std::vector< const PlayerObject * >::const_iterator second = first; ++second;
    std::vector< const PlayerObject * >::const_iterator third = second; ++third;

    agent->addSayMessage( new ThreePlayerMessage( (*first)->unum() + 11,
                                                  (*first)->pos() + (*first)->vel(),
                                                  (*second)->unum() + 11,
                                                  (*second)->pos() + (*second)->vel(),
                                                  (*third)->unum() + 11,
                                                  (*third)->pos() + (*third)->vel() ) );
    M_opponent_send_time[(*first)->unum()] = wm.time();
    M_opponent_send_time[(*second)->unum()] = wm.time();
    M_opponent_send_time[(*third)->unum()] = wm.time();

    dlog.addText( Logger::TEAM,
                  __FILE__": say three oppoinents %d %d %d",
                  (*first)->unum(),
                  (*second)->unum(),
                  (*third)->unum() );
    //agent->debugClient().addMessage( "Say_O%d_O%d_O%d",
    //                                 (*first)->unum(),
    //                                 (*second)->unum(),
    //                                 (*third)->unum() );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
DefaultCommunication::saySelf( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    const int len = agent->effector().getSayMessageLength();
    if ( static_cast< int >( len + SelfMessage::slength() ) > ServerParam::i().playerSayMsgSize() )
    {
        return false;
    }

    //     if ( std::fabs( wm.self().pos().x - wm.defenseLineX() ) > 40.0 )
    //     {
    //         return false;
    //     }

    if ( std::fabs( wm.self().distFromBall() ) > 35.0 )
    {
        return false;
    }

    if ( M_teammate_send_time[wm.self().unum()].cycle() >= wm.time().cycle() )
    {
        return false;
    }

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( opp_min < self_min
         && opp_min < mate_min )
    {
        return false;
    }

    bool send_self = false;

    if ( len > 0 )
    {
        // if another type message is already registered, self info is appended to the message.
        send_self = true;
    }

    if ( ! send_self
         && wm.time().cycle() % 11 == ( wm.self().unum() - 1 ) )
    {
        const PlayerObject * ball_nearest_teammate = NULL;
        const PlayerObject * second_ball_nearest_teammate = NULL;

        const PlayerPtrCont::const_iterator t_end = wm.teammatesFromBall().end();
        for ( PlayerPtrCont::const_iterator t = wm.teammatesFromBall().begin();
              t != t_end;
              ++t )
        {
            if ( (*t)->isGhost() || (*t)->posCount() >= 10 ) continue;

            if ( ! ball_nearest_teammate )
            {
                ball_nearest_teammate = *t;
                dlog.addText( Logger::TEAM,
                              __FILE__": (saySelf) ball_nearest_teammate %d (%.1f %.1f)",
                              (*t)->unum(),
                              (*t)->pos().x,  (*t)->pos().y );
            }
            else if ( ! second_ball_nearest_teammate )
            {
                second_ball_nearest_teammate = *t;
                dlog.addText( Logger::TEAM,
                              __FILE__": (saySelf) second_ball_nearest_teammate %d (%.1f %.1f)",
                              (*t)->unum(),
                              (*t)->pos().x,  (*t)->pos().y );
                break;
            }
        }

        if ( ball_nearest_teammate
             && ball_nearest_teammate->distFromBall() < wm.ball().distFromSelf()
             && ( ! second_ball_nearest_teammate
                  || second_ball_nearest_teammate->distFromBall() > wm.ball().distFromSelf() )
             )
        {
            send_self = true;
        }
    }

    if ( send_self )
    {
        agent->addSayMessage( new SelfMessage( agent->effector().queuedNextMyPos(),
                                               agent->effector().queuedNextMyBody(),
                                               wm.self().stamina() ) );
        M_teammate_send_time[wm.self().unum()] = wm.time();

        dlog.addText( Logger::TEAM,
                      __FILE__": say self status" );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
DefaultCommunication::attentiontoSomeone( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();

    if ( wm.self().pos().x > wm.offsideLineX() - 15.0
         && wm.interceptTable()->selfReachCycle() <= 3 )
    {
        std::vector< const PlayerObject * > candidates;
        const PlayerPtrCont::const_iterator end = wm.teammatesFromSelf().end();
        for ( PlayerPtrCont::const_iterator p = wm.teammatesFromSelf().begin();
              p != end;
              ++p )
        {
            if ( (*p)->goalie() ) continue;
            if ( (*p)->unum() == Unum_Unknown ) continue;
            if ( (*p)->pos().x > wm.offsideLineX() + 1.0 ) continue;

            if ( (*p)->distFromSelf() > 20.0 ) break;

            candidates.push_back( *p );
        }

        const Vector2D self_next = agent->effector().queuedNextSelfPos();

        const PlayerObject * target_teammate = NULL;
        double max_x = -10000000.0;
        for ( std::vector< const PlayerObject * >::const_iterator p = candidates.begin();
              p != candidates.end();
              ++p )
        {
            Vector2D diff = (*p)->pos() + (*p)->vel() - self_next;

            double x = diff.x * ( 1.0 - diff.absY() / 40.0 ) ;

            if ( x > max_x )
            {
                max_x = x;
                target_teammate = *p;
            }
        }

        if ( target_teammate )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": attentionto most front teammate",
                          wm.offsideLineX() );
            agent->debugClient().addMessage( "AttFrontMate%d", target_teammate->unum() );
            agent->doAttentionto( wm.ourSide(), target_teammate->unum() );
            return;
        }

        // maybe ball owner
        if ( wm.self().attentiontoUnum() > 0 )
        {
            dlog.addText( Logger::TEAM,
                          __FILE__": attentionto off. maybe ball owner",
                          wm.offsideLineX() );
            agent->debugClient().addMessage( "AttOffBOwner" );
            agent->doAttentiontoOff();
        }
        return;
    }

    const PlayerObject * fastest_teammate = wm.interceptTable()->fastestTeammate();
    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( fastest_teammate
         && fastest_teammate->unum() != Unum_Unknown )
    {
        if ( mate_min < self_min
             && mate_min <= opp_min + 1
             && mate_min <= 5 + std::min( 4, fastest_teammate->posCount() )
             && wm.ball().inertiaPoint( mate_min ).dist2( agent->effector().queuedNextSelfPos() )
             < std::pow( 35.0, 2 ) )
        {
            // set attention to ball nearest teammate
            agent->debugClient().addMessage( "AttBallOwner%d", fastest_teammate->unum() );
            agent->doAttentionto( wm.ourSide(), fastest_teammate->unum() );
            return;
        }
    }

    const PlayerObject * nearest_teammate = wm.getTeammateNearestToBall( 5 );

    if ( nearest_teammate
         && nearest_teammate->unum() != Unum_Unknown
         && opp_min <= 3
         && opp_min <= mate_min
         && opp_min <= self_min
         && nearest_teammate->distFromSelf() < 45.0
         && nearest_teammate->distFromBall() < 20.0 )
    {
        agent->debugClient().addMessage( "AttBallNearest(1)%d", nearest_teammate->unum() );
        agent->doAttentionto( wm.ourSide(), nearest_teammate->unum() );
        return;
    }

    if ( nearest_teammate
         && nearest_teammate->unum() != Unum_Unknown
         && wm.ball().posCount() >= 3
         && nearest_teammate->distFromBall() < 20.0 )
    {
        agent->debugClient().addMessage( "AttBallNearest(2)%d", nearest_teammate->unum() );
        agent->doAttentionto( wm.ourSide(), nearest_teammate->unum() );
        return;
    }

    if ( nearest_teammate
         && nearest_teammate->unum() != Unum_Unknown
         && nearest_teammate->distFromSelf() < 45.0
         && nearest_teammate->distFromBall() < 2.0 )
    {
        agent->debugClient().addMessage( "AttBallNearest(3)%d", nearest_teammate->unum() );
        agent->doAttentionto( wm.ourSide(), nearest_teammate->unum() );
        return;
    }

    agent->debugClient().addMessage( "AttOff" );
    agent->doAttentiontoOff();
}
