// -*-c++-*-

/*!
  \file defense_system.cpp
  \brief team defense system manager Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA, Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "defense_system.h"

#include "strategy.h"

#include <rcsc/player/world_model.h>
#include <rcsc/player/intercept_table.h>

#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>

#include <rcsc/geom/vector_2d.h>

using namespace rcsc;

/*-------------------------------------------------------------------*/
/*!

*/
double
DefenseSystem::get_defender_dash_power( const WorldModel & wm,
                                        const Vector2D & home_pos )
{
    static bool S_recover_mode = false;

    if ( wm.self().staminaModel().capacityIsEmpty() )
    {
        S_recover_mode = false;
        return std::min( ServerParam::i().maxDashPower(),
                         wm.self().stamina() + wm.self().playerType().extraStamina() );
    }
    else if ( wm.self().stamina() < ServerParam::i().staminaMax() * 0.5 )
    {
        S_recover_mode = true;
    }
    else if ( wm.self().stamina()
              > ServerParam::i().staminaMax() * 0.85 )
    {
        S_recover_mode = false;
    }

    const double ball_xdiff
        //= wm.ball().pos().x - home_pos.x;
        = wm.ball().pos().x - wm.self().pos().x;

    const PlayerType & ptype = wm.self().playerType();
    const double my_inc = ptype.staminaIncMax() * wm.self().recovery();

    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    double dash_power;
    if ( S_recover_mode )
    {
#if 1
        // 2009-06-29
        if ( opp_min <= mate_min + 1
             && wm.self().pos().x < home_pos.x - 10.0 )
        {
            dlog.addText( Logger::TEAM,
                                __FILE__": (get_defender)dash_power) recover mode, but opponent ball. max power" );
            dash_power = ServerParam::i().maxDashPower();
        }
        else
#endif
        if ( wm.defenseLineX() > wm.self().pos().x )
        {
            dlog.addText( Logger::TEAM,
                                __FILE__": get_dash_power. correct DF line & recover" );
            dash_power = my_inc;
        }
        else if ( ball_xdiff < 5.0 )
        {
            dash_power = ServerParam::i().maxDashPower();
        }
        else if ( ball_xdiff < 10.0 )
        {
            dash_power = ServerParam::i().maxDashPower();
            dash_power *= 0.7;
            //dash_power
            //    = ptype.getDashPowerToKeepSpeed( 0.7, wm.self().effort() );
        }
        else if ( ball_xdiff < 20.0 )
        {
            dash_power = std::max( 0.0, my_inc - 10.0 );
        }
        else // >= 20.0
        {
            dash_power = std::max( 0.0, my_inc - 20.0 );
        }

        dlog.addText( Logger::TEAM,
                            __FILE__": get_dash_power. recover mode dash_power= %.1f",
                            dash_power );

        return dash_power;
    }

    // normal case

#if 1
    // added 2006/06/11 03:34
    if ( wm.ball().pos().x > 0.0
         && wm.self().pos().x < home_pos.x
         && wm.self().pos().x > wm.defenseLineX() - 0.05 ) // 20080712
    {
        double power_for_max_speed = ptype.getDashPowerToKeepMaxSpeed( wm.self().effort() );
        double defense_dash_dist = wm.self().pos().dist( Vector2D( -48.0, 0.0 ) );
        int cycles_to_reach = ptype.cyclesToReachDistance( defense_dash_dist );
        int available_dash_cycles = ptype.getMaxDashCyclesSavingRecovery( power_for_max_speed,
                                                                          wm.self().stamina(),
                                                                          wm.self().recovery() );
        if ( available_dash_cycles < cycles_to_reach )
        {
            dlog.addText( Logger::TEAM,
                                __FILE__": get_dash_power. keep stamina for defense back dash,"
                                " power_for_max=%.1f"
                                " dash_dist=%.1f, reach_cycle=%d, dashable_cycle=%d",
                                power_for_max_speed,
                                defense_dash_dist, cycles_to_reach, available_dash_cycles );
            dash_power = std::max( 0.0, my_inc - 20.0 );
            return dash_power;
        }
    }
#endif

#if 1
    if ( wm.defenseLineX() < wm.self().pos().x - 2.0
         && wm.self().pos().x < home_pos.x )
    {
        dash_power = my_inc * 0.5;
        dlog.addText( Logger::TEAM,
                            __FILE__": get_dash_power. over defense line. power=%.1f",
                            dash_power );
    }
    else
#endif
    if ( wm.self().pos().x < -30.0
         && wm.defenseLineX() > wm.self().pos().x )
    {
        dash_power = ServerParam::i().maxDashPower();
        dlog.addText( Logger::TEAM,
                            __FILE__": get_dash_power. correct dash power for the defense line. power=%.1f",
                            dash_power );
    }
    else if ( home_pos.x < wm.self().pos().x )
    {
        dash_power = ServerParam::i().maxDashPower();
        dlog.addText( Logger::TEAM,
                            __FILE__": get_dash_power. max power to go to the behind home position. power=%.1f",
                            dash_power );
    }
    else if ( ball_xdiff > 20.0 )
    {
        dash_power = my_inc;
        dlog.addText( Logger::TEAM,
                            __FILE__": get_dash_power. correct dash power to save stamina(1). power=%.1f",
                            dash_power );
    }
    else if ( ball_xdiff > 10.0 )
    {
        dash_power = ServerParam::i().maxDashPower();
        dash_power *= 0.6;
        //dash_power = mytype.getDashPowerToKeepSpeed( 0.6, wm.self().effort() );
        dlog.addText( Logger::TEAM,
                            __FILE__": get_dash_power. correct dash power to save stamina(2). power=%.1f",
                            dash_power );
    }
    else if ( ball_xdiff > 5.0 )
    {
        dash_power = ServerParam::i().maxDashPower();
        dash_power *= 0.85;
        //dash_power = mytype.getDashPowerToKeepSpeed( 0.85, wm.self().effort() );
        dlog.addText( Logger::TEAM,
                            __FILE__": get_dash_power. correct dash power to save stamina(3). power=%.1f",
                            dash_power );
    }
    else
    {
        dash_power = ServerParam::i().maxDashPower();
        dlog.addText( Logger::TEAM,
                            __FILE__": get_dash_power. max power. power=%.1f",
                            dash_power );
    }

    return dash_power;
}

/*-------------------------------------------------------------------*/
/*!

*/
Vector2D
DefenseSystem::get_block_opponent_trap_point( const WorldModel & wm )
{
    int opp_min = wm.interceptTable()->opponentReachCycle();
    Vector2D opp_trap_pos = wm.ball().inertiaPoint( opp_min );

#ifdef DEBUG_PRINT_BLODK
    dlog.addText( Logger::BLOCK,
                  __FILE__": opponent trap pos (%.3f %.3f)",
                  opp_trap_pos.x, opp_trap_pos.y );
#endif

    const PlayerObject * opp = wm.interceptTable()->fastestOpponent();
    if ( opp )
    {
        Vector2D tmp_trap_pos = opp_trap_pos;
        if ( opp_min == 0 )
        {
            if ( opp->bodyCount() <= 1 )
            {
                tmp_trap_pos = opp->inertiaPoint( 2 )
                    + Vector2D::polar2vector( 0.4, opp->body() );
#ifdef DEBUG_PRINT_BLODK
                dlog.addText( Logger::BLOCK,
                              __FILE__": adjust opponent trap point to the body direction(1) (%.3f %.3f)",
                              tmp_trap_pos.x, tmp_trap_pos.y );
#endif
            }
            else
            {
                tmp_trap_pos = opp->inertiaPoint( 10 );
#ifdef DEBUG_PRINT_BLODK
                dlog.addText( Logger::BLOCK,
                              __FILE__": adjust opponent trap point to the inertia point (%.3f %.3f)",
                              tmp_trap_pos.x, tmp_trap_pos.y );
#endif
            }
        }
        else if ( opp_min <= 1
                  && opp->distFromSelf() < ServerParam::i().visibleDistance()
                  && opp->bodyCount() <= 1 )
        {
            tmp_trap_pos += Vector2D::polar2vector( 0.4, opp->body() );
#ifdef DEBUG_PRINT_BLOCK
            dlog.addText( Logger::BLOCK,
                          __FILE__": adjust opponent trap point to the body direction(2) (%.3f %.3f)",
                          tmp_trap_pos.x, tmp_trap_pos.y );
#endif
        }

        Vector2D center( -44.0, 0.0 );
        if ( tmp_trap_pos.dist2( center ) < opp_trap_pos.dist2( center ) )
        {
#ifdef DEBUG_PRINT_BLOCK
            dlog.addText( Logger::BLOCK,
                          __FILE__": change opponent trap pos (%.3f %.3f) -> (%.3f %.3f)",
                          opp_trap_pos.x, opp_trap_pos.y,
                          tmp_trap_pos.x, tmp_trap_pos.y );
#endif
            opp_trap_pos = tmp_trap_pos;
        }
    }

    return opp_trap_pos;
}

/*-------------------------------------------------------------------*/
/*!

*/
Vector2D
DefenseSystem::get_block_center_point( const WorldModel & wm )
{
    Vector2D opp_trap_pos = get_block_opponent_trap_point( wm );

    //
    // searth the best point
    //
    Vector2D center_pos( -44.0, 0.0 );
    if ( opp_trap_pos.x < -38.0
         && opp_trap_pos.absY() < 7.0 )
    {
        center_pos.x = -52.5;
#if 0
        center_pos.y = opp_trap_pos.y * 0.6;
#else
        // 2009-06-17
        center_pos.y = -2.0 * sign( wm.self().pos().y );
#endif
        dlog.addText( Logger::BLOCK,
                      __FILE__": center (1) (%.1f %.1f)",
                      center_pos.x, center_pos.y );
    }
    else if ( opp_trap_pos.x < -38.0
              && opp_trap_pos.absY() < 9.0 )
    {
        center_pos.x = std::min( -49.0, opp_trap_pos.x - 0.2 );
        //center_pos.y = opp_trap_pos.y * 0.6;
        Vector2D goal_pos( - ServerParam::i().pitchHalfLength(),
                           ServerParam::i().goalHalfWidth() * 0.5 );
        if ( opp_trap_pos.y > 0.0 )
        {
            goal_pos.y *= -1.0;
        }

        Line2D opp_line( opp_trap_pos, goal_pos );
        center_pos.y = opp_line.getY( center_pos.x );
        if ( center_pos.y == Line2D::ERROR_VALUE )
        {
            center_pos.y = opp_trap_pos.y * 0.6;
        }

        dlog.addText( Logger::BLOCK,
                      __FILE__": center (2) (%.1f %.1f)",
                      center_pos.x, center_pos.y );
    }
    else if ( opp_trap_pos.x < -38.0
              && opp_trap_pos.absY() < 12.0 )
    {
        //center_pos.x = -50.0;
        center_pos.x = std::min( -49.0, opp_trap_pos.x - 0.2 );
        //center_pos.y = 2.5 * sign( opp_trap_pos.y );
        //center_pos.y = 6.5 * sign( opp_trap_pos.y );
        center_pos.y = opp_trap_pos.y * 0.8;
        dlog.addText( Logger::BLOCK,
                      __FILE__": center (3) (%.1f %.1f)",
                      center_pos.x, center_pos.y );
    }
    else if ( opp_trap_pos.x < -30.0
              && 2.0 < opp_trap_pos.absY()
              && opp_trap_pos.absY() < 8.0 )
    {
        center_pos.x = -50.0;
        center_pos.y = opp_trap_pos.y * 0.9;
        dlog.addText( Logger::BLOCK,
                      __FILE__": center (4) (%.1f %.1f)",
                      center_pos.x, center_pos.y );
    }
    else if ( opp_trap_pos.absY() > 25.0 )
    {
        center_pos.x = -44.0;
#if 0
        center_pos.y = 20.0 * sign( opp_trap_pos.y );
#else
        // 2009-06-17
        center_pos.y = 16.0 * sign( opp_trap_pos.y );
#endif
        dlog.addText( Logger::BLOCK,
                      __FILE__": center (5) (%.1f %.1f)",
                      center_pos.x, center_pos.y );
    }
    else if ( opp_trap_pos.absY() > 20.0 )
    {
        center_pos.x = -44.0;
        //center_pos.y = 15.0 * sign( opp_trap_pos.y );
        center_pos.y = 5.0 * sign( opp_trap_pos.y );
        dlog.addText( Logger::BLOCK,
                      __FILE__": center (6) (%.1f %.1f)",
                      center_pos.x, center_pos.y );
    }
    else if ( opp_trap_pos.absY() > 15.0 )
    {
        center_pos.x = -44.0;
        //center_pos.y = 10.0 * sign( opp_trap_pos.y );
        center_pos.y = 5.0 * sign( opp_trap_pos.y );
        dlog.addText( Logger::BLOCK,
                      __FILE__": center (7) (%.1f %.1f)",
                      center_pos.x, center_pos.y );
    }
    else
    {
        dlog.addText( Logger::BLOCK,
                      __FILE__": center(default) (%.1f %.1f)",
                      center_pos.x, center_pos.y );
    }

    return center_pos;
}
