// -*-c++-*-

/*!
  \file field_analyzer.h
  \brief miscellaneous field analysis class Header File
*/

/*
 *Copyright:

  Copyright (C) Hidehisa AKIYAMA

  This code is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3, or (at your option)
  any later version.

  This code is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this code; see the file COPYING.  If not, write to
  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef FIELD_ANALYZER_H
#define FIELD_ANALYZER_H

#include "player_graph.h"

#include <rcsc/geom/voronoi_diagram.h>
#include <rcsc/geom/vector_2d.h>
#include <cmath>

namespace rcsc {
class ServerParam;
class PlayerType;
class AbstractPlayerObject;
class StaminaModel;
}

class FieldAnalyzer {
private:

    rcsc::VoronoiDiagram M_all_players_voronoi_diagram;
    rcsc::VoronoiDiagram M_teammates_voronoi_diagram;
    rcsc::VoronoiDiagram M_pass_voronoi_diagram;
    PlayerGraph M_field_player_graph;

    FieldAnalyzer();
public:

    static
    FieldAnalyzer & instance();

    static
    const
    FieldAnalyzer & i()
      {
          return instance();
      }

    static
    int estimate_min_reach_cycle( const rcsc::Vector2D & player_pos,
                                  const double & player_speed_max,
                                  const rcsc::Vector2D & target_point,
                                  const rcsc::AngleDeg & target_move_angle );

    static
    double estimate_virtual_dash_distance( const rcsc::AbstractPlayerObject * player );

    static
    int predict_player_turn_cycle( const rcsc::PlayerType * player_type,
                                   const rcsc::AngleDeg & player_body,
                                   const double & player_speed,
                                   const double & target_dist,
                                   const rcsc::AngleDeg & target_angle,
                                   const double & dist_thr );

    static
    int predict_self_reach_cycle( const rcsc::WorldModel & wm,
                                  const rcsc::Vector2D & target_point,
                                  const double & dist_thr,
                                  const int wait_cycle,
                                  const bool save_recovery,
                                  rcsc::StaminaModel * stamina );

    /*!
      \param dist_thr kickable_area or catchable_area
      \param penalty_distance result of estimate_virtual_dash_distance()
      \param wait_cycle penalty of kick, tackle or observation delay.
     */
    static
    int predict_player_reach_cycle( const rcsc::AbstractPlayerObject * player,
                                    const rcsc::Vector2D & target_point,
                                    const double & dist_thr,
                                    const double & penalty_distance,
                                    const int body_count_thr,
                                    const int default_n_turn,
                                    const int wait_cycle );

    static
    bool can_shoot_from( const rcsc::WorldModel & wm,
                         const rcsc::Vector2D & pos,
                         const int valid_opponent_threshold );
    static
    double get_shoot_free_angle_from_pos( const rcsc::WorldModel & wm,
                                          const rcsc::Vector2D & pos,
                                          const long valid_opponent_threshold );
    static
    double get_shoot_minimum_free_angle( const rcsc::WorldModel & wm,
                                         const rcsc::Vector2D & goal,
                                         const rcsc::Vector2D & pos,
                                         const int valid_opponent_threshold );

    const
    rcsc::VoronoiDiagram & allPlayersVoronoiDiagram() const
      {
          return M_all_players_voronoi_diagram;
      }

    const
    rcsc::VoronoiDiagram & teammatesVoronoiDiagram() const
      {
          return M_teammates_voronoi_diagram;
      }

    const
    rcsc::VoronoiDiagram & passVoronoiDiagram() const
      {
          return M_pass_voronoi_diagram;
      }

    const
    PlayerGraph & fieldPlayerGraph() const
      {
          return M_field_player_graph;
      }

    void update( const rcsc::WorldModel & wm );


private:

    void updateVoronoiDiagram( const rcsc::WorldModel & wm );
    void updatePlayerGraph( const rcsc::WorldModel & wm );

    void writeDebugLog();

};


#include <rcsc/player/abstract_player_object.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/player_type.h>


/*-------------------------------------------------------------------*/
/*!

 */
inline
int
FieldAnalyzer::estimate_min_reach_cycle( const rcsc::Vector2D & player_pos,
                                         const double & player_speed_max,
                                         const rcsc::Vector2D & target_first_point,
                                         const rcsc::AngleDeg & target_move_angle )
{
    rcsc::Vector2D target_to_player = ( player_pos - target_first_point ).rotatedVector( -target_move_angle );
    return ( target_to_player.x < -1.0
             ? -1
             : std::max( 1, static_cast< int >( std::floor( target_to_player.absY() / player_speed_max ) ) ) );
}

inline
double
FieldAnalyzer::estimate_virtual_dash_distance( const rcsc::AbstractPlayerObject * player )
{
    const int pos_count = std::min( 10, // Magic Number
                                    std::min( player->seenPosCount(),
                                              player->posCount() ) );
    const double max_speed = player->playerTypePtr()->realSpeedMax();

    double d = 0.0;
    for ( int i = 1; i <= pos_count; ++i ) // start_value==1 to set the initial_value<1
    {
        d += max_speed * std::exp( - (i*i) / 20.0 ); // Magic Number
    }

    return d;
}

inline
int
FieldAnalyzer::predict_player_turn_cycle( const rcsc::PlayerType * ptype,
                                          const rcsc::AngleDeg & player_body,
                                          const double & player_speed,
                                          const double & target_dist,
                                          const rcsc::AngleDeg & target_angle,
                                          const double & dist_thr )
{
    int n_turn = 0;

    double angle_diff = ( target_angle - player_body ).abs();
    double turn_margin = 180.0;
    if ( dist_thr < target_dist )
    {
        turn_margin = std::max( 15.0, // Magic Number
                                rcsc::AngleDeg::asin_deg( dist_thr / target_dist ) );
    }

    double speed = player_speed;
    while ( angle_diff > turn_margin )
    {
        angle_diff -= ptype->effectiveTurn( rcsc::ServerParam::i().maxMoment(), speed );
        speed *= ptype->playerDecay();
        ++n_turn;
    }

    return n_turn;
}


#endif
