// TLogPlayer.cpp: implementation of the TLogPlayer class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Soccermonitor.h"
#include "SoccermonitorDoc.h"
#include "TLogPlayer.h"
#include "DCycleSlider.h"
#include "RCGLoadingDialog.h"
#include <string.h>
#include <stdio.h>

using namespace std;

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

BEGIN_MESSAGE_MAP(TLogPlayer, CWnd)
	//{{AFX_MSG_MAP(TLogPlayer)
	ON_WM_TIMER()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


IMPLEMENT_SERIAL(TLogPlayer, CWnd, MONITOR_VERSION)




const int TLogPlayer::S_COACH_UNUM = 100;
const int TLogPlayer::S_SCORE_REWIND_COUNT = 50;


inline
long nstonl(const short ns)
{
    return htonl((long)(((double)(short)ntohs(ns) / SHOWINFO_SCALE) * SHOWINFO_SCALE2));
}

inline
short nltons(const long &nl)
{
    double val = (double)(long)ntohl((long)nl) / SHOWINFO_SCALE2;
    return (short)htons((short)(val * SHOWINFO_SCALE + (val > 0.0 ? 0.5 : -0.5)));
}


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

TLogPlayer::TLogPlayer(CSoccermonitorDoc* pDoc)
{
	int	i;
	
	m_pDoc = pDoc;

	M_running = false;
	for ( i = 0; i < MAX_CYCLES; i++ )
    {
        m_pLog[i] = NULL;
    }

    M_protocol_version = 2;

    M_ball_locus_pen = new CPen;
    M_ball_locus_pen2 = new CPen;
    M_left_player_locus_pen = new CPen;
    M_right_player_locus_pen = new CPen;
    M_player_locus_pen2 = new CPen;

    if ( ! M_ball_locus_pen->CreatePen(PS_DOT, 1, COLOR_BALL_LOCUS) )
    {
        M_ball_locus_pen->CreateStockObject(WHITE_PEN);
    }
    if ( ! M_ball_locus_pen2->CreatePen(PS_DOT, 1, COLOR_YELLOW) )
    {
        M_ball_locus_pen2->CreateStockObject(WHITE_PEN);
    }

    if ( ! M_left_player_locus_pen->CreatePen(PS_DOT, 1, CSoccermonitorDoc::COLOR_TEAM_L) )
    {
        M_left_player_locus_pen->CreateStockObject(WHITE_PEN);
    }
    if ( ! M_right_player_locus_pen->CreatePen(PS_DOT, 1, CSoccermonitorDoc::COLOR_TEAM_R) )
    {
        M_right_player_locus_pen->CreateStockObject(WHITE_PEN);
    }

    if ( ! M_player_locus_pen2->CreatePen(PS_DOT, 1, COLOR_PLAYER_LOCUS2) )
    {
        M_player_locus_pen2->CreateStockObject(WHITE_PEN);
    }

	InitLog();
}

TLogPlayer::~TLogPlayer()
{
	int	i;

	for ( i = 0; i < MAX_CYCLES; i++)
    {
		if (m_pLog[i] != NULL)
        {
			delete m_pLog[i];
            m_pLog[i] = NULL;
        }
    }

    M_ball_locus_pen->DeleteObject();
    M_ball_locus_pen2->DeleteObject();
    M_left_player_locus_pen->DeleteObject();
    M_right_player_locus_pen->DeleteObject();
    M_player_locus_pen2->DeleteObject();

    delete M_ball_locus_pen;
    delete M_ball_locus_pen2;
    delete M_left_player_locus_pen;
    delete M_right_player_locus_pen;
    delete M_player_locus_pen2;
}

/*-----------------------------------------------------------------------------
Function: void TLogPlayer::Serialize(CArchive& ar)
Date    : 20.05.98
-------------------------------------------------------------------------------
Parameter: reference to the archive to store/load to/from
Returnval: (void)
Description: does the storing and loading of logfiles
-----------------------------------------------------------------------------*/
void TLogPlayer::Serialize(CArchive& ar)
{
	if ( ar.IsStoring() )
	{
		// is it windows or unix format
		// name = ar.GetFile()->GetFileName();
	}
	else
	{
		// load logfile

        BeginWaitCursor();

		InitLog();

        CString filepath = ar.GetFile()->GetFilePath();
        CGZFile gzfile(filepath);
        if ( ! gzfile.Good() )
        {
            AfxMessageBox("Cannot load file!");
            return;
        }

        const int flen = (int)ar.GetFile()->GetLength();

        BYTE header[4];
        gzfile.Read(header, 4);
        TRACE3("header %c %c %c\n", header[0], header[1], header[2]);
        TRACE1("  version %d\n", header[3]);

        bool success = false;
        if ( header[0] == 'U' && header[1] == 'L' && header[2] == 'G' )
        {
            M_log_version = (int)header[3];
            if ( M_log_version == 3 )
             {
                success = true;
                M_protocol_version = 2;
                loadRCG3(gzfile, flen);
            }
            else if ( M_log_version == 2 )
            {
                success = true;
                M_protocol_version = 1;
                loadRCG2(gzfile, flen);
            }
        }

        if ( ! success )
        {
            gzfile.Rewind();
            loadRCG1(gzfile, flen);
            if ( M_max_cycles > 1 )
            {
                M_log_version = 1;
                M_protocol_version = 1;
                M_server_param.psize = 0.8;
                M_server_param.kmargin = 1.0;
                M_player_types[0].player_size = 0.8;
                M_player_types[0].kickable_margin = 1.0;
                success = true;
            }
        }
        
        EndWaitCursor();
        
        if ( ! success )
        {
            AfxMessageBox("Unsupported log file!!");
            return;
        }

        M_game_title = ar.GetFile()->GetFileName();
        CWinApp* pApp = AfxGetApp();
        if ( pApp && pApp->m_pMainWnd )
        {
            pApp->m_pMainWnd->SetWindowText(M_game_title + " - SoccerWindow");
        }
        M_running = false;
        M_has_data = true;
        rewind();        
    }
}



void TLogPlayer::initRCLData()
{
    M_has_rcl_data = false;
    for ( int j = 0; j < MAX_CYCLES; j++ )
    {
        M_rcl_say[j].clear();
        M_rcl_attentionto[j].clear();
        M_rcl_pointto[j].clear();
    }
}


void TLogPlayer::loadRCL(CGZFile & gzfile)
{
    BYTE header[4];
    gzfile.Read(header, 4);
    if ( header[0] != '0' || header[1] != '\t' && header[2] != '(' )
    {
        AfxMessageBox("rcl format error!");
        return;
    }
    gzfile.Rewind();


    const char *PlaymodeString[] = PLAYMODE_STRINGS;
    int i;

    CString left_team_name, right_team_name;

    char buf[10000];
    char *next;
    long cycle;
    PlayMode pmode;

    RCGLoadingDialog *loading = new RCGLoadingDialog;
    loading->Create(IDD_RCG_LOAD);
    loading->CenterWindow();
    //loading->M_loading.SetRange32(1, file_len);
    loading->M_loading.SetRange32(0, 6000);
    loading->ShowWindow(SW_SHOW);

    int read_count = 1000;

    // init
    initRCLData();

    while ( gzfile.Good() && gzfile.Gets(buf, 10000) )
    {
        buf[9999] = '\0';
        // read cycle
        cycle = strtol(buf, &next, 10);

        if ( cycle > MAX_CYCLES - 1 )
        {
            break;
        }
        if ( read_count == 0 )
        {
            read_count = 1000;
            if ( cycle > 9000 )
            {
                loading->M_loading.SetPos((int)cycle - 6000 + 1);
            }
            if ( cycle > 6000 )
            {
                loading->M_loading.SetPos((int)cycle - 3000 + 1);
            }
            else
            {
                loading->M_loading.SetPos((int)cycle + 1);
            }
        }
        read_count--;

        while ( *next == ' ' || *next == '\t' ) next++;

        if ( *next == '(' ) // referee
        {
            while ( *next != ' ' ) next++; // skip "(referee "
            next++; // skip space
            for ( i = 0; i < (int)PM_MAX; i++ )
            {
                if ( ! strncmp(next, PlaymodeString[i], strlen(PlaymodeString[i])) )
                {
                    pmode = (PlayMode)i;
                    break;
                }
            }
            continue;
        }

        if ( strncmp(next, "Recv", 4) != 0 ) // not Recv
        {
            continue;
        }

        while ( *next != ' ' ) next++; // skip "Recv "
        next++;

        char idstr[32];
        sscanf(next, "%s", idstr);
        CString name(idstr);
        int team_name_len = 0;

        // get team name
        if ( left_team_name.IsEmpty() )
        {
            team_name_len = name.ReverseFind('_');
            if ( team_name_len != -1 )
            {
                left_team_name = name.Left(team_name_len);
            }
            if ( strncmp(left_team_name, M_team_name_left, 16) )
            {
                initRCLData();
                AfxMessageBox("left team name in rcl data is incorrect.");
                break;
            }
        }
        else if ( right_team_name.IsEmpty() )
        {
            if ( strncmp(name, left_team_name, left_team_name.GetLength())
                 || name.ReverseFind('_') != left_team_name.GetLength() )
            {
                team_name_len = name.ReverseFind('_');
                if ( team_name_len != -1 )
                {
                    right_team_name = name.Left(team_name_len);
                }
                if ( strncmp(right_team_name, M_team_name_right, 16) )
                {
                    initRCLData();
                    AfxMessageBox("right team name in rcl data is incorrect.");
                    break;
                }
            }
        }

        if ( cycle == 0 )
        {
            continue;
        }

        if ( pmode == PM_BEFORE_KICK_OFF
             || pmode == PM_OFFSIDE_L || pmode == PM_OFFSIDE_L
             || pmode == PM_BACK_PASS_L || pmode == PM_BACK_PASS_R
             || pmode == PM_FREE_KICK_FAULT_L || pmode == PM_FREE_KICK_FAULT_R
             || pmode == PM_CATCH_FAULT_L || pmode == PM_CATCH_FAULT_R
             || pmode == PM_GOAL_L || pmode == PM_GOAL_R )
        {
            continue;
        }
        if ( pmode == PM_TIME_OVER )
        {
            break;
        }


        char side;
        int unum;

        // check team name
        if ( ! strncmp(name, left_team_name, left_team_name.GetLength())
             && name.ReverseFind('_') == left_team_name.GetLength() )
        {
            next += left_team_name.GetLength();
            side = 'l';
        }
        else if ( ! strncmp(name, right_team_name, right_team_name.GetLength())
                  && name.ReverseFind('_') == right_team_name.GetLength() )
        {
            next += right_team_name.GetLength();
            side = 'r';
        }
        else
        {
            break;
        }

        next++; // skip '_'
        if ( *next == 'C' ) // coach
        {
            unum = S_COACH_UNUM;
        }
        else
        {
            unum = (int)strtol(next, NULL, 10);
            if ( side == 'r' ) unum += 11;
        }

        bool has_data = false;
        while ( 1 )
        {
            while ( *next != '\0' && *next != '(' )
            {
                next++;
            }
            if ( *next == '\0' )
            {
                break;
            }

            if ( ! strncmp(next, "(say", strlen("(say")) )
            {
                RCLSay rcl_say(unum);

                while ( *next != ' ' ) next++;
                next++;
                bool quated = false;
                if ( unum == S_COACH_UNUM ) // coach
                {
                    rcl_say.message = next;
                    rcl_say.message.Delete(rcl_say.message.GetLength() - 1, 1);
                    M_rcl_coach_say[cycle].push_back(rcl_say);
                }
                else if ( *next == '\"' )
                {
                    next++;
                    while ( *next != '\"' )
                    {
                        rcl_say.message += *next;
                        next++;
                    }
                    M_rcl_say[cycle].push_back(rcl_say);
                }
                else
                {
                    while ( *next != ')' )
                    {
                        rcl_say.message += *next;
                        next++;
                    }
                    M_rcl_say[cycle].push_back(rcl_say);
                }

            }
            else if ( ! strncmp(next, "(attentionto", strlen("(attentionto")) )
            {
                while ( *next != ' ' ) next++;
                next++;
                char tmpname[32];
                int num;
                if ( sscanf(next, "%s %d", tmpname, &num ) == 2 )
                {
                    RCLAttentionto rcl_att(side, unum);
                    rcl_att.target_name = tmpname;
                    rcl_att.target_num = num;
                    M_rcl_attentionto[cycle].push_back(rcl_att);
                }
            }
            else if ( ! strncmp(next, "(pointto", strlen("(pointto")) )
            {
                while ( *next != ' ' ) next++;
                next++;
                double dist, dir;
                if ( sscanf(next, "%lf %lf", &dist, &dir) == 2 )
                {
                    RCLPointto rcl_pt(side, unum);
                    rcl_pt.dist = dist;
                    rcl_pt.dir = dir;
                    M_rcl_pointto[cycle].push_back(rcl_pt);
                }
            }

            next++;
        }
    }
    M_has_rcl_data = true;

    for ( long c = 1; c <= cycle; c++ )
    {
        M_rcl_say[c].sort();
    }
    
    loading->DestroyWindow();
    delete loading;
}

void TLogPlayer::loadRCG1(CGZFile & gzfile, const int file_len)
{
    int i;
    dispinfo_t buf;
    char playmode;
    team_t team[2];
    short_showinfo_t2 sshow2;
    unsigned int mustRead = sizeof(dispinfo_t);

    RCGLoadingDialog *loading = new RCGLoadingDialog;
    loading->Create(IDD_RCG_LOAD);
    loading->CenterWindow();
    //loading->M_loading.SetRange32(1, file_len);
    loading->M_loading.SetRange32(0, 6000);
    loading->ShowWindow(SW_SHOW);

    int count = 300;
    while ( 1 )
    {
        if ( count == 0 )
        {
            count = 300;
            //loading->M_loading.SetPos(gzfile.Tell());
            loading->M_loading.SetPos((int)ntohs(buf.body.show.time) + 1);
        }
        count--;

        if ( gzfile.Read(&buf, mustRead) != (int)mustRead )
        {
            break;
        }

        if ( ntohs(buf.mode) == SHOW_MODE )
        {
            pos2ball(&sshow2.ball, &(buf.body.show.pos[0]));
            for ( i = 1; i < MAX_PLAYER * 2 + 1; i++ )
            {
                pos2player(&sshow2.pos[i-1], &buf.body.show.pos[i]);
            }
            team[0] = buf.body.show.team[0];
            team[1] = buf.body.show.team[1];
            playmode = buf.body.show.pmode;
            sshow2.time = buf.body.show.time;
            addLoginfo(playmode, team, sshow2);
        }
    }

    char tname[17];
    if ( team[0].name )
    {
        strncpy(tname, team[0].name, 16);
        M_team_name_left.Format("%s", team[0].name);
    }
    if ( team[1].name )
    {
        strncpy(tname, team[1].name, 16);
        M_team_name_right.Format("%s", team[1].name);
    }

    loading->DestroyWindow();
    delete loading;
}

void TLogPlayer::loadRCG2(CGZFile & gzfile, const int file_len)
{
    dispinfo_t buf;
    char playmode;
    team_t team[2];
    short_showinfo_t2 sshow2;

    unsigned int mustRead;
    short len;
    int i;

    CString errmsg;

    RCGLoadingDialog *loading = new RCGLoadingDialog;
    loading->Create(IDD_RCG_LOAD);
    loading->CenterWindow();
    //loading->M_loading.SetRange32(1, file_len);
    loading->M_loading.SetRange32(0, 6000);
    loading->ShowWindow(SW_SHOW);

    int count = 500;
    bool flag = true;
    while ( flag )
    {
        if ( count == 0 )
        {
            count = 500;
            loading->M_loading.SetPos((int)ntohs(buf.body.show.time) + 1);
        }
        count--;

    	if ( gzfile.Read(&(buf.mode), sizeof(buf.mode) ) == 0 )
        {
            break;
        }
    	buf.mode = ntohs(buf.mode);

        switch ( buf.mode ) {
        case NO_INFO:
            break;
        case SHOW_MODE:
            mustRead = sizeof(showinfo_t);
            if ( gzfile.Read(&buf.body.show, mustRead) != (int)mustRead )
            {
                errmsg = "SHOW_MODE";
                flag = false;
                break;
            }

            pos2ball(&sshow2.ball, &(buf.body.show.pos[0]));
            for ( i = 1; i < MAX_PLAYER * 2 + 1; i++ )
            {
                pos2player(&sshow2.pos[i-1], &buf.body.show.pos[i]);
            }
            team[0] = buf.body.show.team[0];
            team[1] = buf.body.show.team[1];
            playmode = buf.body.show.pmode;
            sshow2.time = buf.body.show.time;

            addLoginfo(playmode, team, sshow2);
            break;
        case MSG_MODE:
    		mustRead = sizeof(buf.body.msg.board);
    		if ( gzfile.Read(&(buf.body.msg.board), mustRead) != (int)mustRead )
            {
                errmsg = "MSG_MODE board";
                flag = false;
                break;
            }
    		buf.body.msg.board = ntohs(buf.body.msg.board);

    		// read the string length
	    	mustRead = sizeof(len);
    		if ( gzfile.Read(&len, mustRead) != (int)mustRead )
            {
                errmsg = "MSG_MODE len";
                flag = false;
                break;
            }

       		// read the message
    		mustRead = ntohs(len);
    		if ( gzfile.Read(buf.body.msg.message, mustRead) != (int)mustRead )
            {
                errmsg = "MSG_MODE message";
                flag = false;
                break;
            }
            buf.body.msg.message[mustRead-2] = '\0';
    		//addMsginfo(&buf.body.msg);	
    		break;
        default:
            //flag = false;
            break;
        };
    }
 
    if ( ! flag )
    {
        AfxMessageBox("read error v2 log!!");
    }

    char tname[17];
    if ( team[0].name )
    {
        strncpy(tname, team[0].name, 16);
        M_team_name_left.Format("%s", tname);
    }
    if ( team[1].name )
    {
        strncpy(tname, team[1].name, 16);
        M_team_name_right.Format("%s", tname);
    }

    loading->DestroyWindow();
    delete loading;
}


void TLogPlayer::loadRCG3(CGZFile & gzfile, const int file_len)
{
    short mode;
    char playmode;
    team_t team[2];
    msginfo_t msg;
    player_type_t ptinfo;
    server_params_t	sparams;
    //server_params_t2 sparam2;
    player_params_t pparams;

	unsigned int mustRead;
	short len;

    short_showinfo_t2 short_showinfo;

    CString errmsg;
    
    RCGLoadingDialog *loading = new RCGLoadingDialog;
    loading->Create(IDD_RCG_LOAD);
    loading->CenterWindow();
    //loading->M_loading.SetRange32(1, file_len);
    loading->M_loading.SetRange32(0, 6000);
    loading->ShowWindow(SW_SHOW);
    
    int count = 500, read_count = 0;;
    bool flag = true;
	// read up to a show cycle first
	while ( flag )
    {
        read_count++;
        if ( count == 0 )
        {
            count = 500;
            int curtime = (int)ntohs(short_showinfo.time);
            loading->M_loading.SetPos(curtime % 6000 + 1);
        }
        count--;

    	// read the type of next msg
    	if ( gzfile.Read(&mode, sizeof(mode) ) == 0 )
        {
            break; 
        }
        mode = ntohs(mode);

    	switch ( mode ) {
    	case NO_INFO:
            break;
    	case SHOW_MODE:
    		mustRead = sizeof(short_showinfo_t2);
    		if ( gzfile.Read(&short_showinfo, mustRead) != (int)mustRead )
            {
                errmsg = "SHOW_MODE";
                flag = false;
                break;
            }
            addLoginfo(playmode, team, short_showinfo);
            break;

    	case MSG_MODE:
    		// read the board info
    		mustRead = sizeof(msg.board);
    		if ( gzfile.Read(&(msg.board), mustRead) != (int)mustRead )
            {
                errmsg = "MSG_MODE board ";
                flag = false;
                break;
            }
    		msg.board = ntohs(msg.board);
    		// read the string length
    		mustRead = sizeof(len);
    		if ( gzfile.Read(&len, mustRead) != (int)mustRead )
            {
                errmsg = "MSG_MODE len";
                flag = false;
                break;
            }
    		// read the message
    		mustRead = ntohs(len);
    		if ( gzfile.Read(msg.message, mustRead) != (int)mustRead )
            {
                errmsg = "MSG_MODE message ";
                flag = false;
            }
            msg.message[mustRead-2] = '\0';
    		// addMsginfo(&msg);	
	        break;

        case DRAW_MODE:
            break;
        case BLANK_MODE:
            break;
        case PM_MODE:
            mustRead = sizeof(char);
    		if ( gzfile.Read(&playmode, mustRead) != (int)mustRead )
            {
                errmsg = "PM_MODE";
                flag = false;
                break;
            }
            break;
        case TEAM_MODE:
            mustRead = sizeof(team_t) * 2;
            if ( gzfile.Read(team, mustRead) != (int)mustRead )
            {
                errmsg = "TEAM_MODE";
                flag = false;
                break;
            }
            break;
        case PT_MODE:
            mustRead = sizeof(player_type_t);
            if ( gzfile.Read(&ptinfo, mustRead) != (int)mustRead )
            {
                errmsg = "PT_MODE";
                flag = false;
                break;
            }
            setParam(&ptinfo);
            break;
        case PARAM_MODE:
#if 0
            mustRead = sizeof(server_params_t2);
            if ( gzfile.Read(&sparam2, mustRead) != (int)mustRead )
            {
                flag = false;
                break;
            }
            M_server_param.convert(&sparam2);
#endif
            mustRead = sizeof(server_params_t);
            if ( gzfile.Read(&sparams, mustRead) != (int)mustRead )
            {
                errmsg = "PARAM_MODE";
                flag = false;
                break;
            }
            M_server_param.convert(&sparams);
            break;
        case PPARAM_MODE:
            mustRead = sizeof(player_params_t);
            if ( gzfile.Read(&pparams, mustRead) != (int)mustRead )
            {
                errmsg = "PPARAM_MODE";
                flag = false;
                break;
            }
            M_player_param.convert(&pparams);
            break;
        default:
            //errmsg.Format("Unknown Mode?? %d  read_count %d", mode, read_count);
            //flag = false;
            break;
    	}
    }

    if ( flag == false )
    {
        CString msg = "read error v3 log at " + errmsg + " !!";
        AfxMessageBox(msg);
    }

    char tname[17];
    if ( team[0].name )
    {
        strncpy(tname, team[0].name, 16);
        M_team_name_left.Format("%s", tname);
    }
    if ( team[1].name )
    {
        strncpy(tname, team[1].name, 16);
        M_team_name_right.Format("%s", tname);
    }

    loading->DestroyWindow();
    delete loading;
}


bool TLogPlayer::saveSegmentLog(const CString& filename,
                                short markin, short markout)
{
    CFile file;
	short startcycle = ntohs(m_pLog[markin]->time);
	if ( ! file.Open(filename, CFile::modeCreate | CFile::modeWrite) )
	{
		return false;
	}

    if ( M_protocol_version >= 2 )
    {
        saveRCG3(file, markin, markout);
    }
    else
    {
        saveRCG2(file, markin, markout);
    }

    return true;
}



void TLogPlayer::saveRCG2(CFile &file, const int start, const int end)
{
    RCGLoadingDialog *loading = new RCGLoadingDialog;
    loading->Create(IDD_RCG_LOAD);
    loading->CenterWindow();
    loading->M_loading.SetRange32(start, end);
    loading->ShowWindow(SW_SHOW);


    int i;
    BYTE pmode = 0;
    short mark;
    short startcycle = ntohs(m_pLog[start]->time);

    // header
	char logtag[3] = {'U', 'L', 'G'};
    BYTE ver = 2;
	file.Write(logtag, 3);
	file.Write(&ver, sizeof(BYTE));

    // each cycle
    // only SHOW
    showinfo_t buf;
    const int COUNT_INC = (end - start) / 20 + 1;
    int count = COUNT_INC;
	for ( i = start; i <= end; i++ )
	{
        if ( count == 0 )
        {
            count = COUNT_INC;
            loading->M_loading.SetPos(i);
        }
        count--;

		if ( m_pLog[i] == NULL ) continue;

		mark = htons(SHOW_MODE);
	    fillShowinfo2ToSegment1(buf, *m_pLog[i], startcycle);

		file.Write(&mark, sizeof(mark));
		file.Write(&buf, sizeof(showinfo_t));
	}

	file.Close();

    loading->DestroyWindow();//>ShowWindow(SW_HIDE);
    delete loading;
}

void TLogPlayer::saveRCG3(CFile &file, const int start, const int end)
{

    RCGLoadingDialog *loading = new RCGLoadingDialog;
    loading->Create(IDD_RCG_LOAD);
    loading->CenterWindow();
    loading->M_loading.SetRange32(start, end);
    loading->ShowWindow(SW_SHOW);

    int i;
    BYTE pmode = 0;
    team_t teams[2];
    short mark;
    short startcycle = ntohs(m_pLog[start]->time);

    // header
	char logtag[3] = {'U', 'L', 'G'};
    BYTE ver = 3;
	file.Write(logtag, 3);
	file.Write(&ver, sizeof(BYTE));

    // params
    server_params_t sparam;
    M_server_param.toStruct(&sparam);
    mark = htons(PARAM_MODE);
    file.Write(&mark, sizeof(mark));
    file.Write(&sparam, sizeof(server_params_t));

    player_params_t pparam;
    M_player_param.toStruct(&pparam);
    mark = htons(PPARAM_MODE);
    file.Write(&mark, sizeof(mark));
    file.Write(&pparam, sizeof(player_params_t));

    player_type_t ptinfo;
    mark = htons(PT_MODE);
    for ( i = 0; i < numPlayerTypes(); i++ )
    {
        M_player_types[i].toStruct(&ptinfo);
        file.Write(&mark, sizeof(mark));
        file.Write(&ptinfo, sizeof(player_type_t));
    }

    // first playmode
    mark = htons(PM_MODE);
    pmode = m_pLog[start]->pmode;
    file.Write(&mark, sizeof(mark));
    file.Write(&pmode, sizeof(pmode));

    // first team info
	strncpy(teams[0].name, M_team_name_left, 16);
	strncpy(teams[1].name, M_team_name_right, 16);
    teams[0].score = htons(m_pLog[start]->scoreL);
    teams[1].score = htons(m_pLog[start]->scoreR);
    mark = htons(TEAM_MODE);
    file.Write(&mark, sizeof(mark));
    file.Write(&teams, sizeof(teams));

    // each cycle
    short_showinfo_t2 buf;
    const int COUNT_INC = (end - start) / 20 + 1;
    int count = COUNT_INC;
	for ( i = start; i <= end; i++ )
	{
        if ( count == 0 )
        {
            count = COUNT_INC;
            loading->M_loading.SetPos(i);
        }
        count--;

		if ( m_pLog[i] == NULL ) continue;

		if ( pmode != m_pLog[i]->pmode )
		{
			mark = htons(PM_MODE);
			pmode = m_pLog[i]->pmode;
            file.Write(&mark, sizeof(mark));
            file.Write(&pmode, sizeof(pmode));
		}
		if ( teams[0].score != htons(m_pLog[i]->scoreL)
             || teams[1].score != htons(m_pLog[i]->scoreR) )
		{
			teams[0].score = htons(m_pLog[i]->scoreL);
			teams[1].score = htons(m_pLog[i]->scoreR);
			mark = htons(TEAM_MODE);
			file.Write(&mark, sizeof(mark));
			file.Write(&teams, sizeof(teams));
		}

		mark = htons(SHOW_MODE);
		fillShowinfo2ToSegment2(buf, *m_pLog[i], startcycle);

		file.Write(&mark, sizeof(mark));
		file.Write(&buf, sizeof(short_showinfo_t2));
	}

	file.Close();

    loading->DestroyWindow();//ShowWindow(SW_HIDE);
    delete loading;
}




void TLogPlayer::InitLog()
{
	int	i;

	// if the log is running stop it
    stop();

	// initialize the info pointers
	for ( i = 0; i < MAX_CYCLES; i++)
	{
		if ( m_pLog[i] != NULL )
		{
			delete m_pLog[i];
			m_pLog[i] = NULL;
		}
		m_strMsg[i].Empty();
	}

    M_scored_pos.clear();
    //M_cycle_infos.clear();

	M_running = false;
	M_has_data = false;
    M_forward = true;
	M_max_cycles = 0;
	M_current_cycle = 0;
    M_timer_id = 0;
	M_timer_msec = DEFAULT_SPEED;

    M_team_name_left.Empty();
    M_team_name_right.Empty();

    M_server_param.setDefaultParam();
    M_player_types[0].setDefaultParam();
    M_num_player_types = 0;
    for ( i = 1; i < 7; i++ )
    {
        M_player_types[i] = M_player_types[0];
    }

    M_game_title.Empty();

    CWnd *pFrame = AfxGetApp()->m_pMainWnd;
    if ( pFrame )
    {
        pFrame->SetWindowText(_T("SoccerWindow")); 
    }

    initRCLData();
    M_has_rcl_data = false;
}

/*
bool TLogPlayer::addLoginfo(const char pmode, const team_t *team,
                            const short_showinfo_t2 &sshow2)
{
    myshowinfo_t2 myshow;

    // store the values
    myshow.pmode = pmode;
	myshow.scoreL = ntohs((short)team[0].score);		
	myshow.scoreR = ntohs((short)team[1].score);

    myshow.ball = sshow2.ball;
    memcpy(myshow.pos, sshow2.pos, sizeof(sshow2.pos));

    myshow.time = (int)sshow2.time;
    M_cycle_infos.push_back(myshow);

    M_max_cycles = (int)M_cycle_infos.size();
	return true;
}
*/

bool TLogPlayer::addLoginfo(const char pmode, const team_t *team,
                            const short_showinfo_t2 &sshow2)
{
    static int index = 0;
    static short s_last_score_l = 0;
    static short s_last_score_r = 0;

    if ( ntohs(sshow2.time) <= 1 )
    {
        index = 0;
        s_last_score_l = s_last_score_r = 0;
    }

	// create storage for the values
	m_pLog[index] = new myshowinfo_t2;
    if ( m_pLog[index] == NULL )
    {
        return false;
    }

	// store the values
    m_pLog[index]->pmode = pmode;
	m_pLog[index]->scoreL = (short)ntohs((short)team[0].score);		
	m_pLog[index]->scoreR = (short)ntohs((short)team[1].score);
    if ( s_last_score_l != m_pLog[index]->scoreL
         || s_last_score_r != m_pLog[index]->scoreR )
    {
        s_last_score_l = m_pLog[index]->scoreL;
        s_last_score_r = m_pLog[index]->scoreR;
        M_scored_pos.push_back(index);
    }

    m_pLog[index]->ball = sshow2.ball;
    memcpy(m_pLog[index]->pos, sshow2.pos, sizeof(sshow2.pos));
    
    m_pLog[index]->time = (int)sshow2.time;

    index++;
    M_max_cycles = index;

	return true;

}

// for monitor client
bool TLogPlayer::addShowinfo(showinfo_t2* pInfo)
{
	int	i;
	//int	index = ntohs(pInfo->time) ;
    //index = index - 1;
    static int index = 0;
    static short s_last_score_l = 0;
    static short s_last_score_r = 0;

    if ( ntohs(pInfo->time) <= 1 )
    {
        index = 0;
        s_last_score_l = s_last_score_r = 0;
    }

	if ( index < 0 || index > MAX_CYCLES )
	{
		// we are not within a game
		if ( m_pLog[0] != NULL )
		{
			delete m_pLog[0];
			m_pLog[0] = NULL;
		}
		index = 0;
	}

	// test if cycle is fitting and if this item is not already existing
	if ( m_pLog[index] != NULL )
	{
		//TRACE1("Rejected a Cycle %d\n", index);
		//return FALSE;
		delete m_pLog[index];
        m_pLog[index] = NULL;
	}

	// create storage for the values
	m_pLog[index] = new myshowinfo_t2;
    if ( m_pLog[index] == NULL )
    {
        return false;
    }

	// store the values
	m_pLog[index]->pmode = pInfo->pmode;
    if ( M_team_name_left.IsEmpty() && pInfo->team[0].name )
    {
        char tname[17];
        strncpy(tname, pInfo->team[0].name, 16);
        M_team_name_left.Format("%s", tname);
    }
    if ( M_team_name_right.IsEmpty() && pInfo->team[1].name )
    {
        char tname[17];
        strncpy(tname, pInfo->team[1].name, 16);
        M_team_name_right.Format("%s", tname);
    }
	m_pLog[index]->scoreL = (short)ntohs((short)pInfo->team[0].score);		
	m_pLog[index]->scoreR = (short)ntohs((short)pInfo->team[1].score);
    if ( s_last_score_l != m_pLog[index]->scoreL
         || s_last_score_r != m_pLog[index]->scoreR )
    {
        s_last_score_l = m_pLog[index]->scoreL;
        s_last_score_r = m_pLog[index]->scoreR;
        M_scored_pos.push_back(index);
    }


    m_pLog[index]->ball = pInfo->ball;
	for ( i = 0; i < MAX_PLAYER *2 ; i++ )
    {
        m_pLog[index]->pos[i] = pInfo->pos[i];
    }
    m_pLog[index]->time = (int)pInfo->time;

	//TRACE1("get info %d\n", (int)ntohs(pInfo->time));
	if ( index >= M_max_cycles )
	{
		M_max_cycles = index + 1;
		M_has_data = true;
	}

    // playmode is not before_kick_off
    char pm = pInfo->pmode - 1;
    if ( pm != 0 && pm != 1 )
    {
        index++;
    }

    //TRACE2("add log index %d pm %d\n", index, pm);

	return true;
}

bool TLogPlayer::addMsginfo(msginfo_t* info)
{
	// a player or the referee sent a message
	if ( M_max_cycles <= 0 || M_max_cycles > MAX_CYCLES )
	{
		m_strMsg[0] += info->message;
		m_byteBoard[0] = (BYTE) info->board;
	}
	else
	{
		m_strMsg[M_max_cycles - 1] += info->message;
		m_byteBoard[M_max_cycles - 1] = (BYTE) info->board;
	}
	
	return true;
}

void TLogPlayer::start(bool forward)
{
	// create the window object for the timer
	if ( m_hWnd == NULL )
	{
		POSITION pos = m_pDoc->GetFirstViewPosition();
		CView* pView = m_pDoc->GetNextView(pos);
		if ( ! Create(NULL, NULL, WS_CHILD, CRect(0,0,0,0), pView, 0, NULL) )
		{
			TRACE0("Can not create the CWnd object for Logplayer\n");
		}
	}

	// set the timer of the window
	if ( ! M_running )
	{
		M_running = true;
		M_timer_id = 1;
		M_timer_id = SetTimer(M_timer_id, M_timer_msec, NULL);
	}
	M_forward = forward;
}

void TLogPlayer::step()
{
	int	iCycle = M_current_cycle + 1;

    stop();

    // search for the next real cycle
    while ( m_pLog[iCycle] == NULL && iCycle < M_max_cycles )
    {
        iCycle++;
    }

    setCurrentCycle(iCycle);
}

void TLogPlayer::stepBack()
{
	int	iCycle = M_current_cycle - 1;

    if ( iCycle < 0 )
    {
		return;
    }

    stop();

    // search for the next real cycle
    while ( m_pLog[iCycle] == NULL && iCycle >= 0 )
    {
        iCycle--;
    }

    setCurrentCycle(iCycle);
}


void TLogPlayer::accelerate()
{
	// double the speed
	if ( M_timer_msec > 10 )
    {
		M_timer_msec /= 2;
        if ( M_timer_msec < 10 ) M_timer_msec = 10;
    }

	if ( M_running )
	{
		KillTimer(M_timer_id);
		M_timer_id = 0;
		M_running = false;
	}

    start(M_forward);
}

void TLogPlayer::decelerate()
{
	// double the speed
	if ( M_timer_msec < 1000 ) 
    {
		M_timer_msec *= 2;
        if ( M_timer_msec > 1000 ) M_timer_msec = 1000;
    }

	if ( M_running )
	{
		KillTimer(M_timer_id);
		M_timer_id = 0;
		M_running = false;
	}

    start(M_forward);
}

void TLogPlayer::jump()
{
    DCycleSlider	tDialog;
    
    int max_game_cyc = (int)ntohs((short)m_pLog[getMaxCycle() - 1]->time);
    tDialog.m_pDoc = m_pDoc;
    tDialog.m_sStartNum.Format("%d", 1);
    tDialog.m_sStopNum.Format("%d", max_game_cyc);
    tDialog.m_iMin = 1;
    //tDialog.m_iMax = getMaxCycle() - 1;
    tDialog.m_iMax = max_game_cyc;
    //tDialog.m_iPos = getCurrentCycle();
    tDialog.m_iPos = (int)ntohs((short)m_pLog[getCurrentCycle()]->time);
    tDialog.DoModal();
}

void TLogPlayer::jumpToNextScore()
{
    for ( vector<int>::iterator it = M_scored_pos.begin();
          it != M_scored_pos.end();
          ++it )
    {
        /*
        if ( M_current_cycle < *it
             && *it - S_SCORE_REWIND_COUNT < M_current_cycle )
        {
            break;
        }
        */
        if ( *it > M_current_cycle + S_SCORE_REWIND_COUNT )
        {
            setCurrentCycle(*it - S_SCORE_REWIND_COUNT);
            return;
        }
    }
}


/*
void TLogPlayer::FillShowStruct()
{
	// do not play not existing cycles
	if ( M_cycle_infos.size() > M_current_cycle )
	{
        const myshowinfo_t2& cur = M_cycle_infos[M_current_cycle];
		// fill in a showinfo_t structure
		m_tShow.pmode = cur.pmode;
		m_tShow.team[0].score = cur.scoreL;
		m_tShow.team[1].score = cur.scoreR;
		strcpy(m_tShow.team[0].name, M_team_name_left);
		strcpy(m_tShow.team[1].name, M_team_name_right);

        m_tShow.ball = cur.ball;
		ntohBall(&m_tShow.ball);

        for ( int i = 0; i < MAX_PLAYER * 2; i++ )	
		{
            m_tShow.pos[i] = cur.pos[i];
			ntohPlayer(&m_tShow.pos[i]);
		}

		m_tShow.time = ntohs((short)cur.time);	
	}
}
*/

void TLogPlayer::FillShowStruct()
{
	// do not play not existing cycles
	if ( m_pLog[M_current_cycle] != NULL )
	{
		// fill in a showinfo_t structure
		m_tShow.pmode = m_pLog[M_current_cycle]->pmode;
		m_tShow.team[0].score = m_pLog[M_current_cycle]->scoreL;
		m_tShow.team[1].score = m_pLog[M_current_cycle]->scoreR;
		//strcpy(m_tShow.team[0].name, M_team_name_left);
		//strcpy(m_tShow.team[1].name, M_team_name_right);

        m_tShow.ball = m_pLog[M_current_cycle]->ball;
		ntohBall(&m_tShow.ball);

        for ( int i = 0; i < MAX_PLAYER * 2; i++ )	
		{
            m_tShow.pos[i] = m_pLog[M_current_cycle]->pos[i];
			ntohPlayer(&m_tShow.pos[i]);
		}

		m_tShow.time = ntohs((short)m_pLog[M_current_cycle]->time);	
        TRACE3("%d my  %d - %d\n", m_tShow.time, m_pLog[M_current_cycle]->scoreL, m_pLog[M_current_cycle]->scoreR);
        TRACE3("%d set %d - %d\n", m_tShow.time, m_tShow.team[0].score, m_tShow.team[1].score);
	}
}

void TLogPlayer::fillShowinfo2ToSegment1(showinfo_t &to, const myshowinfo_t2 &from,
                                         short startcycle)
{
    to.pmode = from.pmode;

    strncpy(to.team[0].name, M_team_name_left, 16);
    to.team[0].score = htons(from.scoreL);
    strncpy(to.team[1].name, M_team_name_right, 16);
    to.team[1].score = htons(from.scoreR);

    ball2pos(&to.pos[0], &from.ball);

    for ( int i = 1; i < 23; i++ )
    {
        player2pos(&to.pos[i], &from.pos[i - 1]);
        to.pos[i].side = htons(i < 12 ? 1 : -1);
        to.pos[i].unum = htons(i < 12 ? i : i - 11);
    }

    to.time = htons(ntohs(from.time) - startcycle + 1);
}

void TLogPlayer::fillShowinfo2ToSegment2(short_showinfo_t2 &to,
                                         const myshowinfo_t2 &from,
                                         const short startcycle)
{
    to.ball = from.ball;
    memcpy(to.pos, from.pos, sizeof(from.pos));

    to.time = htons(ntohs(from.time) - startcycle + 1);
}


void TLogPlayer::setGameCycle(const int cyc)
{
    int left = 0, right = M_max_cycles - 1;
    int pos;

    while ( left < right )
    {
        pos = (left + right) / 2;
        int poscyc = ntohs((short)m_pLog[pos]->time);
        if ( poscyc < cyc )
        {
            left = pos + 1;            
        }
        else if ( cyc < poscyc )
        {
            right = pos;
        }
        else
        {
            setCurrentCycle(pos);
            return;
        }
    }

    pos = (left + right) / 2;
    if ( 0 <= pos && pos < M_max_cycles )
    {
        setCurrentCycle(pos);
    }
}

void TLogPlayer::setCurrentCycle(int pos)
{ 
	if ( pos < 0 )
	{
		M_current_cycle = 0;
        stop();
		return;
	}
    else if ( pos >= M_max_cycles )
	{
		M_current_cycle = M_max_cycles - 1;
        stop();
		return;
	}

	M_current_cycle = pos; 
	//if ( M_cycle_infos.size() > M_current_cycle )
    if ( m_pLog[M_current_cycle] != NULL )
	{
		// fill in the information of the first cycle
		FillShowStruct();
		// draw the scene
		m_pDoc->SetInfo(&m_tShow);
	}

	// show the messages
    /*
	if ( m_strMsg[M_current_cycle].GetLength() > 0 )
    {
		m_pDoc->ShowString(tm, m_strMsg[M_current_cycle], m_byteBoard[M_current_cycle]);
    }
    */

	m_pDoc->RedrawDoc();
}

/////////////////////////////////////////////////////////////////////////////
// TLogPlayer message handlers

void TLogPlayer::OnTimer(UINT nIDEvent) 
{
	// increase the counter for the cycles
	if ( M_forward )
    {
        setCurrentCycle(M_current_cycle + 1);
    }
	else
    {
        setCurrentCycle(M_current_cycle - 1);
    }

	CWnd::OnTimer(nIDEvent);
}

void TLogPlayer::saveTextData(CString filename, const int start, const int end,
                              const BOOL save_ptype, const BOOL save_sparam,
                              const BOOL save_detail, const BOOL save_count)
{
    CFile text_file;
    const char* playmodeString[] = PLAYMODE_STRINGS;

    if ( start < 0 || getMaxCycle() < end )
    {
        AfxMessageBox("Range Over!");
        return;
    }

    if ( filename.IsEmpty()
         || ! text_file.Open(filename, CFile::modeCreate | CFile::modeWrite) )
    {
        AfxMessageBox("Cannot create file!");
        return;
    }

    // progress bar
    RCGLoadingDialog *loading = new RCGLoadingDialog;
    loading->Create(IDD_RCG_LOAD);
    loading->CenterWindow();
    loading->M_loading.SetRange32(start, end);
    loading->ShowWindow(SW_SHOW);

    CString text;
    int i, j;

    // game info
    int last_cyc = getMaxCycle() - 1;
    if ( last_cyc >= 0 )
    {
        text.Format("%s %d vs %s %d\n",
                    M_team_name_left, (int)m_pLog[last_cyc]->scoreL,
                    M_team_name_right, (int)m_pLog[last_cyc]->scoreR);
    }
    else
    {
        text.Format("%s vs %s\n", M_team_name_left, M_team_name_right);
    }
    text_file.Write(text, text.GetLength());

    // server param
    if ( save_sparam )
    {
        text = "----------\n";
        text_file.Write(text, text.GetLength());
        M_server_param.toString(text);
        text_file.Write(text, text.GetLength());
    }

    // player types
    if ( save_ptype )
    {
        text = "----------\nhetero players\n";
        text_file.Write(text, text.GetLength());
        //text.Format("hetero player\n seed %d\n", M_player_param.random_seed);
        //text_file.Write(text, text.GetLength());
        for ( i = 0; i < M_num_player_types; i++ )
        {
            M_player_types[i].toString(text);
            text_file.Write(text, text.GetLength());
        }
    }

    const int COUNT_INC = (end - start) / 20 + 1;
    int count = COUNT_INC;
    const int VER = M_protocol_version;
    CString tmp;
    for ( i = start; i <= end; i++ )
    {
        if ( count == 0 )
        {
            count = COUNT_INC;
            loading->M_loading.SetPos(i);
        }
        count--;

        if ( m_pLog[i] != NULL )
        {
            // time playmode
            text.Format("-----\n%d %s\n",
                        (int)(short)ntohs(m_pLog[i]->time),
                        playmodeString[m_pLog[i]->pmode - 1]);
            //text_file.Write(text, text.GetLength());

            // ball
            tmp.Format(" ball (%.3f %.3f)",
                        (double)(long)ntohl((long)m_pLog[i]->ball.x) / SHOWINFO_SCALE2,
                        (double)(long)ntohl((long)m_pLog[i]->ball.y) / SHOWINFO_SCALE2);
            text += tmp;
            if ( save_detail )
            {
                tmp.Format(" (%.3f %.3f)",
                            (double)(long)ntohl((long)m_pLog[i]->ball.deltax) / SHOWINFO_SCALE2,
                            (double)(long)ntohl((long)m_pLog[i]->ball.deltay) / SHOWINFO_SCALE2);
                text += tmp;
            }
            text += '\n';
            //text_file.Write(text, text.GetLength());

            for ( j = 0; j < MAX_PLAYER * 2; j++ )
            {
                const player_t &player = m_pLog[i]->pos[j];
                tmp.Format(" %c %2d",
                            (j < 11) ? 'l' : 'r',
                            (j < 11) ? j + 1 : j - 11 + 1);
                text += tmp;
                if ( (short)ntohs(player.mode) == DISABLE )
                {
                    text += '\n';
                    continue;
                }
                if ( VER >= 2 )
                {
                    tmp.Format(" (t %d)", (int)(short)ntohs((short)player.type));
                    text += tmp;
                }
                tmp.Format(" (p %6.2f %6.2f)",
                           (double)(long)ntohl((long)player.x) / SHOWINFO_SCALE2,
                           (double)(long)ntohl((long)player.y) / SHOWINFO_SCALE2);
                text += tmp;
                if ( save_detail)
                {
                    if ( VER >= 2 )
                    {
                        tmp.Format(" (v %5.2f %5.2f) (b %6.1f) (n %5.1f) (vw %3.0f %s)",
                            (double)(long)ntohl((long)player.deltax) / SHOWINFO_SCALE2,
                            (double)(long)ntohl((long)player.deltay) / SHOWINFO_SCALE2,
                            (double)(long)ntohl((long)player.body_angle) / SHOWINFO_SCALE2 * 180.0 / PI,
                            (double)(long)ntohl((long)player.head_angle) / SHOWINFO_SCALE2 * 180.0 / PI,
                            (double)(long)ntohl((long)player.view_width) / SHOWINFO_SCALE2 * 180.0 / PI,
                            (short)ntohs((short)player.view_quality) ? "high" : "low ");
                        text += tmp;
                        tmp.Format(" (stamina %5.1f (e %5.3f) (r %5.3f))",
                            (double)(long)ntohl((long)player.stamina) / SHOWINFO_SCALE2,
                            (double)(long)ntohl((long)player.effort) / SHOWINFO_SCALE2,
                            (double)(long)ntohl((long)player.recovery) / SHOWINFO_SCALE2);
                        text += tmp;
                    }
                    else
                    {
                        tmp.Format(" (b %6.1f)",
                            (double)(long)ntohl((long)player.body_angle) / SHOWINFO_SCALE2 * 180.0 / PI);
                        text += tmp;
                    }
                }
                if ( save_count )
                {
                    tmp.Format(" (count (k  %4d) (d  %4d) (t  %4d) (s %4d) (n %4d) (c %4d) (m %4d) (c %4d))",
                               (int)(short)ntohs((short)player.kick_count),
                               (int)(short)ntohs((short)player.dash_count),
                               (int)(short)ntohs((short)player.turn_count),
                               (int)(short)ntohs((short)player.say_count),
                               (int)(short)ntohs((short)player.tneck_count),
                               (int)(short)ntohs((short)player.catch_count),
                               (int)(short)ntohs((short)player.move_count),
                               (int)(short)ntohs((short)player.chg_view_count));
                    text += tmp;
                }
                text += '\n';
                //text_file.Write(text, text.GetLength());
            }
            text_file.Write(text, text.GetLength());
        }
    }

    text_file.Close();

    //loading->ShowWindow(SW_HIDE);
    loading->DestroyWindow();
    delete loading;
    MessageBox("File Saved Successfully!", "", MB_OK);
}

void TLogPlayer::pos2player(player_t *to, const pos_t *from)
{
	to->mode			= from->enable;
  	to->type			= 0;
	to->x				= nstonl((short)from->x);
	to->y				= nstonl((short)from->y);
	to->deltax			= 0;
	to->deltay			= 0;

    to->body_angle
        = htonl((long)((double)(short)ntohs((short)from->angle) / 180.0 * PI * SHOWINFO_SCALE2));
	to->head_angle		= to->body_angle;
	to->view_width		= htonl((long)(90.0 * SHOWINFO_SCALE2));
	to->view_quality	= htons(1);

    to->stamina		= htonl((long)(3500.0 * SHOWINFO_SCALE2));
	to->effort			= htonl((long)(1.0 * SHOWINFO_SCALE2));
	to->recovery		= htonl((long)(1.0 * SHOWINFO_SCALE2));

	to->kick_count		= 0;
	to->dash_count		= 0;
	to->turn_count		= 0;
	to->say_count		= 0;
	to->tneck_count	= 0;
    to->catch_count	= 0;
	to->move_count		= 0;
	to->chg_view_count	= 0;
}

void TLogPlayer::player2pos(pos_t *to, const player_t *from)
{
	to->enable = from->mode;
	to->x = nltons((long)from->x);
	to->y = nltons((long)from->y);

    double bang = (double)(long)ntohl((long)from->body_angle) / SHOWINFO_SCALE2 / PI * 180.0;
    if ( bang > 0.0 ) bang += 0.5;
    else bang -= 0.5;
    if ( bang > 180.0 ) bang -= 360.0;
    else if ( bang < -180.0 ) bang += 360.0;
    to->angle = htons((short)(bang));
}

void TLogPlayer::ntohPlayer(player_t *tmp)
{
    tmp->mode = ntohs(tmp->mode) ;
    tmp->type = ntohs(tmp->type) ; 
    
    tmp->x = ntohl((long)(tmp->x)) ;
    tmp->y = ntohl((long)(tmp->y)) ;
    tmp->deltax = ntohl((long)(tmp->deltax)) ;
    tmp->deltay = ntohl((long)(tmp->deltay)) ;
    tmp->body_angle = ntohl((long)tmp->body_angle) ;
    tmp->head_angle = ntohl((long)tmp->head_angle) ;
    tmp->view_width = ntohl((long)tmp->view_width) ;		
    tmp->view_quality = ntohs((short)(tmp->view_quality)) ;
    
    tmp->stamina = ntohl((long)(tmp->stamina)) ;
    tmp->effort = ntohl((long)(tmp->effort)) ;
    tmp->recovery = ntohl((long)(tmp->recovery)) ;
    
    tmp->kick_count = ntohs((short)tmp->kick_count) ;
    tmp->dash_count = ntohs((short)tmp->dash_count) ;
    tmp->turn_count = ntohs((short)tmp->turn_count) ;
    tmp->say_count = ntohs((short)tmp->say_count) ;
    tmp->tneck_count = ntohs((short)tmp->tneck_count) ;
    tmp->catch_count = ntohs((short)tmp->catch_count) ;
    tmp->move_count = ntohs((short)tmp->move_count) ;
    tmp->chg_view_count = ntohs((short)tmp->chg_view_count) ;
}

void TLogPlayer::ntohBall(ball_t *tmp)
{
   tmp->deltax = ntohl((long)tmp->deltax);
   tmp->deltay = ntohl((long)tmp->deltay);
   tmp->x      = ntohl((long)tmp->x);
   tmp->y      = ntohl((long)tmp->y);
}

void TLogPlayer::ball2pos(pos_t *to, const ball_t *from)
{
   to->enable = htons(STAND);
   to->x      = nltons((long)from->x);
   to->y      = nltons((long)from->y);
   to->angle  = 0;
}

void TLogPlayer::pos2ball(ball_t *to, const pos_t *from)
{
   to->deltax = 0;
   to->deltay = 0;
   to->x      = nstonl((short)from->x);
   to->y      = nstonl((short)from->y);
}


void TLogPlayer::setParam(player_type_t *ptinfo)
{
    const int id = (int)ntohs(ptinfo->id);
    if ( id < 0 || id > 6 )
    {
        return;
    }
    M_player_types[id].id = id;
    M_player_types[id].player_speed_max = nltohd(ptinfo->player_speed_max);
    M_player_types[id].stamina_inc_max = nltohd(ptinfo->stamina_inc_max);
    M_player_types[id].player_decay = nltohd(ptinfo->player_decay);
    M_player_types[id].inertia_moment = nltohd(ptinfo->inertia_moment);
    M_player_types[id].dash_power_rate = nltohd(ptinfo->dash_power_rate);
    M_player_types[id].player_size = nltohd(ptinfo->player_size);
    M_player_types[id].kickable_margin = nltohd(ptinfo->kickable_margin);
    M_player_types[id].kick_rand = nltohd(ptinfo->kick_rand);
    M_player_types[id].extra_stamina = nltohd(ptinfo->extra_stamina);
    M_player_types[id].effort_max = nltohd(ptinfo->effort_max);
    M_player_types[id].effort_min = nltohd(ptinfo->effort_min);

    M_num_player_types++;
}




void TLogPlayer::drawBallLocus(CDC *pDC, const int start,
                               const int cx, const int cy,
                               const bool reverse_x)
{
    if ( start < 0 )
    {
        return;
    }

    const int CURRENT = M_current_cycle;

    CPen* oldPen = pDC->SelectObject(M_ball_locus_pen);
    pDC->SetBkMode(TRANSPARENT);

    int ix, iy;
    double bx, by;

    if ( m_pLog[start] != NULL )
    {
        bx = (double)(long)ntohl((long)m_pLog[start]->ball.x) / SHOWINFO_SCALE2;
        by = (double)(long)ntohl((long)m_pLog[start]->ball.y) / SHOWINFO_SCALE2;
        if ( reverse_x )
        {
            bx *= -1.0;
            by *= -1.0;
        }
        ix = cx + m_pDoc->scale(bx);
        iy = cy + m_pDoc->scale(by);
        pDC->MoveTo(ix, iy);
    }

    for ( int i = start + 1; i < CURRENT + 1; i++ )
    {
        if ( m_pLog[i] != NULL )
        {
            bx = (double)(long)ntohl((long)m_pLog[i]->ball.x) / SHOWINFO_SCALE2;
            by = (double)(long)ntohl((long)m_pLog[i]->ball.y) / SHOWINFO_SCALE2;
            if ( reverse_x )
            {
                bx *= -1.0;
                by *= -1.0;
            }
            ix = cx + m_pDoc->scale(bx);
            iy = cy + m_pDoc->scale(by);
            if ( is_trace_draw_pmode(m_pLog[i-1]->pmode, m_pLog[i]->pmode) )
            {
                // not play_on
                if ( m_pLog[i]->pmode - 1 != 2 )
                {
                    pDC->SelectObject(M_ball_locus_pen2);
                }
                else
                {
                    pDC->SelectObject(M_ball_locus_pen);
                }
                pDC->LineTo(ix, iy);
            }
            else
            {
                pDC->MoveTo(ix, iy);
            }
        }
    }
    pDC->SelectObject(oldPen);
}


void TLogPlayer::drawPlayerLocus(CDC *pDC, const int start, const int selected,
                                 const int cx, const int cy,
                                 const bool reverse_x)
{
    if ( start < 0 || selected <= 0 )
    {
        return;
    }

    const int CURRENT = M_current_cycle;
    CPen* oldPen;
    if ( selected <= 11 )
    {
        oldPen = pDC->SelectObject(M_left_player_locus_pen);
    }
    else
    {
        oldPen = pDC->SelectObject(M_right_player_locus_pen);
    }
    pDC->SetBkMode(TRANSPARENT);

    int ix, iy;
    double bx, by;

    if ( m_pLog[start] != NULL )
    {
        bx = (double)(long)ntohl((long)m_pLog[start]->pos[selected - 1].x) / SHOWINFO_SCALE2;
        by = (double)(long)ntohl((long)m_pLog[start]->pos[selected - 1].y) / SHOWINFO_SCALE2;
        if ( reverse_x )
        {
            bx *= -1.0;
            by *= -1.0;
        }
        ix = cx + m_pDoc->scale(bx);
        iy = cy + m_pDoc->scale(by);
        pDC->MoveTo(ix, iy);
    }

    for ( int i = start + 1; i < CURRENT + 1; i++ )
    {
        if ( m_pLog[i] != NULL /*&& m_pLog[i]->pos[selected - 1].mode*/ )
        {
            bx = (double)(long)ntohl((long)m_pLog[i]->pos[selected - 1].x) / SHOWINFO_SCALE2;
            by = (double)(long)ntohl((long)m_pLog[i]->pos[selected - 1].y) / SHOWINFO_SCALE2;
            if ( reverse_x )
            {
                bx *= -1.0;
                by *= -1.0;
            }
            ix = cx + m_pDoc->scale(bx);
            iy = cy + m_pDoc->scale(by);
            if ( is_trace_draw_pmode(m_pLog[i-1]->pmode, m_pLog[i]->pmode) )
            {
                // not play_on
                if ( m_pLog[i]->pmode - 1 != 2 )
                {
                    pDC->SelectObject(M_player_locus_pen2);
                }
                else
                {
                    if ( selected <= 11 ) pDC->SelectObject(M_left_player_locus_pen);
                    else                  pDC->SelectObject(M_right_player_locus_pen);
                }
                pDC->LineTo(ix, iy);
            }
            else
            {
                pDC->MoveTo(ix, iy);
            }
        }
    }
    pDC->SelectObject(oldPen);
}

