// TLogPlayer.h: interface for the TLogPlayer class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_TLOGPLAYER_H__C24D2441_6256_11D1_B242_0000C09B5CBE__INCLUDED_)
#define AFX_TLOGPLAYER_H__C24D2441_6256_11D1_B242_0000C09B5CBE__INCLUDED_

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#include "param.h"
#include "GZFile.h"
#include "geometry.h"

#include <vector>
#include <list>

#define MAX_CYCLES 20000
#define DEFAULT_SPEED 100

typedef struct {
	char pmode;
	short scoreL;
	short scoreR;
	player_t pos[22];
	ball_t   ball;
	int time;
} myshowinfo_t2 ;


struct DrawPlayerT {
    //int type_;
    //short mode_;
    Vector2D pos_;
    Vector2D vel_;
    double body_; // radian
    double face_; // radian body + neck
    //double view_width_; // radian
    double stamina_;
    double effort_;
    double recovery_;
    //double size_;
    //double kickable_area_;

    DrawPlayerT(const player_t &player)
        : //type_((int)player.type),
          //mode_(player.mode),
          pos_((double)player.x / SHOWINFO_SCALE2,
               (double)player.y / SHOWINFO_SCALE2),
          vel_((double)player.deltax / SHOWINFO_SCALE2,
               (double)player.deltay / SHOWINFO_SCALE2),
          body_((double)player.body_angle / SHOWINFO_SCALE2),
          face_((double)player.head_angle / SHOWINFO_SCALE2),
          //view_width_((double)player.view_width / SHOWINFO_SCALE2),
          stamina_((double)player.stamina / SHOWINFO_SCALE2),
          effort_((double)player.effort / SHOWINFO_SCALE2),
          recovery_((double)player.recovery / SHOWINFO_SCALE2)
    { 
        face_ += body_;
        if ( face_ > PI ) face_ -= 2.0 * PI;
        else if ( face_ < -PI ) face_ += 2.0 * PI;
    }

    void reverse()
    {
        pos_ *= -1.0;
        vel_ *= -1.0;
        body_ += PI; if ( body_ > PI ) body_ -= 2.0 * PI;
        face_ += PI; if ( face_ > PI ) face_ -= 2.0 * PI;
    }
};

struct RCLSay {
    RCLSay(int n)
      : unum(n), message("")
    { }
    int unum;
    CString message;
};

inline
bool operator<(const RCLSay &lhs, const RCLSay &rhs)
{
    return lhs.unum < rhs.unum;
}

struct RCLAttentionto {
    RCLAttentionto(char s, int n)
      : side(s), unum(n), target_name("")
    { }
    char side;
    int unum;
    CString target_name;
    int target_num;
};

struct RCLPointto {
    RCLPointto(char s, int n)
        : side(s), unum(n)
    { }
    char side;
    int unum;
    double dist;
    double dir;
};


inline
bool is_trace_draw_pmode(const BYTE prev_pm, const BYTE pm)
{
    switch ( pm - 1 ) {
    case 0: // bko
    case 1: // over
        return false;
    case 2: // play_on
        return true;
    case 3: // koff l
    case 4: // koff r
        if ( prev_pm - 1 != pm - 1 ) return false;
        return true;
    case 5: // kin l
    case 6: // kin r
    case 7: // free kick l
    case 8: // free kick r
    case 9: // ckick l
    case 10: // ckick r
    case 11: // gkick l
    case 12: // gkick r
        return true;
    case 13: // goal_l
    case 14: // goal_r
        if ( prev_pm - 1 == pm - 1 ) return false;
        return true;
    case 15: // drop
        return false;
    case 16: // offside l
    case 17: // offside r
    case 31: // back pass l
    case 32: // back pass r
    case 33: // free kick fault l
    case 34: // free kick fault r
    case 35: // catch fault l
    case 36: // catch fault r
    case 37: // indirect free kick l
    case 38: // indirect free kick r
        return true;
    /*
        case 18: // penalty l
        case 19: // penalty r
        case 20: // first half over
        case 21: // pause
        case 22: // human_judge
        case 23: // foul charge l
        case 24: // foul charge r
        case 25: // foul push l
        case 26: // foul push r
        case 27: // foul multi l
        case 28: // foul multi r
        case 29: // foul ballout l
        case 30: // foul ballout r
    */
    /*
    case 39: //  "penalty_setup_l"
    case 40: // "penalty_setup_r"
    case 41: // "penalty_ready_l"
    case 42: // "penalty_ready_r"
    case 43: //" penalty_taken_l"
    case 44: // "penalty_taken_r"
    case 45: // "penalty_miss_l"
    case 46: // "penalty_miss_r"
    case 47: // "penalty_score_l"
    case 48: // "penalty_score_r"
    case 49: // "unknown"
    */
    default:
        return false;
    }

    return true;
}


class CSoccermonitorDoc;


class TLogPlayer : public CWnd {
public:
    static const int S_COACH_UNUM;
    static const int S_SCORE_REWIND_COUNT;

    TLogPlayer() { };
    TLogPlayer(CSoccermonitorDoc* pDoc);
    virtual ~TLogPlayer();

    // serialization
    DECLARE_SERIAL(TLogPlayer)

    virtual void Serialize( CArchive& archive );
    
    void	InitLog();
    bool addLoginfo(const char pmode, const team_t* team,
                     const short_showinfo_t2& sshow2);
    bool addShowinfo(showinfo_t2* info);
    bool addMsginfo(msginfo_t* info);

    const myshowinfo_t2* getShow(const idx) const
    {
        if ( idx < 0 || M_max_cycles - 1 < idx )
        {
            return NULL;
        }
        return m_pLog[idx];
    }

    void start(bool forward=true);
    void stop()
    {
	    if ( M_running )
    	{
	    	KillTimer(M_timer_id);
		    M_timer_id = 0;
    		M_timer_msec = DEFAULT_SPEED;
	    	M_running = false;
    	}
    }
    void rewind()
    {
        setCurrentCycle(0);
    }
    void jump();
    void end()
    {
        setCurrentCycle(M_max_cycles - 1);
    }
    void step();
    void stepBack();
    void accelerate();
    void decelerate();
    void jumpToNextScore();

    bool saveSegmentLog(const CString &filename,
                        short markin, short markout);
    void saveTextData(CString filename, const int start, const int end,
                      const BOOL save_ptype, const BOOL save_sparam,
                      const BOOL save_detail, const BOOL save_count);
    
    bool hasNextScore() const
    {
        if ( ! M_scored_pos.empty()
             && M_current_cycle < M_scored_pos.back() - S_SCORE_REWIND_COUNT )
        {
            return true;
        }
        return false;
    }
    bool isRunning() const
    {
        return M_running;
    };
    bool isForward() const
    {
        return M_forward;
    }
    bool hasData() const
    {
        return M_has_data;
    }

    bool hasRCLData() const
    {
        return M_has_rcl_data;
    }
    const std::list<RCLSay>& getRCLCoachSay(int cycle) const
    {
        if ( cycle < 0 || MAX_CYCLES < cycle )
        {
            return M_rcl_coach_say[0];
        }
        return M_rcl_coach_say[cycle];
    }
    const std::list<RCLSay>& getRCLSay(int cycle) const
    {
        if ( cycle < 0 || MAX_CYCLES < cycle )
        {
            return M_rcl_say[0];
        }
        return M_rcl_say[cycle];
    }
    const std::list<RCLAttentionto>& getRCLAttentionto(int cycle) const
    {
        if ( cycle < 0 || MAX_CYCLES < cycle )
        {
            return M_rcl_attentionto[0];
        }
        return M_rcl_attentionto[cycle];
    }
    const std::list<RCLPointto>& getRCLPointto(int cycle) const
    {
        if ( cycle < 0 || MAX_CYCLES < cycle )
        {
            return M_rcl_pointto[0];
        }
        return M_rcl_pointto[cycle];
    }

    int	getMaxCycle() const
    {
        return M_max_cycles;
    }
    int	getCurrentCycle() const
    {
        return M_current_cycle;
    }
    void setGameCycle(const int cyc);
    void setCurrentCycle(int pos);
    


    void drawBallLocus(CDC *pDC, const int start,
                       const int cx, const int cy, const bool reverse_x);
    void drawPlayerLocus(CDC *pDC, const int start, const int selected,
                         const int cx, const int cy, const bool reverse_x);

    void loadRCL(CGZFile& gzfile);

private:
    void FillShowStruct();


    void initRCLData();
    void loadRCG1(CGZFile& gzfile, const int file_len);
    void loadRCG2(CGZFile& gzfile, const int file_len);
    void loadRCG3(CGZFile& gzfile, const int file_len);

    void fillShowinfo2ToSegment1(showinfo_t &to, const myshowinfo_t2 &from,
                                 short startcycle);
    void fillShowinfo2ToSegment2(short_showinfo_t2 &to,
                                 const myshowinfo_t2 &from,
                                 const short startcycle);
    void saveRCG2(CFile &file, const int start, const int end);
    void saveRCG3(CFile &file, const int start, const int end);
    
    CString M_game_title;

    int M_protocol_version;
    int M_log_version;
    
    int	M_max_cycles;
    int	M_current_cycle;
    CString M_team_name_left;
    CString M_team_name_right;
    showinfo_t2		m_tShow;
    showinfo_t2 M_prev_show;

    //std::vector<myshowinfo_t2> M_cycle_infos;
    myshowinfo_t2*	m_pLog[MAX_CYCLES];
    CString			m_strMsg[MAX_CYCLES];


    std::vector<int> M_scored_pos;

    bool M_has_rcl_data;
    std::list<RCLSay> M_rcl_say[MAX_CYCLES];
    std::list<RCLAttentionto> M_rcl_attentionto[MAX_CYCLES];
    std::list<RCLPointto> M_rcl_pointto[MAX_CYCLES];
    std::list<RCLSay> M_rcl_coach_say[MAX_CYCLES];

    ServerParam M_server_param;
    PlayerType M_player_types[8];
    int M_num_player_types;
    PlayerParam M_player_param;
    BYTE				m_byteBoard[MAX_CYCLES];
    // has been loaded
    CSoccermonitorDoc*	m_pDoc;
    int M_timer_id;
    int	M_timer_msec;
    bool M_running;
    bool M_forward;
    bool M_has_data;
    
    CPen* M_ball_locus_pen;
    CPen* M_ball_locus_pen2;
    CPen* M_left_player_locus_pen;
    CPen* M_right_player_locus_pen;
    CPen* M_player_locus_pen2;

    // Operations
public:
    const CString & gameTitle() const
    {
        return M_game_title;
    }

    void pos2ball(ball_t *to, const pos_t *from);
    void ball2pos(pos_t *to, const ball_t *from);
    void ntohBall(ball_t *tmp);
    void ntohPlayer(player_t *tmp);
    void pos2player(player_t *to, const pos_t *from);
    void player2pos(pos_t *to, const player_t *from);
    
    int protocolVersion() const
    {
        return M_protocol_version;
    }
    void setProtocolVersion(const int ver)
    {
        M_protocol_version = ver;
    }
    int logVersion() const
    {
        return M_log_version;
    }
    void setParam(player_type_t *ptinfo);
    void setParam(player_params_t *pparam)
    {
        M_player_param.convert(pparam);    
    }
    void setParam(server_params_t *sparam)
    {
        M_server_param.convert(sparam);
    }

    const CString & getTeamNameLeft() const
    {
        return M_team_name_left;
    }
    const CString & getTeamNameRight() const
    {
        return M_team_name_right;
    }
    
    const ServerParam & serverParam() const
    {
        return M_server_param;
    }
    const PlayerType & playerType(const int id) const
    {
        if ( id < 0 || 6 < id )
        {
            return M_player_types[0];
        }
        return M_player_types[id];
    }
    const PlayerParam & playerParam() const
    {
        return M_player_param;
    }
    
    int numPlayerTypes() const
    {
        return M_num_player_types;
    }

	// Overrides
	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(TLogPlayer)
	//}}AFX_VIRTUAL

	// Generated message map functions
protected:
	//{{AFX_MSG(TLogPlayer)
	afx_msg void OnTimer(UINT nIDEvent);
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()
};

#endif // !defined(AFX_TLOGPLAYER_H__C24D2441_6256_11D1_B242_0000C09B5CBE__INCLUDED_)


