/*
 * Copyright 2012 K.K.DNAFORM
 * This file is part of idr_paraclu program.
 * Idr_paraclu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, any later version.
 *
 * Idr_paraclu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Foobar. If not, see <http://www.gnu.org/licenses/>.
 *
 * FILE: 
 * 		automatedClustering.cc
 * USAGE: 
 * 		automatedClustering inputdir outputdir tpm idr outputdir2 project
 * 			inputdir	directory containing input files (You need at least 2 files)
 * 			outputdir	directory containing output files
 * 			tpm			TPM used for a threshold in clustering
 * 			idr		 	IDR used for a threshold
 *			outputdir2	directory containing files for scatter plot of hierarchical stability
 *			project		project name (you like)
 * DESCRIPTION: 
 * 		Executing clustering for all pairs of CAGE data replicates.
 * CREATED:	
 *		2012.04.17
 * REVISION:
 * 		2012.04.23	Adding counting the number of reproducible/irreproducible clusters.
 * 		2012.04.23	Adding creating files for scatter plot of hierarchical stability.
 * 		2012.04.25	Adding creating directory for output files.
 * 		2012.06.25	Adding project name as a prefix of output files.
 */
#include <stdlib.h>
#include <fstream>
#include <iostream>
#include <sstream>
#include <string>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <vector>
#include <set>
#include <dirent.h>
#include <error.h>
#include <errno.h>

#include "automatedClustering.h"

using namespace std;

// ----------------------------------------
//  Declaring constant variables
// ----------------------------------------
// ******* [MOD] : 2012/04/25 : START
//const char			USAGE_MESSAGE[]			= "Usage : automatedClustering inputdir outputdir tpm idr\n";
// ******* [MOD] : 2012/06/25 : START
//const char			USAGE_MESSAGE[]			= "Usage : automatedClustering inputdir outputdir tpm idr outputdir2\n";
const char			USAGE_MESSAGE[]			= "Usage : automatedClustering inputdir outputdir tpm idr outputdir2 project\n";
// ******* [MOD] : 2012/06/25 : END
// ******* [MOD] : 2012/04/25 : END
// IDR script file name
const char			IDR_MAPPEDINPUT[]		= "batch-consistency-analysis-mappedInput.r";
const char			IDR_FUNCTIONS[]			= "functions-all-06-24-10.r";
// ******* [ADD] : 2012/06/25 : START
// Prefix of output files
const char			PREFIX_OUTPUT[]			= "cluster";
// ******* [ADD] : 2012/06/25 : END
// top peaks file name
const char			TOPPEAKS_FILE[]			= "toppeaks.bed";
// bottom peaks file name
const char			BOTTOMPEAKS_FILE[]		= "bottompeaks.bed";
// ******* [ADD] : 2012/04/23 : START
// Output file name containing the number of clusters
const char			CLUSTERNUM_FILE[]		= "clusternum.txt";
// Output file name of information for a pair of replicates.
const char			REPLICATE_FILE[]		= "replicates.txt";
// Output file name for scatter plot of hierarchical stability
const char			STABILITY_FILE[]		= "stabilityplot.txt";
// Header of scatter plot file 
const char			STABILITY_HEADER[]		= "1\t\"ID1\"\t\"score1\"\t\"ID2\"\t\"score2\"\t\"idr\"\n";
// ******* [ADD] : 2012/04/23 : END
// Directory creating mode
const mode_t		DIR_MODE				= S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH;
// Command creating CTSS
const char			MAKECTSS_COMMAND[]		= "make_ctss3.sh %s %s";
// Modified paraclu command
const char			PARACLU_COMMAND[]		= "paraclu-mod-tpm %f %s > %s";
// Command calculating stability
const char			CALSTAB_COMMAND[]		= "paraclu-cut.sh %s > %s";
// Command excluding clusters with over 1000 stability
const char			DISCARD_COMMAND[]		= "awk '$5 < 1000 {print}' %s > %s";
// Command calculating hierarchical stability
const char			CALHIERARCHIC_COMMAND[]	= "makeCluster.sh %s %s";
// intersectBedOhmiya command
const char			INTERSECTBED_COMMAND[]	= "intersectBED2.sh %s %s 0.9 %s";
// IDR command
const char			IDR_COMMAND[]			= "Rscript %s/batch-consistency-analysis-mappedInput.r %s %s";
// Command extracting reproducible clusters
const char			IDRTOBED_COMMAND[]		= "convertIDRtoBED2.pl %s %f %s";
// Command excluding redundant clusters
const char			REMOVESIMILAR_COMMAND[]	= "removeSimilarCluster.pl %s %s";
// Command discarding clusters with more than 200 bp length
const char			REMOVELONG_COMMAND[]	= "awk '($3-$2) <= 200 {print}' %s > %s";
// Command extracting top peaks
const char			TOPPEAKS_COMMAND[]		= "intersectBedOhmiya -s -wo -a %s -b %s | awk '$2 <= $8 && $3 >= $9 {print}' | sort -k1,1 -k6,6 -k2,2n -k3,3n | uniq | groupByOrg -g 1,2,3,6 -opCols 11 -ops count -full | awk '$19 == 1 {print $1 \"\t\" $2 \"\t\" $3 \"\t\" $4 \"\t\" $5 \"\t\" $6}' > %s"; 
// Command extracting bottom peaks
const char			BOTTOMPEAKS_COMMAND[]	= "intersectBedOhmiya -s -wo -a %s -b %s | awk '$2 >= $8 && $3 <= $9 {print}' | sort -k1,1 -k6,6 -k2,2n -k3,3n | uniq | groupByOrg -g 1,2,3,6 -opCols 11 -ops count -full | awk '$19 == 1 {print $1 \"\t\" $2 \"\t\" $3 \"\t\" $4 \"\t\" $5 \"\t\" $6}' > %s";
// Command merging replicates
const char			MERGE_COMMAND[]			= "cat %s | mergeBedOhmiya -s > %s";


// ----------------------------------------
//  Declaring variables
// ----------------------------------------
// Directory containing output files (full path)
string				outputDirFull;
// ******* [ADD] : 2012/04/25 : START
// Directory containing scatter plot files (full path)
string				outputDir2Full;
// ******* [ADD] : 2012/04/25 : END
// Sequential number for temporary files
static int			fileNumSeq = 0;

// ******* [ADD] : 2012/04/23 : START
// Information of the number of clusters
struct Clusters {
	string			filename;				// Replicate file name
	int				reproducible;			// Num of reproducible clusters
	int				irreproducible;			// Num of irreproducible clusters
};
// ******* [ADD] : 2012/04/23 : END



/*
 * FUNCTION NAME	: getLine
 * DESCRIPTION		: Reading a line from a file
 * PARAMETERS		:
 * 	(1)	(IN)	fp		- File descriptor
 * 	(2)	(OUT)	line	- Reading line
 * RETURN VALUE		: The number of reading characters
 * EXCEPTION		: None
 * NOTE				: None
 * REVISION			:
 * 	2012.04.17		Creating
 */
int getLine( FILE *fp, string& line )
{
	int		i = 0;
	int		c = 0;
	int		len = 0;

	line.erase();
	while( ( c = fgetc( fp ) ) != EOF ) {
		if( c == '\n' ) {
			if( i == 0 ) {
				i = -1;
			}
			break;
		}
		line += (char)c;
		i ++;
	}
	return i;
}



/*
 * FUNCTION NAME	: mkTmpFile
 * DESCRIPTION		: Creating a temporary file name.
 * PARAMETERS		: None
 * RETURN VALUE		: Temporary file name
 * EXCEPTION		: None
 * NOTE				: None
 * REVISION			:
 * 	2012.04.17		Creating
 */
string mkTmpFile( void )
{
	char		tmp[10];
	memset( tmp, 0x0, sizeof(tmp) );
	sprintf( tmp, "tmp_%05d", fileNumSeq ); 
	fileNumSeq ++;

	return (string)tmp;
}



/*
 * FUNCTION NAME	: runRollback
 * DESCRIPTION		: Executing roll back.
 * PARAMETERS		: None
 * RETURN VALUE		: None
 * EXCEPTION		: None
 * NOTE				: None
 * REVISION			:
 * 	2012.04.17		Creating
 */
void runRollback( void )
{
	struct stat	statbuf;

	if( stat( outputDirFull.c_str(), &statbuf ) == 0 && S_ISDIR( statbuf.st_mode ) ) {
		// ----------------------------------------
		// Removing files in the directory for 
		// output files.
		// ----------------------------------------
		DIR*	pDir = opendir( outputDirFull.c_str() );
		if( pDir == NULL ) {
			error( 1, errno, "Failed to open directory : %s", outputDirFull.c_str() );
		}
		char*	szCurEntry = new char[sizeof(struct dirent) + 256];
		struct dirent*	pEntry = (struct dirent*)szCurEntry;
		struct dirent*	pReadDir;
		string	removeFile;
		string	removeFileFull;
		while( 1 ) {
			readdir_r( pDir, pEntry, &pReadDir );
			if( pReadDir == NULL ) {
				break;
			}
			// Getting the file name without path.
			removeFile = pEntry->d_name;
			if( removeFile == "." || removeFile == ".." ) {
				continue;
			}
			// Getting the file name with path.
			removeFileFull = outputDirFull;
			if( removeFileFull.at( removeFileFull.length() - 1 ) != '/' ) {
				removeFileFull += "/";
			}
			removeFileFull += removeFile;
			unlink( removeFileFull.c_str() );
		}
		if( closedir( pDir ) != 0 ) {
			error( 1, errno, "Failed to close directory : %s", outputDirFull.c_str() );
		}

		delete [] szCurEntry;

		// ----------------------------------------
		// Removing the directory for output files.
		// ----------------------------------------
		rmdir( outputDirFull.c_str() );
	}
}



/*
 * FUNCTION NAME	: makeIDRScript
 * DESCRIPTION		: Creating IDR script file.
 * PARAMETERS		: None
 * RETURN VALUE		: None
 * EXCEPTION		: None
 * NOTE				: None
 * REVISION			:
 * 	2012.04.17		Creating
 */
void makeIDRScript( void )
{
	int		i;
	size_t	fwb;

	// Creating the batch-consistency-analysis-mappedInput.r file.
	string			idrfile1 = outputDirFull;
	if( idrfile1.at( idrfile1.length() - 1 ) != '/' ) {
		idrfile1 += "/";
	}
	idrfile1 += IDR_MAPPEDINPUT;
	FILE	*idr1 = fopen( idrfile1.c_str(), "w" );
	if( idr1 == NULL ) {
		runRollback();
		error( 1, errno, "Failed to open : %s", idrfile1.c_str() );
	}
	for( i = 0; i < 54; i ++ ) {
		fwb = fwrite( IDR_SCRIPT1[i], sizeof(char), strlen( IDR_SCRIPT1[i] ), idr1 );
		if( fwb < 1 ) {
			runRollback();
			error( 1, errno, "Failed to write : %s", IDR_SCRIPT1[i] );
		}
		if( i == 2 ) {
			char		tmp[1000];
			memset( tmp, 0x0, sizeof(tmp) );
			sprintf( tmp, "source(\"%s/functions-all-06-24-10.r\")\n", outputDirFull.c_str() );
			fwb = fwrite( tmp, sizeof(char), strlen( tmp ), idr1 );
			if( fwb < 1 ) {
				runRollback();
				error( 1, errno, "Failed to write : %s", tmp );
			}
		}
	}
	if( EOF == fclose( idr1 ) ) {
		runRollback();
		error( 1, errno, "Failed to close : %s", idrfile1.c_str() );
	}

	// Creating the functions-all-06-24-10.r file.
	string		idrfile2 = outputDirFull;
	if( idrfile2.at( idrfile2.length() - 1 ) != '/' ) {
		idrfile2 += "/";
	}
	idrfile2 += IDR_FUNCTIONS;
	FILE	*idr2 = fopen( idrfile2.c_str(), "w" );
	if( idr2 == NULL ) {
		runRollback();
		error( 1, errno, "Failed to open : %s", idrfile2.c_str() );
	}
	for( i = 0; i < 1757; i ++ ) {
		fwb = fwrite( IDR_SCRIPT2[i], sizeof(char), strlen( IDR_SCRIPT2[i] ), idr2 );
		if( fwb < 1 ) {
			runRollback();
			error( 1, errno, "Failed to write : %s", IDR_SCRIPT2[i] );
		}
	}
	if( EOF == fclose( idr2 ) ) {
		runRollback();
		error( 1, errno, "Failed to close : %s", idrfile2.c_str() );
	}
}


/*
 * FUNCTION NAME	: getClusterName
 * DESCRIPTION		: Getting a cluster name.
 * PARAMETERS		:
 * 	(1)	(IN)	line	- Reading line from a file
 * 	(2)	(OUT)	cluster	- Cluster name
 * RETURN VALUE		: None
 * EXCEPTION		: None
 * NOTE				: None
 * REVISION			:
 * 	2012.04.17		Creating
 */
void getClusterName( string& line, string& cluster )
{
	istringstream	in( line );
	char			c;
	string			value;

	// Getting chromosome
	value.erase();
	while( in.get( c ) && ( c != ' ' && c != '\t' ) ) {
		value.push_back( c );
	}
	cluster = value + "_";

	// Getting start position.
	value.erase();
	while( in.get( c ) && ( c == ' ' || c == '\t' ) );
	value.push_back( c );
	while( in.get( c ) && ( c != ' ' && c != '\t' ) ) {
		value.push_back( c );
	}
	cluster = cluster + value + "_";

	// Getting end position.
	value.erase();
	while( in.get( c ) && ( c == ' ' || c == '\t' ) );
	value.push_back( c );
	while( in.get( c ) && ( c != ' ' && c != '\t' ) ) {
		value.push_back( c );
	}
	cluster = cluster + value + "_";

	// Getting strand.
	value.erase();
	for( int i = 0; i < 2; i ++ ) {
		while( in.get( c ) && ( c == ' ' || c == '\t' ) );
		while( in.get( c ) && ( c != ' ' && c != '\t' ) );
	}
	while( in.get( c ) && ( c == ' ' || c == '\t' ) );
	value.push_back( c );
	while( in.get( c ) && ( c != ' ' && c != '\t' ) ) {
		value.push_back( c );
	}
	cluster += value;
}


/*
 * FUNCTION NAME	: clearOutput
 * DESCRIPTION		: Removing unnecessary files in the directory
 * 					  containing output files.
 * PARAMETERS		:
 * 	(1)	(IN)	topfile		- Top peaks file name
 * 	(2)	(IN)	bottomfile	- Bottom peaks file name
 * RETURN VALUE		: None
 * EXCEPTION		: None
 * NOTE				: None
 * REVISION			:
 * 	2012.04.17		Creating
 */
void clearOutput( string& topfile, string& bottomfile )
{
	DIR*	pDir = opendir( outputDirFull.c_str() );
	if( pDir == NULL ) {
		runRollback();
		error( 1, errno, "Failed to open directory : %s", outputDirFull.c_str() );
	}
	char*		szCurEntry = new char[sizeof(struct dirent) + 256];
	struct dirent*	pEntry = (struct dirent*)szCurEntry;
	struct dirent*	pReadDir;
	string		removeFile;
	string		removeFileFull;

	while( 1 ) {
		readdir_r( pDir, pEntry, &pReadDir );
		if( pReadDir == NULL ) {
			break;
		}
		// Getting file name without path.
		removeFile = pEntry->d_name;
		if( removeFile == "." || removeFile == ".." ) {
			continue;
		}
// ******* [ADD] : 2012/04/23 : START
// ******* [MOD] : 2012/06/25 : START
//		if( removeFile == CLUSTERNUM_FILE || removeFile == REPLICATE_FILE ||
//			removeFile == STABILITY_FILE ) {
//			continue;
//		}
		if( removeFile.find( CLUSTERNUM_FILE ) != string::npos ||
			removeFile.find( REPLICATE_FILE ) != string::npos ||
			removeFile.find( STABILITY_FILE ) != string::npos ) {
			continue;
		}
// ******* [MOD] : 2012/06/25 : END
// ******* [ADD] : 2012/04/23 : END
		// Constructing file name with path.
		removeFileFull = outputDirFull;
		if( removeFileFull.at( removeFileFull.length() - 1 ) != '/' ) {
			removeFileFull += "/";
		}
		removeFileFull += removeFile;
		if( removeFileFull == topfile || removeFileFull == bottomfile ) {
			continue;
		}
		unlink( removeFileFull.c_str() );
	}
	if( closedir( pDir ) != 0 ) {
		runRollback();
		error( 1, errno, "Failed to close directory : %s", outputDirFull.c_str() );
	}
	delete [] szCurEntry;
}


// ******* [ADD] : 2012/04/23 : START
/*
 * FUNCTION NAME	: getIDR
 * DESCRIPTION		: Getting the IDRs.
 * PARAMETERS		:
 * 	(1)	(IN)	line		- Reading line
 * 	(2)	(IN)	idrvalue	- IDR
 * RETURN VALUE		: None
 * EXCEPTION		: None
 * NOTE				: None
 * REVISION			:
 * 	2012.04.17		Creating
 */
void getIDR( string& line, double& idrvalue )
{
	istringstream		in( line );
	char				c;
	string				value;
	int					i;
	
	value.erase();
	for( i = 0; i < 5; i ++ ) {
		while( in.get( c ) && ( c != ' ' && c != '\t' ) );
		while( in.get( c ) && ( c == ' ' || c == '\t' ) );
	}
	value.push_back( c );
	while( in.get( c ) && ( c != ' ' && c != '\t' ) ) {
		value.push_back( c );
	}
	idrvalue = strtod( value.c_str(), NULL );
}
// ******* [ADD] : 2012/04/23 : END



/*
 * FUNCTION NAME	: main
 * DESCRIPTION		: Main function
 * PARAMETERS		: See the usage
 * RETURN VALUE		: 
 * 		0			- Normal exit
 * 		1			- Abnormal exit
 * EXCEPTION		: None
 * NOTE				: None
 * REVISION			:
 * 	2012.04.17		Creating
 */
int main( int argc, char* argv[] )
{

	// Directory name containing input files without path
	string		inputDir;
	// Directory name containing input files with path
	string		inputDirFull;
	// Input file name without path
	string		inputFileName;
	// Input file name with path
	string		inputFileFull;
	// Directory name containing output files
	string		outputDir;
// ******* [ADD] : 2012/04/25 : START
	// Directory name containing files for scatter plot
	string		outputDir2;
// ******* [ADD] : 2012/04/25 : END
// ******* [ADD] : 2012/06/25 : START
	// Project name
	string		project;
// ******* [ADD] : 2012/06/25 : END
	// TPM
	double		tpm;
	// IDR
	double		idr;
	// Input files list
	vector<string>		inputFileList;
	// Clustering top peaks files list
	vector<string>		topFileList;
	// Clustering bottom peaks files list
	vector<string>		bottomFileList;
// ******* [ADD] : 2012/04/23 : START
	// The number of clusters information
	vector<Clusters>	clustersList;
	char				tmpdata[256];
	string				line;
// ******* [ADD] : 2012/04/23 : END

	int			sts;
	int			i;
	int			j;
// ******* [ADD] : 2012/04/23 : START
	int			k;
// ******* [ADD] : 2012/04/23 : END
	char		cmdimage[1000];
	size_t		fwb;

	// ----------------------------------------
	//  Option processing
	// ----------------------------------------
// ******* [MOD] : 2012/04/25 : START
//	if( argc != 5 ) {
// ******* [MOD] : 2012/06/25 : START
//	if( argc != 6 ) {
	if( argc != 7 ) {
// ******* [MOD] : 2012/06/25 : END
// ******* [MOD] : 2012/04/25 : END
		cout << USAGE_MESSAGE << endl;
		return 1;
	}
	inputDir = argv[1];
	outputDir = argv[2];
	tpm = strtod( argv[3], NULL );
	idr = strtod( argv[4], NULL );
// ******* [ADD] : 2012/04/25 : START
	outputDir2 = argv[5];
// ******* [ADD] : 2012/04/25 : END
// ******* [ADD] : 2012/06/25 : START
	project = argv[6]; 
	istringstream	in( project );
	char			c;
	while( in.get( c ) ) {
		if( c == ' ' || c == '\t' ) {
			error( 1, 1, "You cannot use blank and tab characters." );
		}
	}
// ******* [ADD] : 2012/06/25 : END

	// ----------------------------------------
	//  Getting the current directory
	// ----------------------------------------
	char		curDir[256];
	memset( curDir, 0x0, sizeof(curDir) );
	getcwd( curDir, sizeof(curDir) );
	if( inputDir.substr( 0, 1 ) != "." && inputDir.substr( 0, 1 ) != "/" ) {
		// ----------------------------------------
		// Creating absolute path of the directory 
		// containing input files, when the path 
		// is not neither absolute nor relative 
		// path.
		// ----------------------------------------
		inputDirFull = curDir;
		if( inputDirFull.at( inputDirFull.length() - 1 ) != '/' ) {
			inputDirFull += "/";
		}
		inputDirFull += inputDir;
	} else {
		inputDirFull = inputDir;
	}

	if( outputDir.substr( 0, 1 ) != "." && outputDir.substr( 0, 1 ) != "/" ) {
		// ----------------------------------------
		// Creating absolute path of the directory
		// for output files, when the path is not
		// neither absolute nor relative path.
		// ----------------------------------------
		outputDirFull = curDir;
		if( outputDirFull.at( outputDirFull.length() - 1 ) != '/' ) {
			outputDirFull += "/";
		}
		outputDirFull += outputDir;
	} else {
		outputDirFull = outputDir;
	}

// ******* [ADD] : 2012/04/25 : START
	if( outputDir2.substr( 0, 1 ) != "." && outputDir2.substr( 0, 1 ) != "/" ) {
		// ----------------------------------------
		// Creating absolute path of the directory 
		// for scatter plot files, when the path is 
		// not neither absolute nor relative path.
		// ----------------------------------------
		outputDir2Full = curDir;
		if( outputDir2Full.at( outputDir2Full.length() - 1 ) != '/' ) {
			outputDir2Full += "/";
		}
		outputDir2Full += outputDir2;
	} else {
		outputDir2Full = outputDir2;
	}
// ******* [ADD] : 2012/04/25 : END
	// ----------------------------------------
	// Creating the directory for output files.
	// When the directory is already exist,
	// doing nothing.
	// ----------------------------------------
	if( mkdir( outputDirFull.c_str(), DIR_MODE ) != 0 && errno != EEXIST ) {
		error( 1, errno, "Failed to make directory : %s", outputDirFull.c_str() );
	}
// ******* [ADD] : 2012/04/25 : START
	if( mkdir( outputDir2Full.c_str(), DIR_MODE ) != 0 && errno != EEXIST ) {
		error( 1, errno, "Failed to make directory : %s", outputDir2Full.c_str() );
	}
// ******* [ADD] : 2012/04/25 : END

	// ----------------------------------------
	//  Reading the directory containing input
	//  files.
	// ----------------------------------------
	DIR*	pInputDir;
	pInputDir = opendir( inputDirFull.c_str() );
	if( pInputDir == NULL ) {
		// On failure of opening the directory
		// the program exit abnormally.
		runRollback();
		error( 1, errno, "Failed to open directory : %s", inputDirFull.c_str() );
	}
	char*		szCurEntry = new char[sizeof(struct dirent) + 256];
	struct dirent*	pEntry = (struct dirent*)szCurEntry;
	struct dirent*	pReadDir;
	struct stat		statbuf;
	// Loop for reading input file names.
	while( 1 ) {
		readdir_r( pInputDir, pEntry, &pReadDir );
		if( pReadDir == NULL ) {
			break;
		}
		// Getting a file name without path.
		inputFileName = pEntry->d_name;
		if( inputFileName == "." || inputFileName == ".." ) {
			continue;
		}
		// Constructing a file name with absolute path.
		inputFileFull = inputDirFull;
		if( inputFileFull.at( inputFileFull.length() - 1 ) != '/' ) {
			inputFileFull += "/";
		}
		inputFileFull += inputFileName;
		sts = stat( inputFileFull.c_str(), &statbuf );
		if( sts == -1 ) {
			runRollback();
			error( 1, errno, "Stat occured error. file = %s", inputFileFull.c_str() );
		}
		if( !S_ISREG( statbuf.st_mode ) ) {
			continue;
		}
		inputFileList.push_back( inputFileFull );
	}
	if( closedir( pInputDir ) != 0 ) {
		runRollback();
		error( 1, errno, "Failed to close directory : %s", inputDirFull.c_str() );
	}
	delete [] szCurEntry;

	// ----------------------------------------
	// Creating the IDR script file.
	// ----------------------------------------
	makeIDRScript();

	// ----------------------------------------
	// Loop for clustering.
	// Executing clustering all combinations
	// of clusters, respectively.
	// ----------------------------------------
	for( i = 0; i < inputFileList.size() - 1; i ++ ) {
		for( j = i + 1; j < inputFileList.size(); j ++ ) {
			// ----------------------------------------
			// Creating CTSS files.
			// ----------------------------------------
			string		ctssfile1 = outputDirFull;
			string		ctssfile2 = outputDirFull;
			if( ctssfile1.at( ctssfile1.length() - 1 ) != '/' ) {
				ctssfile1 += "/";
				ctssfile2 += "/";
			}
			ctssfile1.append( mkTmpFile() );
			ctssfile2.append( mkTmpFile() );
			// Creating a CTSS file for a former replicate.
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 MAKECTSS_COMMAND,
					 inputFileList[i].c_str(),
					 ctssfile1.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}
			// Creating a CTSS file for a later replicate.
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 MAKECTSS_COMMAND,
					 inputFileList[j].c_str(),
					 ctssfile2.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Running modified paraclu.
			// ----------------------------------------
			string		paraclufile1 = outputDirFull;
			string		paraclufile2 = outputDirFull;
			if( paraclufile1.at( paraclufile1.length() - 1 ) != '/' ) {
				paraclufile1 += "/";
				paraclufile2 += "/";
			}
			paraclufile1.append( mkTmpFile() );
			paraclufile2.append( mkTmpFile() );
			// Running paraclu using the former replicate's data.
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 PARACLU_COMMAND,
					 tpm,
					 ctssfile1.c_str(),
					 paraclufile1.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}
			// Running paraclu using the later replicate's data.
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 PARACLU_COMMAND,
					 tpm,
					 ctssfile2.c_str(),
					 paraclufile2.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Calculating stability
			// ----------------------------------------
			string		stabfile1 = outputDirFull;
			string		stabfile2 = outputDirFull;
			if( stabfile1.at( stabfile1.length() - 1 ) != '/' ) {
				stabfile1 += "/";
				stabfile2 += "/";
			}
			stabfile1.append( mkTmpFile() );
			stabfile2.append( mkTmpFile() );
			// Calculating stability for the former replicate.
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 CALSTAB_COMMAND,
					 paraclufile1.c_str(),
					 stabfile1.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}
			// Calculating stability for the later replicate. 
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 CALSTAB_COMMAND,
					 paraclufile2.c_str(),
					 stabfile2.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Excluding clusters with 1000 stability.
			// ----------------------------------------
			string		discardfile1 = outputDirFull;
			string		discardfile2 = outputDirFull;
			if( discardfile1.at( discardfile1.length() - 1 ) != '/' ) {
				discardfile1 += "/";
				discardfile2 += "/";
			}
			discardfile1.append( mkTmpFile() );
			discardfile2.append( mkTmpFile() );
			// Excluding clusters with 1000 stability 
			// for the former replicate.
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 DISCARD_COMMAND,
					 stabfile1.c_str(),
					 discardfile1.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}
			// Excluding clusters with 1000 stability
			// for the later replicate.
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 DISCARD_COMMAND,
					 stabfile2.c_str(),
					 discardfile2.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Calculating hierarchical stability.
			// ----------------------------------------
			string		hierarchic1 = outputDirFull;
			string		hierarchic2 = outputDirFull;
			if( hierarchic1.at( hierarchic1.length() - 1 ) != '/' ) {
				hierarchic1 += "/";
				hierarchic2 += "/";
			}
			hierarchic1.append( mkTmpFile() );
			hierarchic2.append( mkTmpFile() );
			// Calculating hierarchical stability for the former replicate.
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 CALHIERARCHIC_COMMAND,
					 discardfile1.c_str(),
					 hierarchic1.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}
			// Calculating hierarchical stability for the later replicate.
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 CALHIERARCHIC_COMMAND,
					 discardfile2.c_str(),
					 hierarchic2.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Executing intersectBedOhmiya.
			// ----------------------------------------
			string		intersectfile = outputDirFull;
			if( intersectfile.at( intersectfile.length() - 1 ) != '/' ) {
				intersectfile += "/";
			}
			intersectfile.append( mkTmpFile() );
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 INTERSECTBED_COMMAND,
					 hierarchic1.c_str(),
					 hierarchic2.c_str(),
					 intersectfile.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Calculating IDR.
			// ----------------------------------------
			string		idrfile = outputDirFull;
			if( idrfile.at( idrfile.length() - 1 ) != '/' ) {
				idrfile += "/";
			}
			idrfile.append( mkTmpFile() );
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 IDR_COMMAND,
					 outputDirFull.c_str(),
					 intersectfile.c_str(),
					 idrfile.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Extracting reproducible clusters.
			// ----------------------------------------
			string		bedfile = outputDirFull;
			string		idrresultfile;
			if( bedfile.at( bedfile.length() - 1 ) != '/' ) {
				bedfile += "/";
			}
			bedfile.append( mkTmpFile() );
			idrresultfile = idrfile + "-overlapped-peaks.txt";
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 IDRTOBED_COMMAND,
					 idrresultfile.c_str(),
					 idr,
					 bedfile.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

// ******* [ADD] : 2012/04/23 : START
			// ------------------------------------------------------------
			// Counting the number of reproducible/irreproducible clusters.
			// ------------------------------------------------------------
			Clusters	clusters;
			clusters.reproducible = 0;
			clusters.irreproducible = 0;
			FILE*		overlapped = fopen( idrresultfile.c_str(), "r" );
			if( overlapped == NULL ) {
				runRollback();
				error( 1, errno, "Failed to open : %s", idrresultfile.c_str() );
			}
			// Skipping reading header lines.
			getLine( overlapped, line );
			while( getLine( overlapped, line ) != 0 ) {
				double		idrvalue;
				getIDR( line, idrvalue );
				if( idrvalue < 0.1 ) {
					// If a cluster is reproducible, 
					// the counter increase.
					clusters.reproducible ++;
				} else {
					// If a cluster is irreproducible,
					// the counter increase.
					clusters.irreproducible ++;
				}
			}
			int		index = inputFileList[i].find_last_of( "/" );
			string	file1 = inputFileList[i].substr( index + 1, inputFileList[i].length() - 
													 ( index + 1 ) );
			index = inputFileList[j].find_last_of( "/" );
			string	file2 = inputFileList[j].substr( index + 1, inputFileList[j].length() - 
													 ( index + 1 ) );
			clusters.filename = file1 + " vs " + file2;
			clustersList.push_back( clusters );
// ******* [ADD] : 2012/04/23 : END

// ******* [ADD] : 2012/04/23 : START
			// ----------------------------------------
			// Creating the scatter plot for stability.
			// ----------------------------------------
			if( i == 0 && j == 1 ) {
				FILE*		overlapped2 = fopen( idrresultfile.c_str(), "r" );
				if( overlapped2 == NULL ) {
					runRollback();
					error( 1, errno, "Failed to open : %s", idrresultfile.c_str() );
				}
// ******* [MOD] : 2012/04/25 : START
//				string		stabilityFile = outputDirFull;
				string		stabilityFile = outputDir2Full;
				if( stabilityFile.at( stabilityFile.length() - 1 ) != '/' ) {
					stabilityFile += "/";
				}
// ******* [ADD] : 2012/06/25 : START
				stabilityFile = stabilityFile + PREFIX_OUTPUT + "_" + project + "_";
// ******* [ADD] : 2012/06/25 : END
				stabilityFile += STABILITY_FILE;
// ******* [MOD] : 2012/04/25 : END
				FILE*		stab = fopen( stabilityFile.c_str(), "w" );	
				if( stab == NULL ) {
					runRollback();
					error( 1, errno, "Failed to open : %s", stabilityFile.c_str() );
				}
				// Creating the new header line instead of current one.
				getLine( overlapped2, line );
				fwb = fwrite( STABILITY_HEADER, sizeof(char), strlen( STABILITY_HEADER ), stab );
				if( fwb < 1 ) {
					runRollback();
					error( 1, errno, "Failed to write : %s", STABILITY_HEADER );
				}
				while( getLine( overlapped2, line ) != 0 ) {
					istringstream	in( line );
					string			value;
					char			c;
					value.erase();
					for( k = 0; k < 6; k ++ ) {
						while( in.get( c ) && ( c != ' ' && c != '\t' ) ) {
							value.push_back( c );
						}
						if( k < 5 ) {
							value += "\t";
						} else {
							value += "\n";
						}
						fwb = fwrite( value.c_str(), sizeof(char), value.length(), stab );
						if( fwb < 1 ) {
							runRollback();
							error( 1, errno, "Failed to write : %s", value.c_str() );
						}
						value.erase();
						while( in.get( c ) && ( c == ' ' || c == '\t' ) );
						value.push_back( c );
					}
				}
				if( EOF == fclose( overlapped2 ) ) {
					runRollback();
					error( 1, errno, "Failed to close : %s", idrresultfile.c_str() );
				}
				if( EOF == fclose( stab ) ) {
					runRollback();
					error( 1, errno, "Failed to close : %s", stabilityFile.c_str() );
				}
			}
// ******* [ADD] : 2012/04/23 : END

			// ----------------------------------------
			// Discarding redundant clusters.
			// ----------------------------------------
			bedfile += ".bed";
			string		nonredundant = outputDirFull;
			if( nonredundant.at( nonredundant.length() - 1 ) != '/' ) {
				nonredundant += "/";
			}
			nonredundant.append( mkTmpFile() );
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 REMOVESIMILAR_COMMAND,
					 bedfile.c_str(),
					 nonredundant.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Excluding clusters with more than 
			// 200 bp length.
			// ----------------------------------------
			string		shortclusters = outputDirFull;
			if( shortclusters.at( shortclusters.length() - 1 ) != '/' ) {
				shortclusters += "/";
			}
			shortclusters.append( mkTmpFile() );
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 REMOVELONG_COMMAND,
					 nonredundant.c_str(),
					 shortclusters.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Extracting top peaks.
			// ----------------------------------------
			string		toppeaks = outputDirFull;
			if( toppeaks.at( toppeaks.length() - 1 ) != '/' ) {
				toppeaks += "/";
			}
			toppeaks.append( mkTmpFile() );
			topFileList.push_back( toppeaks );
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 TOPPEAKS_COMMAND,
					 shortclusters.c_str(),
					 shortclusters.c_str(),
					 toppeaks.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}

			// ----------------------------------------
			// Extracting bottom peaks.
			// ----------------------------------------
			string		bottompeaks = outputDirFull;
			if( bottompeaks.at( bottompeaks.length() - 1 ) != '/' ) {
				bottompeaks += "/";
			}
		bottompeaks.append( mkTmpFile() );
			bottomFileList.push_back( bottompeaks );
			string		bottompeaks_tmp = bottompeaks + ".tmp";
			memset( cmdimage, 0x0, sizeof(cmdimage) );
			sprintf( cmdimage,
					 BOTTOMPEAKS_COMMAND,
					 shortclusters.c_str(),
					 shortclusters.c_str(),
					 bottompeaks_tmp.c_str() );
			sts = system( cmdimage );
			if( sts != 0 ) {
				runRollback();
				error( 1, errno, "Failed to command : %s", cmdimage );
			}
			// Discarding clusters not overlapping other clusters
			// from bottom peaks collection.
			set<string>		topPeaksList;
			string			line;
			string			cluster;
			FILE*	top = fopen( toppeaks.c_str(), "r" );
			if( top == NULL ) {
				runRollback();
				error( 1, errno, "Failed to open : %s", toppeaks.c_str() );
			}
			while( getLine( top, line ) != 0 ) {
				getClusterName( line, cluster );
				topPeaksList.insert( cluster );
			}
			if( EOF == fclose( top ) ) {
				runRollback();
				error( 1, errno, "Failed to close : %s", toppeaks.c_str() );
			}
			FILE*	bottom_tmp = fopen( bottompeaks_tmp.c_str(), "r" );
			if( bottom_tmp == NULL ) {
				runRollback();
				error( 1, errno, "Failed to open : %s", bottompeaks_tmp.c_str() );
			}
			FILE*	bottom = fopen( bottompeaks.c_str(), "w" );
			if( bottom == NULL ) {
				runRollback();
				error( 1, errno, "Failed to open : %s", bottompeaks.c_str() );
			}
			while( getLine( bottom_tmp, line ) != 0 ) {
				getClusterName( line, cluster );
				if( topPeaksList.find( cluster ) != topPeaksList.end() ) {
					continue;
				}
				line += "\n";
				fwb = fwrite( line.c_str(), sizeof(char), line.length(), bottom );
				if( fwb < 1 ) {
					runRollback();
					error( 1, errno, "Failed to write : %s", line.c_str() );
				}
			}
			if( EOF == fclose( bottom_tmp ) ) {
				runRollback();
				error( 1, errno, "Failed to close : %s", bottompeaks_tmp.c_str() );
			}
			if( EOF == fclose( bottom ) ) {
				runRollback();
				error( 1, errno, "Failed to close : %s", bottompeaks.c_str() );
			}
		}
	}

	// ----------------------------------------
	// Merging the result for each a pair of
	// replicates for the top and bottom peaks
	// respectively.
	// ----------------------------------------
// ******* [MOD] : 2012/04/25 : START 
	string		toppeaksfile = outputDirFull;
	if( toppeaksfile.at( toppeaksfile.length() - 1 ) != '/' ) {
		toppeaksfile += "/";
	}
// ******* [ADD] : 2012/06/25 : START
	toppeaksfile.append( PREFIX_OUTPUT );
	toppeaksfile = toppeaksfile + "_" + project + "_";
// ******* [ADD] : 2012/06/25 : END
	toppeaksfile.append( TOPPEAKS_FILE );
// ******* [MOD] : 2012/04/25 : END
	string		mergeFiles = "";
	// Merging the results for the top peaks.
	for( i = 0; i < topFileList.size(); i ++ ) {
		mergeFiles = mergeFiles + topFileList[i] + " ";
	}
	memset( cmdimage, 0x0, sizeof(cmdimage) );
	sprintf( cmdimage,
			 MERGE_COMMAND,
			 mergeFiles.c_str(),
			 toppeaksfile.c_str() );
	sts = system( cmdimage );
	if( sts != 0 ) {
		runRollback();
		error( 1, errno, "Failed to command : %s", cmdimage );
	}
	// Merging the results for the bottom peaks.
// ******* [MOD] : 2012/04/25 : START
	string		bottompeaksfile = outputDirFull;
	if( bottompeaksfile.at( bottompeaksfile.length() - 1 ) != '/' ) {
		bottompeaksfile += "/";
	}
// ******* [ADD] : 2012/06/25 : START
	bottompeaksfile.append( PREFIX_OUTPUT );
	bottompeaksfile = bottompeaksfile + "_" + project + "_";
// ******* [ADD] : 2012/06/25 : END
	bottompeaksfile.append( BOTTOMPEAKS_FILE );
// ******* [MOD] : 2012/04/25 : END
	mergeFiles = "";
	for( i = 0; i < bottomFileList.size(); i ++ ) {
		mergeFiles = mergeFiles + bottomFileList[i] + " ";
	}
	memset( cmdimage, 0x0, sizeof(cmdimage) );
	sprintf( cmdimage,
			 MERGE_COMMAND,
			 mergeFiles.c_str(),
			 bottompeaksfile.c_str() );
	sts = system( cmdimage );
	if( sts != 0 ) {
		runRollback();
		error( 1, errno, "Failed to command : %s", cmdimage );
	}

// ******* [ADD] : 2012/04/23 : START
	// ----------------------------------------
	// Counting the number of clusters.
	// ----------------------------------------
// ******* [MOD] : 2012/04/25 : START
//	string		clusterNumFile = outputDirFull;
	string		clusterNumFile = outputDir2Full;
	if( clusterNumFile.at( clusterNumFile.length() - 1 ) != '/' ) {
		clusterNumFile += "/";
	}
// ******* [ADD] : 2012/06/25 : START
	clusterNumFile = clusterNumFile + PREFIX_OUTPUT + "_" + project + "_";
// ******* [ADD] : 2012/06/25 : END
	clusterNumFile += CLUSTERNUM_FILE;
// ******* [MOD] : 2012/04/25 : END
	FILE*	countfile = fopen( clusterNumFile.c_str(), "w" );
	if( countfile == NULL ) {
		runRollback();
		error( 1, errno, "Failed to open : %s", clusterNumFile.c_str() );
	}
// ******* [MOD] : 2012/04/25 : START
//	string		replicateFile = outputDirFull;
	string		replicateFile = outputDir2Full;
	if( replicateFile.at( replicateFile.length() - 1 ) != '/' ) {
		replicateFile += "/";
	}
// ******* [ADD] : 2012/06/25 : START
	replicateFile = replicateFile + PREFIX_OUTPUT + "_" + project + "_";
// ******* [ADD] : 2012/06/25 : END
	replicateFile += REPLICATE_FILE;
// ******* [MOD] : 2012/04/25 : END
	FILE*	replicate = fopen( replicateFile.c_str(), "w" );
	if( replicate == NULL ) {
		runRollback();
		error( 1, errno, "Failed to open : %s", replicateFile.c_str() );
	}
	for( i = 0; i < 3; i ++ ) {
		for( j = 0; j < clustersList.size(); j ++ ) {
			if( i == 0 ) {
				memset( tmpdata, 0x0, sizeof(tmpdata) );
				sprintf( tmpdata, "Replicate pair %d\t%s\n", j, clustersList[j].filename.c_str() );
				fwb = fwrite( tmpdata, sizeof(char), strlen( tmpdata ), replicate );
				if( fwb < 1 ) {
					runRollback();
					error( 1, errno, "Failed to write : %s", tmpdata );
				}
				memset( tmpdata, 0x0, sizeof(tmpdata) );
				if( j != ( clustersList.size() - 1 ) ) {
					sprintf( tmpdata, "Replicate pair %d\t", j );		
				} else {
					sprintf( tmpdata, "Replicate pair %d\n", j );
				}
				fwb = fwrite( tmpdata, sizeof(char), strlen( tmpdata ), countfile );
				if( fwb < 1 ) {
					runRollback();
					error( 1, errno, "Failed to write : %s", line.c_str() );
				}
			} else if( i == 1 ) {
				memset( tmpdata, 0x0, sizeof(tmpdata) );
				if( j != ( clustersList.size() - 1 ) ) {
					sprintf( tmpdata, "%d\t", clustersList[j].reproducible );
				} else {
					sprintf( tmpdata, "%d\n", clustersList[j].reproducible );
				}
				fwb = fwrite( tmpdata, sizeof(char), strlen( tmpdata ), countfile );
				if( fwb < 1 ) {
					runRollback();
					error( 1, errno, "Failed to write : %s", tmpdata );
				}
			} else {
				memset( tmpdata, 0x0, sizeof(tmpdata) );
				if( j != ( clustersList.size() - 1 ) ) {
					sprintf( tmpdata, "%d\t", clustersList[j].irreproducible );
				} else {
					sprintf( tmpdata, "%d\n", clustersList[j].irreproducible );
				}
				fwb = fwrite( tmpdata, sizeof(char), strlen( tmpdata ), countfile );
				if( fwb < 1 ) {
					runRollback();
					error( 1, errno, "Failed to write : %s", tmpdata );
				}
			}
		}
	}
	if( EOF == fclose( countfile ) ) {
		runRollback();
		error( 1, errno, "Failed to close : %s", clusterNumFile.c_str() );
	}
	if( EOF == fclose( replicate ) ) {
		runRollback();
		error( 1, errno, "Failed to close : %s", replicateFile.c_str() );
	}
// ******* [ADD] : 2012/04/23 : END

	// --------------------------------------------
	// Removing unnecessary files in the directory
	// containing output files.
	// --------------------------------------------
	clearOutput( toppeaksfile, bottompeaksfile );

	return 0;

}

